/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "commctrl.h"
#include <gdk/gdkkeysyms.h>


typedef struct _W32LdrProgressData
{
  W32LdrWindowData wd;
  gint min;   /* ja:最小値(デフォルト0) */
  gint max;   /* ja:最大値(デフォルト100) */
  gint value; /* ja:現在値(デフォルト0) */
  gint step;  /* ja:増分値(デフォルト10) */
} W32LdrProgressData;


static LRESULT WINAPI
w32prgrs_DefWindowProc (HWND   hWnd,
                        UINT   Msg,
                        WPARAM wParam,
                        LPARAM lParam)
{
  W32LdrWindowData *wd;
  W32LdrProgressData *pd;

  pd = g_object_get_data (G_OBJECT (hWnd), "user_data");
  if (!pd)
    return 0;
  wd = &pd->wd;
  switch (Msg)
    {
      case PBM_SETRANGE:
        return w32prgrs_DefWindowProc (hWnd, PBM_SETRANGE32,
                                            LOWORD (lParam), HIWORD (lParam));
      case PBM_SETPOS:
        {
          gint value;

          value = pd->value;
          pd->value = CLAMP (wParam, pd->min, pd->max);
          if (pd->value != value)
            gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (hWnd),
                        (gdouble)(pd->value - pd->min) / (pd->max - pd->min));
          return value;
        }
      case PBM_DELTAPOS:
        {
          gint value;

          value = pd->value;
          pd->value = CLAMP (pd->value + wParam, pd->min, pd->max);
          if (pd->value != value)
            gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (hWnd),
                        (gdouble)(pd->value - pd->min) / (pd->max - pd->min));
          return value;
        }
      case PBM_SETSTEP:
        {
          gint step;

          step = pd->step;
          pd->step = wParam;
          return step;
        }
      case PBM_STEPIT:
        {
          gint value;

          value = pd->value;
          pd->value += pd->step;
          if (pd->value < pd->min)
            pd->value = pd->max;
          else if (pd->max < pd->value)
            pd->value = pd->min;
          if (pd->value != value)
            gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (hWnd),
                        (gdouble)(pd->value - pd->min) / (pd->max - pd->min));
          return value;
        }
      case PBM_SETRANGE32:
        if (lParam - wParam > 0)
          {
            gint min, max;

            min = pd->min;
            max = pd->max;
            pd->min = wParam;
            pd->max = lParam;
            pd->value = CLAMP (pd->value, pd->min, pd->max);
            if (pd->min != min || pd->max != max)
            gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (hWnd),
                        (gdouble)(pd->value - pd->min) / (pd->max - pd->min));
            return MAKELRESULT (min, max);
          }
        return 0;
      case PBM_GETRANGE:
        if (lParam)
          {
            LPPBRANGE ppBRange;

            ppBRange = GINT_TO_POINTER (lParam);
            ppBRange->iLow = pd->min;
            ppBRange->iHigh = pd->max;
          }
        return wParam ? pd->min : pd->max;
      case PBM_GETPOS:
        return pd->value;
      case PBM_SETBARCOLOR:
        return CLR_DEFAULT;
      case PBM_SETMARQUEE:
        return FALSE;
      case PBM_SETBKCOLOR:
        return CLR_DEFAULT;
    }
  return wd->widechar ? DefWindowProcW (hWnd, Msg, wParam, lParam)
                       : DefWindowProcA (hWnd, Msg, wParam, lParam);
}


/* ja:クリックされたとき */
static gboolean
w32prgrs_button_press_event (GtkWidget      *widget,
                             GdkEventButton *event,
                             gpointer        user_data)
{
  W32LdrWindowData *wd;
  NMHDR nmh;

  switch (event->type)
    {
      case GDK_BUTTON_PRESS:
        switch (event->button)
          {
            case 1: nmh.code = NM_CLICK; break;
            case 3: nmh.code = NM_RCLICK; break;
            default: return FALSE;
          }
        break;
      case GDK_2BUTTON_PRESS:
        switch (event->button)
          {
            case 1: nmh.code = NM_DBLCLK; break;
            case 3: nmh.code = NM_RDBLCLK; break;
            default: return FALSE;
          }
        break;
      default: return FALSE;
    }
  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  nmh.hwndFrom = widget;
  nmh.idFrom = wd->wID;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  return FALSE;
}


/* ja:フォーカス取得 */
static gboolean
w32prgrs_focus_in (GtkWidget     *widget,
                   GdkEventFocus *event,
                   gpointer       user_data)
{
  W32LdrWindowData *wd;
  NMHDR nmh;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  nmh.hwndFrom = widget;
  nmh.idFrom = wd->wID;
  nmh.code = NM_SETFOCUS;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  return FALSE;
}


/* ja:フォーカス喪失 */
static gboolean
w32prgrs_focus_out (GtkWidget     *widget,
                    GdkEventFocus *event,
                    gpointer       user_data)
{
  W32LdrWindowData *wd;
  NMHDR nmh;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  nmh.hwndFrom = widget;
  nmh.idFrom = wd->wID;
  nmh.code = NM_KILLFOCUS;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  return FALSE;
}


/* ja:キーが押された */
static gboolean
w32prgrs_key_press_event (GtkWidget   *widget,
                          GdkEventKey *event,
                          gpointer     user_data)
{
  if (event->keyval == GDK_Return)
    {
      W32LdrWindowData *wd;
      NMHDR nmh;

      wd = g_object_get_data (G_OBJECT (widget), "user_data");
      nmh.hwndFrom = widget;
      nmh.idFrom = wd->wID;
      nmh.code = NM_RETURN;
      if (wd->widechar)
        SendMessageW (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nmh));
      else
        SendMessageA (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nmh));
    }
  return FALSE;
}


GtkWidget *
w32ldr_control_create_prgrs (const gchar    *windowname,
                             const guint32   style,
                             const guint32   exstyle,
                             const gint      width,
                             const gint      height,
                             const guint16   id,
                             HWND            hWndParent,
                             HINSTANCE       hInstance,
                             const gboolean  widechar)
{
  GtkWidget *pbar;
  W32LdrWindowData *wd;
  W32LdrProgressData *pd;

  /* ja:プログレスバー */
  pbar = gtk_progress_bar_new ();
  gtk_progress_bar_set_orientation (GTK_PROGRESS_BAR (pbar),
                            style & PBS_VERTICAL ? GTK_PROGRESS_BOTTOM_TO_TOP
                                                 : GTK_PROGRESS_RIGHT_TO_LEFT);
  g_signal_connect (G_OBJECT (pbar), "button-press-event",
                            G_CALLBACK (w32prgrs_button_press_event), NULL);
  g_signal_connect (G_OBJECT (pbar), "focus-in-event",
                            G_CALLBACK (w32prgrs_focus_in), NULL);
  g_signal_connect (G_OBJECT (pbar), "focus-out-event",
                            G_CALLBACK (w32prgrs_focus_out), NULL);
  g_signal_connect (G_OBJECT (pbar), "key-press-event",
                            G_CALLBACK (w32prgrs_key_press_event), NULL);
  gtk_widget_add_events (pbar, GDK_BUTTON_PRESS_MASK | GDK_FOCUS_CHANGE_MASK
                                                        | GDK_KEY_PRESS_MASK);
  pd = g_malloc (sizeof (W32LdrProgressData));
  wd = &pd->wd;
  wd->dwStyle = style;
  wd->dwExStyle = exstyle;
  wd->hInstance = hInstance;
  wd->lpfnWndProc = w32prgrs_DefWindowProc;
  wd->hWndParent = hWndParent;
  wd->wID = id;
  wd->dwUserData = 0;
  wd->lpDialogFunc = NULL;
  wd->lResult = -1;
  wd->dwUser = 0;
  wd->csa = NULL;
  wd->csw = NULL;
  wd->dwInitParam = 0;
  wd->widechar = widechar;
  wd->resizable = TRUE;
  wd->classname = g_strdup ("msctls_progress32");
  wd->focus = NULL;
  wd->child = NULL;
  pd->min = 0;
  pd->max = 100;
  pd->value = 0;
  pd->step = 10;
  g_object_set_data (G_OBJECT (pbar), "user_data", wd);
  return pbar;
}
