/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "kernel32.h"
#include <gdk/gdkkeysyms.h>


GHashTable *ghash_class = NULL;


/******************************************************************************
*                                                                             *
* ja:クラス関数群                                                             *
*                                                                             *
******************************************************************************/
ATOM
w32ldr_RegisterClassEx (UINT            style,
                        WNDPROC         lpfnWndProc,
                        INT             cbClsExtra,
                        INT             cbWndExtra,
                        HINSTANCE       hInstance,
                        HICON           hIcon,
                        HCURSOR         hCursor,
                        HBRUSH          hbrBackground,
                        const gchar    *menuname,
                        const gchar    *classname,
                        HICON           hIconSm,
                        const gboolean  widechar)
{
  guint16 atom;
  W32LdrClassData *w32class;

  atom = w32ldr_atom_add_name (classname);
  if (!ghash_class)
    ghash_class = g_hash_table_new (g_int_hash, g_int_equal);
  else if (g_hash_table_lookup (ghash_class, GUINT_TO_POINTER ((guint)atom)))
    return 0;
  w32class = g_malloc (sizeof (W32LdrClassData));
  w32class->lpfnWndProc = lpfnWndProc;
  w32class->widechar = widechar;
  w32class->classname = w32ldr_atom_get_name (atom);;
  w32class->atom = atom;
  g_hash_table_insert (ghash_class, GUINT_TO_POINTER ((guint)atom), w32class);
  return atom;
}


/******************************************************************************
*                                                                             *
* ja:ウインドウ関数群                                                         *
*                                                                             *
******************************************************************************/
/* ja:閉じるボタンが押された */
static gboolean
w32wnd_delete_event (GtkWidget *widget,
                     GdkEvent  *event,
                     gpointer   user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  return (wd->lpDialogFunc && wd->lpDialogFunc (widget, WM_COMMAND,
                    MAKEWPARAM (IDCANCEL, BN_CLICKED), GPOINTER_TO_INT (NULL)))
                || (wd->widechar ? SendMessageW (widget, WM_CLOSE, 0, 0)
                                 : SendMessageA (widget, WM_CLOSE, 0, 0)) == 0;
}


/* ja:ESCが押された */
static gboolean
w32wnd_key_press_event (GtkWidget   *widget,
                        GdkEventKey *event,
                        gpointer     user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  return event->keyval == GDK_Escape
                && wd->lpDialogFunc && wd->lpDialogFunc (widget, WM_COMMAND,
                    MAKEWPARAM (IDCANCEL, BN_CLICKED), GPOINTER_TO_INT (NULL));
}


/* ja:リアライズ */
static void
w32wnd_realize (GtkWidget *widget,
                gpointer   user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->csa)
    {
      SendMessageA (widget, WM_CREATE, 0, GPOINTER_TO_INT (wd->csa));
      if (GPOINTER_TO_UINT (wd->csa->lpszName) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszName);
      if (GPOINTER_TO_UINT (wd->csa->lpszClass) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszClass);
      g_free (wd->csa);
      wd->csa = NULL;
    }
  if (wd->csw)
    {
      SendMessageW (widget, WM_CREATE, 0, GPOINTER_TO_INT (wd->csw));
      if (GPOINTER_TO_UINT (wd->csw->lpszName) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszName);
      if (GPOINTER_TO_UINT (wd->csw->lpszClass) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszClass);
      g_free (wd->csw);
      wd->csw = NULL;
    }
}


/* ja:破棄 */
static void
w32wnd_destroy (GtkWidget *widget,
                gpointer   user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    SendMessageW (widget, WM_DESTROY, 0, 0);
  else
    SendMessageA (widget, WM_DESTROY, 0, 0);
  if (wd->csa)
    {
      if (GPOINTER_TO_UINT (wd->csa->lpszName) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszName);
      if (GPOINTER_TO_UINT (wd->csa->lpszClass) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszClass);
      g_free (wd->csa);
    }
  if (wd->csw)
    {
      if (GPOINTER_TO_UINT (wd->csw->lpszName) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszName);
      if (GPOINTER_TO_UINT (wd->csw->lpszClass) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszClass);
      g_free (wd->csw);
    }
  g_free (wd->classname);
  g_free (wd->child);
  g_free (wd);
  g_object_set_data (G_OBJECT (widget), "user_data", NULL);
}


HWND
w32ldr_CreateWindowEx (DWORD           dwExStyle,
                       const gchar    *classname,
                       const gchar    *windowname,
                       DWORD           dwStyle,
                       const gint      x,
                       const gint      y,
                       const gint      width,
                       const gint      height,
                       HWND            hWndParent,
                       HMENU           hMenu,
                       HINSTANCE       hInstance,
                       LPVOID          lpvParam,
                       const gboolean  widechar)
{
  guint16 atom;
  gint i;
  GtkWidget *widget = NULL;
  W32LdrWindowData *wd;
  struct _control
  {
    gchar *name;
    GtkWidget * (*func) (const gchar    *windowname,
                         const guint32   style,
                         const guint32   exstyle,
                         const gint      width,
                         const gint      height,
                         const guint16   id,
                         HWND            hWndParent,
                         HINSTANCE       hInstance,
                         const gboolean  widechar);
  } control[] = {{"BUTTON",            w32ldr_control_create_button},
                 {"EDIT",              w32ldr_control_create_edit},
                 {"STATIC",            w32ldr_control_create_static},
                 {"LISTBOX",           w32ldr_control_create_list},
                 {"SCROLLBAR",         w32ldr_control_create_scroll},
                 {"COMBOBOX",          w32ldr_control_create_combo},
                 {"SysPropertySheet",  w32ldr_control_create_prsht},
                 {"SysListView32",     w32ldr_control_create_lview},
                 {"SysTreeView32",     w32ldr_control_create_tview},
                 {"msctls_progress32", w32ldr_control_create_prgrs},
                 {"msctls_updown32",   w32ldr_control_create_updwn},
                 {NULL,                NULL}};

  atom = w32ldr_atom_find_name (classname);
  for (i = 0; control[i].func; i++)
    if (atom > 0 && w32ldr_atom_find_name (control[i].name) == atom)
      {
        widget = control[i].func (windowname,
                                  dwStyle,
                                  dwExStyle,
                                  width,
                                  height,
                                  GPOINTER_TO_UINT (hMenu),
                                  hWndParent,
                                  hInstance,
                                  widechar);
        break;
      }
  if (!control[i].func)
    {
      W32LdrClassData *w32class;
      W32LdrOverlapData *od;

      /* ja:登録クラスを検索 */
      w32class = ghash_class ? g_hash_table_lookup (ghash_class,
                                        GUINT_TO_POINTER ((guint)atom)) : NULL;
      od = g_malloc (sizeof (W32LdrOverlapData));
      wd = &od->wd;
      wd->dwStyle = dwStyle;
      wd->dwExStyle = dwExStyle;
      wd->hInstance = hInstance;
      wd->lpfnWndProc = w32class ? w32class->lpfnWndProc
                                 : widechar ? DefWindowProcW : DefWindowProcA;
      wd->hWndParent = hWndParent;
      wd->wID = GPOINTER_TO_UINT (hMenu);
      wd->dwUserData = 0;
      wd->lpDialogFunc = NULL;
      wd->lResult = -1;
      wd->dwUser = 0;
      wd->csa = NULL;
      wd->csw = NULL;
      wd->dwInitParam = 0;
      wd->resizable = TRUE;
      wd->widechar = widechar;
      wd->classname = w32class ? g_strdup (w32class->classname) : NULL;
      wd->focus = NULL;
      wd->child = NULL;
      od->table = NULL;
      od->fixed = NULL;
      od->hscroll = NULL;
      od->vscroll = NULL;
      if (dwStyle & WS_CHILD)
        {
          if (dwStyle & WS_DLGFRAME)
            {
              widget = gtk_fixed_new ();
              gtk_fixed_set_has_window (GTK_FIXED (widget), TRUE);
            }
          else
            {
              widget = gtk_drawing_area_new ();
            }
          if (width != CW_USEDEFAULT && height != CW_USEDEFAULT)
            gtk_widget_set_size_request (widget, width, height);
        }
      else
        {
          gchar *title;

          /* ja:フィクスト */
          od->fixed = gtk_fixed_new ();
          gtk_fixed_set_has_window (GTK_FIXED (od->fixed), TRUE);
          gtk_widget_show (od->fixed);
          /* ja:スクロールバー */
          if (dwStyle & WS_HSCROLL)
            {
              od->hscroll = gtk_hscrollbar_new
                    (GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 0, 1, 1, 1)));
              gtk_widget_show (od->hscroll);
            }
          if (dwStyle & WS_VSCROLL)
            {
              od->vscroll = gtk_vscrollbar_new
                    (GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 0, 1, 1, 1)));
              gtk_widget_show (od->vscroll);
            }
          /* ja:テーブル
                最上段からメニュー,クライアント,スクロールバー,ステータス */
          od->table = gtk_table_new (2, 4, FALSE);
          gtk_table_attach_defaults (GTK_TABLE (od->table), od->fixed,
                    od->hscroll && (dwExStyle & WS_EX_LEFTSCROLLBAR) ? 1 : 0,
                    od->hscroll && !(dwExStyle & WS_EX_LEFTSCROLLBAR) ? 1 : 2,
                    1, 2);
          if (od->vscroll)
            gtk_table_attach_defaults (GTK_TABLE (od->table), od->vscroll,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 0 : 1,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 1 : 2,
                                    1, 2);
          if (od->hscroll)
            gtk_table_attach_defaults (GTK_TABLE (od->table), od->hscroll,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 1 : 0,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 2 : 1,
                                    2, 3);
          gtk_widget_show (od->table);
          if (dwStyle & WS_DLGFRAME)
            {
              widget = gtk_dialog_new ();
              gtk_dialog_set_has_separator (GTK_DIALOG (widget), FALSE);
              gtk_window_set_destroy_with_parent (GTK_WINDOW (widget), TRUE);
              gtk_window_set_position (GTK_WINDOW (widget),
                                                GTK_WIN_POS_CENTER_ON_PARENT);
              gtk_container_add (GTK_CONTAINER (GTK_DIALOG (widget)->vbox),
                                                                    od->table);
              g_signal_connect (G_OBJECT (widget), "key-press-event",
                                    G_CALLBACK (w32wnd_key_press_event), NULL);
              gtk_widget_add_events (widget, GDK_KEY_PRESS_MASK);
            }
          else
            {
              widget = gtk_window_new (dwStyle & WS_POPUP ? GTK_WINDOW_POPUP
                                                        : GTK_WINDOW_TOPLEVEL);
              gtk_container_add (GTK_CONTAINER (widget), od->table);
            }
          gtk_window_set_resizable (GTK_WINDOW (widget),
                                                    dwStyle & WS_THICKFRAME);
          title = w32ldr_atom_get_string (windowname);
          if (title)
            {
              gtk_window_set_title (GTK_WINDOW (widget), title);
              g_free (title);
            }
          if (hWndParent)
            gtk_window_set_transient_for (GTK_WINDOW (widget),
                                                    GTK_WINDOW (hWndParent));
          g_signal_connect (G_OBJECT (widget), "delete-event",
                                    G_CALLBACK (w32wnd_delete_event), NULL);
        }
      g_object_set_data (G_OBJECT (widget), "user_data", wd);
    }
  if (!widget)
    return NULL;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    {
      wd->csw = g_malloc (sizeof (CREATESTRUCTW));
      wd->csw->lpCreateParams = lpvParam;
      wd->csw->hInstance = hInstance;
      wd->csw->hMenu = hMenu;
      wd->csw->hwndParent = hWndParent;
      wd->csw->cy = height;
      wd->csw->cx = width;
      wd->csw->y = y;
      wd->csw->x = x;
      wd->csw->style = dwStyle;
      wd->csw->lpszName = GPOINTER_TO_UINT (windowname) > G_MAXUINT16
                        ? g_utf8_to_utf16 (windowname, -1, NULL, NULL, NULL)
                        : (gunichar2 *)windowname;
      wd->csw->lpszClass = GPOINTER_TO_UINT (classname) > G_MAXUINT16
                        ? g_utf8_to_utf16 (classname, -1, NULL, NULL, NULL)
                        : (gunichar2 *)classname;
      wd->csw->dwExStyle = dwExStyle;
    }
  else
    {
      wd->csa = g_malloc (sizeof (CREATESTRUCTA));
      wd->csa->lpCreateParams = lpvParam;
      wd->csa->hInstance = hInstance;
      wd->csa->hMenu = hMenu;
      wd->csa->hwndParent = hWndParent;
      wd->csa->cy = height;
      wd->csa->cx = width;
      wd->csa->y = y;
      wd->csa->x = x;
      wd->csa->style = dwStyle;
      wd->csa->lpszName = GPOINTER_TO_UINT (windowname) > G_MAXUINT16
                                ? w32ldr_utf8_to_mb (windowname) : windowname;
      wd->csa->lpszClass = GPOINTER_TO_UINT (classname) > G_MAXUINT16
                                ? w32ldr_utf8_to_mb (classname) : classname;
      wd->csa->dwExStyle = dwExStyle;
    }
  if (GTK_IS_WINDOW (widget))
    {
      if (x != CW_USEDEFAULT && y != CW_USEDEFAULT)
        gtk_window_move (GTK_WINDOW (widget), x, y);
      if (width != CW_USEDEFAULT && height != CW_USEDEFAULT)
        gtk_window_set_default_size (GTK_WINDOW (widget), width, height);
    }
  else if (hWndParent)
    {
      GtkWidget *fixed;
      W32LdrOverlapData *od;

      if (wd->resizable && width != CW_USEDEFAULT && height != CW_USEDEFAULT)
        gtk_widget_set_size_request (widget, width, height);
      od = g_object_get_data (G_OBJECT (hWndParent), "user_data");
      i = 0;
      if (od->wd.child)
        while (od->wd.child[i])
          i++;
      od->wd.child = g_realloc (od->wd.child, sizeof (GtkWidget *) * (i + 2));
      od->wd.child[i] = widget;
      od->wd.child[i + 1] = NULL;
      /* ja:直近のラベルにアクセラレータを設定する */
      if (wd->focus && !GTK_IS_BUTTON (widget) && !GTK_IS_LABEL (widget))
        while (--i >= 0)
          if (GTK_IS_LABEL (od->wd.child[i])
            && !gtk_label_get_mnemonic_widget (GTK_LABEL (od->wd.child[i])))
            {
              gtk_label_set_mnemonic_widget (GTK_LABEL (od->wd.child[i]),
                                                                    wd->focus);
              break;
            }
      /* ja:フィクストに配置する */
      fixed = GTK_IS_FIXED (hWndParent) ? hWndParent : od->fixed;
      if (fixed)
        gtk_fixed_put (GTK_FIXED (fixed), widget, x, y);
    }
  g_signal_connect (G_OBJECT (widget), "realize",
                                            G_CALLBACK (w32wnd_realize), NULL);
  g_signal_connect_after (G_OBJECT (widget), "destroy",
                                            G_CALLBACK (w32wnd_destroy), NULL);
  if (dwStyle & WS_DISABLED)
    gtk_widget_set_sensitive (widget, FALSE);
  if (dwStyle & WS_VISIBLE)
    gtk_widget_show (widget);
  return widget;
}
