/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "avifmt.h"
#include "misc/misc.h"


/******************************************************************************
*                                                                             *
* ja:シナリオオブジェクト関数                                                 *
*                                                                             *
******************************************************************************/
/*  ja:AVIの一部をシナリオオブジェクトに変換する
    avi_edit,AVI編集ハンドルへのポインタ
      stream,ストリーム番号
       start,変換をはじめるサンプル番号
     samples,サンプル数
         RET,シナリオオブジェクト,NULL:エラー                               */
static gchar *
avi_edit_at_scenario (AviEdit    *avi_edit,
                      const gint  stream,
                      const gint  start,
                      const gint  samples)
{
  gchar *scenario = NULL;
  gint length = 0, start_n, samples_n;

  /* ja:スタート位置とサンプル数の調整 */
  if (start < 0)
    return NULL;
  samples_n = MIN (avi_edit_length (avi_edit) - start, samples);
  if (samples_n <= 0 || avi_edit_length (avi_edit) < samples_n)
    return NULL;

  start_n = start;
  while (samples_n > 0)
    {
      gchar *text;
      gint leng;

      avi_base_get_file (avi_edit, start_n);/* ja:AVIファイルを求める */
      if (avi_edit->file->name)
        {
          /* ja:ファイル */
          text = g_strdup_printf ("%d,file,\'%s\',%d,%d,%d\n", stream,
                avi_edit->file->name, avi_edit->file->param,
                avi_edit->file->start + start_n - avi_edit->offset,
                MIN (avi_edit->file->length - (start_n - avi_edit->offset),
                                                                samples_n));
          samples_n -= avi_edit->file->length - (start_n - avi_edit->offset);
          start_n += avi_edit->file->length - (start_n - avi_edit->offset);
        }
      else if (avi_edit->file->avi_memory && avi_edit->file->bmih)
        {
          const guint8 *p;
          gint i, j = 0, size;

          /* ja:メモリ内ビットマップ */
          size = bm_all_bytes (avi_edit->file->bmih);
          text = g_strdup_printf ("%d,bitmap,%d,\\\n", stream, size);
          leng = g_strlen (text);
          text = g_realloc (text, (leng + size * 3 + (size / 16) * 2 + 2)
                                                            * sizeof (gchar));
          size = bm_header_bytes (avi_edit->file->bmih);
          for (i = 0; i < size; i++)
            {
              g_snprintf (text + leng, 4, " %02x",
                                        ((guint8 *)avi_edit->file->bmih)[i]);
              leng += 3;
              if (j % 16 == 15)
                {
                  text[leng++] = '\\';
                  text[leng++] = '\n';
                }
              j++;
            }
          size = bm_image_bytes (avi_edit->file->bmih);
          p = avi_memory_get_data (avi_edit->file->avi_memory);
          for (i = 0; i < size; i++)
            {
              g_snprintf (text + leng, 4, " %02x", p[i]);
              leng += 3;
              if (j % 16 == 15 && i < size - 1)
                {
                  text[leng++] = '\\';
                  text[leng++] = '\n';
                }
              j++;
            }
          text[leng++] = '\n';
          text[leng] = '\0';
          samples_n--;
          start_n++;
        }
      else
        {
          g_free (scenario);
          return NULL;
        }
      leng = g_strlen (text);
      scenario = g_realloc (scenario, (length + leng + 1) * sizeof (gchar));
      g_memmove (scenario + length, text, leng);
      g_free (text);
      length += leng;
    }
  scenario[length] = '\0';
  return scenario;
}


/*  ja:AVIをシナリオオブジェクトに変換する
      avi_edit,AVI編集ハンドルへのポインタ
    interleave,TRUE:インターリーブする,FALSE:インターリーブしない
           RET,シナリオオブジェクト,NULL:エラー                             */
gchar *
avi_edit_to_scenario (AviEdit        **avi_edit,
                      const gboolean   interleave)
{
  gchar *scenario = NULL;
  gint i, leng, length = 0;

  if (!avi_edit)
    return NULL;
  /* ja:ヘッタ */
  scenario = g_strdup ("#VMAID\n-1,Video maid scenario object,0\n\n");
  length = g_strlen (scenario);
  for (i = 0; avi_edit[i]; i++)
    {
      gchar *text;

      switch (avi_edit_type (avi_edit[i]))
        {
          case AVI_TYPE_VIDEO:
            text = g_strdup_printf ("%d,type,video\n"
                                    "%d,rate,%u\n"
                                    "%d,scale,%u\n"
                                    "%d,priority,%u\n"
                                    "%d,language,%X\n"
                                    "%d,bmih,%d,%d,%d,%d,%d\n",
                        i,
                        i, avi_edit_get_rate (avi_edit[i]),
                        i, avi_edit_get_scale (avi_edit[i]),
                        i, avi_edit_get_priority (avi_edit[i]),
                        i, avi_edit_get_language (avi_edit[i]),
                        i,  bmih_get_width (avi_edit[i]->bmih),
                            bmih_get_height (avi_edit[i]->bmih),
                            bmih_get_bit_count (avi_edit[i]->bmih),
                            bmih_get_x_pixels_per_meter (avi_edit[i]->bmih),
                            bmih_get_y_pixels_per_meter (avi_edit[i]->bmih));
            break;
          case AVI_TYPE_AUDIO:
            text = g_strdup_printf ("%d,type,audio\n"
                                    "%d,priority,%u\n"
                                    "%d,language,%X\n",
                                i,
                                i, avi_edit_get_priority (avi_edit[i]),
                                i, avi_edit_get_language (avi_edit[i]));
            break;
          default:
            g_free (scenario);
            return NULL;
        }
      leng = g_strlen (text);
      scenario = g_realloc (scenario, (length + leng + 2) * sizeof (gchar));
      g_memmove (scenario + length, text, leng);
      g_free (text);
      length += leng;
      if (avi_edit_get_name (avi_edit[i]))
        {
          gchar *name;

          name = g_strescape (avi_edit_get_name (avi_edit[i]), NULL);
          text = name ? g_strdup_printf ("%d,name,\"%s\"\n", i, name)
                      : g_strdup_printf ("%d,name,\'%s\'\n", i,
                                            avi_edit_get_name (avi_edit[i]));
          g_free (name);
          leng = g_strlen (text);
          scenario = g_realloc (scenario,
                                        (length + leng + 2) * sizeof (gchar));
          g_memmove (scenario + length, text, leng);
          length += leng;
          g_free (text);
        }
    }
  if (i <= 0)
    {
      g_free (scenario);
      return NULL;
    }
  scenario[length++] = '\n';
  scenario[length] = '\0';
  /* ja:本体 */
  if (interleave)
    {
      gint current = 0, *pos, streams, timer = 0, times;

      /* ja:インターリーブ */
      for (streams = 0; avi_edit[streams]; streams++);
      /* ja:最も1秒間のフレーム数が少ないストリームを探す */
      for (current = 0; current < streams; current++)
        if (avi_edit_type (avi_edit[current]) == AVI_TYPE_VIDEO)
          break;
      if (current >= streams)
        {
          current = -1;
          times = 1000;
        }
      else
        {
          for (i = 0; i < streams; i++)
            if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                        && (glonglong)avi_edit_get_rate (avi_edit[i])
                                    * avi_edit_get_scale (avi_edit[current])
                            < (glonglong)avi_edit_get_rate (avi_edit[current])
                                            * avi_edit_get_scale (avi_edit[i]))
              current = i;
            times = avi_time_length (avi_edit[current])
                                        / avi_edit_length (avi_edit[current]);
        }
      pos = g_malloc0 (streams * sizeof (gint));
      while (TRUE)
        {
          /* ja:すべて出力したら終了 */
          for (i = 0; i < streams; i++)
            if (pos[i] < avi_edit_length (avi_edit[i]))
              break;
          if (i >= streams)
            break;
          for (i = 0; i < streams; i++)
            if (pos[i] < avi_edit_length (avi_edit[i]))
              {
                gchar *text;
                gint samples;

                samples = current > 0 ? i == current ? 1 : avi_time_sync_sample
                    (avi_edit[current], pos[current] + 1, avi_edit[i]) - pos[i]
                            : avi_time_to_sample (avi_edit[i], timer) - pos[i];
                text = avi_edit_at_scenario (avi_edit[i], i, pos[i], samples);
                if (text)
                  {
                    leng = g_strlen (text);
                    scenario = g_realloc (scenario,
                                        (length + leng + 1) * sizeof (gchar));
                    g_memmove (scenario + length, text, leng);
                    g_free (text);
                    length += leng;
                    if (i != current)
                      pos[i] += samples;
                  }
              }
          if (current >= 0)
            {
              pos[current]++;
              if (avi_edit_length (avi_edit[current]) <= pos[current])
                current = -1;
            }
          else
            {
              timer += times;
            }
        }
      g_free (pos);
    }
  else
    {
      /* ja:非インターリーブ */
      for (i = 0; avi_edit[i]; i++)
        {
          gchar *text;

          text = avi_edit_at_scenario (avi_edit[i], i,
                                            0, avi_edit_length (avi_edit[i]));
          leng = g_strlen (text);
          scenario = g_realloc (scenario,
                                        (length + leng + 1) * sizeof (gchar));
          g_memmove (scenario + length, text, leng);
          g_free (text);
          length += leng;
        }
    }
  scenario[length] = '\0';
  return scenario;
}


/*  ja:シナリオオブジェクトからAVIを作る
    scenario,シナリオオブジェクト
         RET,AVI編集ハンドルへのポインタ,NULL:エラー                        */
AviEdit **
avi_edit_from_scenario (const gchar *scenario)
{
  gchar **lines, *text;
  gint i, j = 0, count = 0;
  AviEdit **avi_edit = NULL;

  if (!scenario)
    return NULL;
  text = g_strdup (scenario);
  /* ja:CRLFをLFに変換する */
  for (i = 0; text[i] != '\0'; i++)
    if (text[i] != '\r' || text[i + 1] != '\n')
      text[j++] = text[i];
  text[j] = '\0';
  /* ja:CRをLFに変換する */
  for (i = 0; text[i] != '\0'; i++)
    if (text[i] == '\r')
      text[i] = '\n';
  /* ja:便宜改行を真の改行に変換する */
  i = j = 0;
  while (text[i] != '\0')
    if (text[i] == '\\' && text[i + 1] == '\n')
      i += 2;
    else
      text[j++] = text[i++];
  text[j] = '\0';
  lines = g_strsplit (text, "\n", 0);
  g_free (text);
  for (i = 0; lines[i]; i++)
    {
      gboolean quote[2];
      gchar **cmd;
      gint stream;

      /* ja:コメント部分を削除,カンマを改行にする */
      quote[0] = quote[1] = FALSE;
      for (j = 0; (lines[i])[j] != '\0'; j++)
        if ((lines[i])[j] == '\'' && !quote[1])
          {
            quote[0] = !quote[0];
          }
        else if ((lines[i])[j] == '\"' && !quote[0] && !quote[1])
          {
            quote[1] = TRUE;
          }
        else if ((lines[i])[j] == '\"' && !quote[0] && quote[1]
                                                && (lines[i])[j - 1] != '\\')
          {
            quote[1] = FALSE;
          }
        else if (!quote[0] && !quote[1])
          {
            if ((lines[i])[j] == ',')
              {
                (lines[i])[j] = '\n';
              }
            else if ((lines[i])[j] == '#')
              {
                (lines[i])[j] = '\0';
                break;
              }
          }
      g_strstrip (lines[i]);
      cmd = g_strsplit (lines[i], "\n", 0);
      for (j = 0; cmd[j]; j++)
        {
          gsize leng;

          g_strstrip (cmd[j]);
          leng = g_strlen (cmd[j]) - 1;
          if (leng >= 1)
            {
              if ((cmd[j])[0] == '\'' && (cmd[j])[leng] == '\'')
                {
                  g_memmove (cmd[j], cmd[j] + 1, --leng * sizeof (gchar));
                  (cmd[j])[leng] = '\0';
                }
              else if ((cmd[j])[0] == '\"' && (cmd[j])[leng] == '\"')
                {
                  gchar *tmp;

                  (cmd[j])[leng] = '\0';
                  tmp = g_strcompress (cmd[j] + 1);
                  if (tmp)
                    {
                      g_free (cmd[j]);
                      cmd[j] = tmp;
                    }
                }
            }
        }
      if (cmd[0] && cmd[1]
                && misc_str_to_val (&stream, cmd[0], 10, TRUE) && stream >= 0)
        {
          gint length, size, value;

          if (count <= stream)/* ja:拡張 */
            {
              avi_edit = g_realloc (avi_edit,
                                            (stream + 1) * sizeof (AviEdit *));
              g_memset (avi_edit + count, 0,
                                    (stream - count + 1) * sizeof (AviEdit *));
              count = stream + 1;
            }
          if (!avi_edit[stream])/* ja:新規 */
            avi_edit[stream] = g_malloc0 (sizeof (AviEdit));
          if (g_ascii_strcasecmp (cmd[1], "type") == 0 && cmd[2])
            {
              if (g_ascii_strcasecmp (cmd[2], "audio") == 0)
                avi_edit[stream]->type = AVI_TYPE_AUDIO;
              else if (g_ascii_strcasecmp (cmd[2], "video") == 0)
                avi_edit[stream]->type = AVI_TYPE_VIDEO;
            }
          else if (g_ascii_strcasecmp (cmd[1], "rate") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 10, FALSE))
                avi_edit_set_rate (avi_edit[stream], value);
            }
          else if (g_ascii_strcasecmp (cmd[1], "scale") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 10, FALSE))
                avi_edit_set_scale (avi_edit[stream], value);
            }
          else if (g_ascii_strcasecmp (cmd[1], "priority") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 10, FALSE))
                avi_edit_set_priority (avi_edit[stream], (guint16)value);
            }
          else if (g_ascii_strcasecmp (cmd[1], "language") == 0 && cmd[2])
            {
              if (misc_str_to_val (&value, cmd[2], 16, FALSE))
                avi_edit_set_language (avi_edit[stream], (guint16)value);
            }
          else if (g_ascii_strcasecmp (cmd[1], "name") == 0 && cmd[2])
            {
              avi_edit_set_name (avi_edit[stream], cmd[2]);
            }
          else if (g_ascii_strcasecmp (cmd[1], "bmih") == 0)
            {
              for (j = 2; cmd[j]; j++);
              if (j >= 7)
                {
                  if (!avi_edit[stream]->bmih)/* ja:新規 */
                    {
                      avi_edit[stream]->bmih = g_malloc0 (BMIH_SIZE);
                      bmih_set_size (avi_edit[stream]->bmih, BMIH_SIZE);
                      bmih_set_planes (avi_edit[stream]->bmih, 1);
                      bmih_set_compression (avi_edit[stream]->bmih,
                                                                BI_COMP_RGB);
                    }
                  if (misc_str_to_val (&value, cmd[2], 10, TRUE) && value > 0)
                    bmih_set_width (avi_edit[stream]->bmih, value);
                  if (misc_str_to_val (&value, cmd[3], 10, TRUE) && value > 0)
                    bmih_set_height (avi_edit[stream]->bmih, value);
                  if (misc_str_to_val (&value, cmd[4], 10, FALSE))
                    {
                      bmih_set_bit_count (avi_edit[stream]->bmih, value);
                      switch (value)
                        {
                          case  1:
                            bmih_set_color_used (avi_edit[stream]->bmih, 2);
                            break;
                          case  4:
                            bmih_set_color_used (avi_edit[stream]->bmih, 16);
                            break;
                          case  8:
                            bmih_set_color_used (avi_edit[stream]->bmih, 256);
                            break;
                          case 16:
                          case 24:
                          case 32:
                            bmih_set_color_used (avi_edit[stream]->bmih, 0);
                        }
                    }
                  if (misc_str_to_val (&value, cmd[5], 10, FALSE))
                    bmih_set_x_pixels_per_meter (avi_edit[stream]->bmih,
                                                                        value);
                  if (misc_str_to_val (&value, cmd[6], 10, FALSE))
                    bmih_set_y_pixels_per_meter (avi_edit[stream]->bmih,
                                                                        value);
                  if (bmih_get_width (avi_edit[stream]->bmih) <= 0
                        || bmih_get_height (avi_edit[stream]->bmih) <= 0
                        || bmih_get_bit_count (avi_edit[stream]->bmih) == 0)
                    {
                      g_free (avi_edit[stream]->bmih);
                      avi_edit[stream]->bmih = NULL;
                    }
                  else
                    {
                      avi_edit[stream]->bmih
                                = g_realloc (avi_edit[stream]->bmih,
                                    bm_header_bytes (avi_edit[stream]->bmih));
                    }
                }
            }
          else if (g_ascii_strcasecmp (cmd[1], "file") == 0 && cmd[2])
            {
              gint param, type;
              AviFile *avi_file;

              if (!cmd[3] || !misc_str_to_val (&param, cmd[3], 10, FALSE))
                param = 0;
              avi_file = avi_file_open (cmd[2], param, &type,
                                avi_edit_get_rate (avi_edit[stream]) <= 0
                                        ? &avi_edit[stream]->rate : NULL,
                                avi_edit_get_scale (avi_edit[stream]) <= 0
                                        ? &avi_edit[stream]->scale : NULL,
                                !avi_edit[stream]->file
                                        ? &avi_edit[stream]->priority : NULL,
                                !avi_edit[stream]->file
                                        ? &avi_edit[stream]->language : NULL,
                                !avi_edit_get_name (avi_edit[stream])
                                        ? &avi_edit[stream]->name : NULL);
              if (avi_file)
                {
                  if (avi_edit_type (avi_edit[stream]) == AVI_TYPE_UNKNOWN)
                    {
                      avi_edit[stream]->type = type;
                    }
                  else if (avi_edit_type (avi_edit[stream]) != type)
                    {
                      avi_file_close (avi_file);
                      avi_file = NULL;
                    }
                }
              if (avi_file)
                {
                  gint start;

                  if (!cmd[3] || !cmd[4]
                            || !misc_str_to_val (&start, cmd[4], 10, TRUE))
                    start = -1;
                  if (!cmd[3] || !cmd[4] || !cmd[5]
                            || !misc_str_to_val (&length, cmd[5], 10, TRUE))
                    length = -1;
                  if (0 < start && start < avi_file->length)
                    {
                      avi_file->start = start;
                      avi_file->length -= start;
                    }
                  if (0 < length && length < avi_file->length)
                      avi_file->length = length;
                  if (!avi_edit[stream]->file)
                    {
                      /* ja:先頭 */
                      avi_edit[stream]->file = avi_file;
                    }
                  else if (g_strfilecmp (avi_edit[stream]->file->name,
                                                        avi_file->name) == 0
                      && avi_edit[stream]->file->start
                        + avi_edit[stream]->file->length == avi_file->start)
                    {
                      /* ja:連続 */
                      avi_edit[stream]->file->length += avi_file->length;
                      avi_file_close (avi_file);
                    }
                  else
                    {
                      /* ja:追加 */
                      avi_edit[stream]->file->next = avi_file;
                      avi_file->prev = avi_edit[stream]->file;
                      avi_edit[stream]->file = avi_file;
                    }
                }
            }
          else if (g_ascii_strcasecmp (cmd[1], "bitmap") == 0 && cmd[2]
                                && misc_str_to_val (&size, cmd[2], 10, FALSE)
                                && size > BMIH_SIZE)
            {
              gsize s;
              gpointer a;

              a = misc_str_to_array (&s, 8, cmd[3], 16, FALSE);
              if (a && s == size && bm_all_bytes (a) == size)
                {
                  gint type;
                  AviFile *avi_file;

                  avi_file = avi_file_open_memory (a, &type);
                  if (avi_file && type == AVI_TYPE_VIDEO)
                    {
                      if (!avi_edit[stream]->file)
                        {
                          /* ja:先頭 */
                          avi_edit[stream]->file = avi_file;
                        }
                      else
                        {
                          /* ja:追加 */
                          avi_edit[stream]->file->next = avi_file;
                          avi_file->prev = avi_edit[stream]->file;
                          avi_edit[stream]->file = avi_file;
                        }
                    }
                }
            }
        }
      g_strfreev (cmd);
    }
  g_strfreev (lines);
  if (avi_edit)
    {
      for (i = 0; i < count; i++)
        {
          /* ja:サンプル数 */
          if (avi_edit[i]->file)
            {
              AviFile *avi_file;

              while (avi_edit[i]->file->prev)
                avi_edit[i]->file = avi_edit[i]->file->prev;
              for (avi_file = avi_edit[i]->file; avi_file;
                                                    avi_file = avi_file->next)
                avi_edit[i]->length += avi_file->length;
            }
          if (avi_edit_length (avi_edit[i]) <= 0)
            {
              avi_edit_close (avi_edit[i]);
              avi_edit[i] = NULL;
            }
          /* ja:展開後のフォーマット */
          if (avi_edit[i] && !avi_edit[i]->bmih
                            && avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            {
              if (avi_edit[i]->file->handler == AVI_STREAM_COMP_DIB)
                {
                  avi_edit[i]->bmih = g_memdup (avi_edit[i]->file->bmih,
                                    bm_header_bytes (avi_edit[i]->file->bmih));
                  if (bmih_get_compression (avi_edit[i]->bmih)
                                                        == AVI_STREAM_COMP_DIB)
                    bmih_set_compression (avi_edit[i]->bmih, BI_COMP_RGB);
                  if (bmih_get_size_image (avi_edit[i]->bmih) <= 0)
                    bmih_set_size_image (avi_edit[i]->bmih,
                                        bm_image_bytes (avi_edit[i]->bmih));
                }
              else
                {
                  IcmObject *icm_object;

                  /* ja:圧縮 */
                  icm_object = icm_open (avi_edit[i]->file->handler,
                                                        ICM_MODE_DECOMPRESS);
                  if (icm_object)
                    {
                      avi_edit[i]->bmih = g_malloc
                                (icm_decompress_get_format_size (icm_object,
                                                    avi_edit[i]->file->bmih));
                      if (icm_decompress_get_format (icm_object,
                                    avi_edit[i]->file->bmih, avi_edit[i]->bmih)
                                                    & icm_close (icm_object))
                        {
                          avi_edit[i]->bmih = g_realloc (avi_edit[i]->bmih,
                                        bm_header_bytes (avi_edit[i]->bmih));
                        }
                      else
                        {
                          avi_edit_close (avi_edit[i]);
                          avi_edit[i] = NULL;
                        }
                    }
                  else
                    {
                      avi_edit_close (avi_edit[i]);
                      avi_edit[i] = NULL;
                    }
                }
            }
          else if (avi_edit[i] && !avi_edit[i]->wfx
                            && avi_edit_type (avi_edit[i]) == AVI_TYPE_AUDIO)
            {
              if (wfx_get_format_tag (avi_edit[i]->file->wfx) == WAVE_FMT_PCM)
                {
                  avi_edit[i]->wfx = g_memdup (avi_edit[i]->file->wfx,
                                    wf_header_bytes (avi_edit[i]->file->wfx));
                }
              else
                {
                  AcmObject *acm_object;

                  /* ja:圧縮 */
                  acm_object = acm_open
                                (wfx_get_format_tag (avi_edit[i]->file->wfx));
                  if (acm_object && (avi_edit[i]->wfx
                                    = g_malloc (acm_decompress_get_format_size
                                        (acm_object, avi_edit[i]->file->wfx)))
                        && (acm_decompress_get_format (acm_object,
                                    avi_edit[i]->file->wfx, avi_edit[i]->wfx)
                                                    & acm_close (acm_object)))
                    {
                      avi_edit[i]->rate
                            = wfx_get_average_bytes_per_sec (avi_edit[i]->wfx);
                      avi_edit[i]->scale
                            = wfx_get_block_align (avi_edit[i]->wfx);
                    }
                  else
                    {
                      avi_edit_close (avi_edit[i]);
                      avi_edit[i] = NULL;
                    }
                }
            }
#ifdef XXX
          g_print("type=%X,rate=%u,scale=%u,length=%d\n",avi_edit[i]->type,avi_edit[i]->rate,avi_edit[i]->scale,avi_edit[i]->length);
          for (avi_file=avi_edit[i]->file;avi_file!=NULL;
                                                  avi_file=avi_file->next)
              g_print("%s,start=%d,length=%d\n",avi_file->name,avi_file->start,avi_file->length);
#endif
        }
      j = 0;
      for (i = 0; i < count; i++)
        if (avi_edit[i])
          avi_edit[j++] = avi_edit[i];
      if (j > 0)
        {
          avi_edit = g_realloc (avi_edit, (j + 1) * sizeof (AviEdit *));
          avi_edit[j] = NULL;
        }
      else
        {
          g_free (avi_edit);
          avi_edit = NULL;
        }
    }
  return avi_edit;
}
