/*
    chunk.h
    copyright (c) 1998-2019 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __CHUNK_H__
#define __CHUNK_H__


#include <limits.h>
#include <stdio.h>


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#if UINT_MAX == 4294967295
typedef unsigned cksize_t;
typedef unsigned fourcc_t;
#elif ULONG_MAX == 4294967295
typedef unsigned long cksize_t;
typedef unsigned long fourcc_t;
#elif USHRT_MAX == 4294967295
typedef unsigned short cksize_t;
typedef unsigned short fourcc_t;
#else
#error Unsigned 32 bit integer type was not found.
#endif


typedef struct chunklist
{
  long offset;                  /* オフセット */
  fourcc_t id;                  /* ID */
  cksize_t size;                /* サイズ */
  fourcc_t form;                /* フォームタイプ */
} CHUNKLIST;
typedef struct chunk
{
  int num;                      /* リストの数 */
  struct chunklist *list;
} CHUNK;


/*  4つの文字からIDを作る
     c1,1文字目
     c2,2文字目
     c3,3文字目
     c4,4文字目
    RET,ID                                                                  */
fourcc_t
make4cc (const char c1,
         const char c2,
         const char c3,
         const char c4);


/*  4つの文字からIDを作る
    fourcc,ID
         n,順番(1..4)
       RET,文字                                                             */
char
getidcc (const fourcc_t fourcc,
         const int      n);


/*  ファイルポインタからチャンク構造体を作る
     fp,ファイルポインタ
    RET,チャンク構造体                                                      */
CHUNK *
chunkopen (FILE *fp);


/*  チャンク構造体を解放する
    RET,0:成功,-1:エラー                                                    */
int
chunkfree (CHUNK *ck);


/*  チャンクの中に入る
     fp,ファイルポインタ
     ck,チャンク構造体
    RET,0:成功,-1:エラー                                                    */
int
chunkin (FILE  *fp,
         CHUNK *ck);


/*  チャンクから出る
     fp,ファイルポインタ
     ck,チャンク構造体
    RET,0:成功,-1:エラー                                                    */
int
chunkout (FILE  *fp,
          CHUNK *ck);


/*  次のチャンチャンクに移動する
     fp,ファイルポインタ
     ck,チャンク構造体
    RET,0:成功,-1:エラー                                                    */
int
chunknext (FILE  *fp,
           CHUNK *ck);


/*  チャンクのIDを取得する
     fp,ファイルポインタ
    RET,ID                                                                  */
fourcc_t
chunkid (FILE *fp);


/*  チャンクのサイズを取得する
     fp,ファイルポインタ
    RET,フォーム                                                            */
cksize_t
chunksize (FILE *fp);


/*  チャンクのフォームを取得する
     fp,ファイルポインタ
    RET,フォーム                                                            */
fourcc_t
chunkform (FILE *fp);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __CHUNK_H__ */
