/*
    Copyright 2016-2018 SOFT-ERG, Przemek Kuczmierczyk (www.softerg.com)
    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice,
       this list of conditions and the following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
    AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
    THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
    (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
    LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
    ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
    NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
    EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

// -----------------------------------------------------------------------------
namespace vpp {
// -----------------------------------------------------------------------------

/**
    \brief Represents a set of swappable on-screen images.

    In typical situations, final images generated by the rendering engine
    are immediately displayed. This is the case of all game engines and
    other interactive applications. The SwapChain class provides a set of
    images (represented by untyped views, i.e. FrameImageView) that can
    be used as rendering targets and then scheduled for actual display on screen.
    There is more than one image in that set, to facilitate double buffering.

    This object is reference-counted and may be passed by value.
*/

class SwapChain
{
public:
    /** \brief Constructs null reference. */
    SwapChain();

    /** \brief Constructs a swapchain for given Surface and Device pair.
    
        In order to construct a swapchain, you need both Device and Surface object.

        By default, the constructor makes a swapchain with system-defined number
        of images. This is recommended, since the minimum and maximum number of images
        are platform-dependent. Do not hardcode any particular number, as it might
        turn out to be unsupported on some combination of OS/graphic card. You can
        retrieve these limits by calling the Surface::getCapabilities() method.
        Provide zero value (or leave default one) to instruct VPP to determine
        the number of images automatically.

        You can also specify image queuing mode as \c imageQueuingMode parameter.
        Unfortunately, availablility of these is also system-dependent and
        popular contemporary cards support only limited selection. You can
        retrieve supported modes by calling the Surface::getPresentationModes()
        method. The SwapChain constructor will try to use specified mode
        if possible, or fallback to another mode if not. See official Vulkan
        docs (the version with extensions docs, look for VkPresentModeKHR description)
        for complete list of queuing modes and their meaning.

        The \c hReplacedSwapChain parameter is useful when we are creating
        a swapchain for the surface that already has a swapchain. The old
        swapchain will always be discarded, but providing a reference to it here
        allows to reuse some resources, making creating the new swapchain faster.
    */
    SwapChain (
        const Device& hDevice,
        const Surface& hSurface,
        unsigned int imageCount = 0,
        VkPresentModeKHR imageQueuingMode = VK_PRESENT_MODE_MAILBOX_KHR,
        const SwapChain& hReplacedSwapChain = SwapChain() );

    /** \brief Checks whether this SwapChain object refers to a valid swapchain. */
    bool valid() const;

    /** \brief Retrieves the Vulkan handle. */
    VkSwapchainKHR handle() const;

    /** \brief Retrieves the device. */
    const Device& device() const;

    /** \brief Retrieves the surface. */
    const Surface& surface() const;

    /** \brief Retrieves the number of images in the swapchain. */
    size_t views() const;

    /** \brief Retrieves the image view for given index. */
    FrameImageView view ( size_t index ) const;

    /** \brief Acquires and locks an image view for rendering.
    
        Call this method to obtain an index of image. Retrieve the
        image view by calling view().

        The image is supposed to be rendered and subsequently scheduled
        for display on the screen via presentDisplayImage().

        This method also handles layout transitions and all other
        internal Vulkan requirements.
    */
    unsigned int acquireDisplayImage ( const Queue& hQueue );

    /** \brief Schedules an image view for display.
    
        Call this method to queue a swapchain image for display.

        The semantics of acquireDisplayImage()/presentDisplayImage() pair
        is like resource allocation/releasing. Each call to acquireDisplayImage()
        must be paired with presentDisplayImage().

        This method also handles layout transitions and all other
        internal Vulkan requirements.
    */
    void presentDisplayImage ( const Queue& hQueue, unsigned int iImage );
};

// -----------------------------------------------------------------------------
} // namespace vpp
// -----------------------------------------------------------------------------
