/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2007 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import org.apache.commons.io.IOUtils;

/**
 * DefaultByLineFileScanner
 * Scan file line by line.
 *
 * @author Yomei Komiya
 * @since 0.9.2
 *
 * @version $Id: DefaultByLineFileScanner.java 6 2007-06-27 13:49:06Z whitestar $
 */
public class DefaultByLineFileScanner implements FileScanner {
	
	/**
	 * File line consumer 
	 */
	private LineConsumer lineConsumer = null;
	

	public DefaultByLineFileScanner() {
		super();
	}
	
	
	public DefaultByLineFileScanner(LineConsumer lineConsumer) {
		super();
		
		this.lineConsumer = lineConsumer;
	}


	/*
	 * @see jp.sourceforge.webframe.io.FileScanner#scan(java.lang.String, java.lang.String)
	 */
	public void scan(
			String filePath,
			String fileEncodingCharset) 
		throws ScanAbortException {

		File file = new File(filePath);
		
		this.scan(file, fileEncodingCharset);
	}
	
	
	/*
	 * @see jp.sourceforge.webframe.io.FileScanner#scan(java.io.File, java.lang.String)
	 */
	public void scan(
			File file,
			String fileEncodingCharset)
		throws ScanAbortException {

		BufferedReader br = null;
		try {
			br	= new BufferedReader(
					new InputStreamReader(
							new FileInputStream(file), fileEncodingCharset));
		
			FileLine fileLine = new FileLine();
			fileLine.setFilePath(file.getAbsolutePath());
			int lineNumber = 0;
			String lineString;
			while ((lineString = br.readLine()) != null) {
				lineNumber++;
				fileLine.setLineNumber(lineNumber);
				fileLine.setLineString(lineString);
			
				this.lineConsumer.consume(fileLine);
			}
		}
		catch (ScanAbortException sae) {
			sae.printStackTrace();
			throw sae;
		}
		catch (IOException ioe) {
			ioe.printStackTrace();
			throw new ScanAbortException(ioe.getMessage(), ioe.getCause());
		}
		finally {
			IOUtils.closeQuietly(br);
		}
	}


	public LineConsumer getLineConsumer() {
		return lineConsumer;
	}


	public void setLineConsumer(LineConsumer lineConsumer) {
		this.lineConsumer = lineConsumer;
	}
	
}
