/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2007 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.springframework.util;

import org.apache.commons.collections.BidiMap;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.context.ConfigurableApplicationContext;

/**
 * SpringContextHolder
 *
 * @author Yomei Komiya
 * @since 0.9.2
 *
 * @version $Id: SpringContextHolder.java 6 2007-06-27 13:49:06Z whitestar $
 */
public class SpringContextHolder {
	
	/**
	 * ConfigurableApplicationContext object
	 */
	private ConfigurableApplicationContext applicationContext = null;
	
	/**
	 * BeanHolder array
	 */
	private SpringBeanHolder[] beanHolders = null;
	
	/**
	 * Singleton bean name <-> object ID bi-directional map
	 */
	private BidiMap singletonNameObjectIdMap = null;


	/**
	 * Constructor
	 * 
	 * @param applicationContext ConfigurableApplicationContext object
	 */
	public SpringContextHolder(
			ConfigurableApplicationContext applicationContext) {
		super();
		
		this.applicationContext = applicationContext;
		this.beanHolders
			= SpringUtils.getSpringBeanHolders(applicationContext);
		this.singletonNameObjectIdMap
			= SpringUtils.getSpringBeanSingletonNameObjectIdMap(this.beanHolders);
	}
	
	
	public ConfigurableApplicationContext getApplicationContext() {
		return applicationContext;
	}


	public BidiMap getSingletonNameObjectIdMap() {
		return singletonNameObjectIdMap;
	}


	public SpringBeanHolder[] getBeanHolders() {
		return beanHolders;
	}


	public SpringBeanHolder[] getBeanHoldersForType(String typeName)
		throws ClassNotFoundException {
		
		Class type = Class.forName(typeName);
		return SpringUtils.getSpringBeanHoldersForType(
				this.applicationContext, type);
	}
	
	
	public SpringBeanHolder[] getBeanHoldersForType(Class type) {
		return SpringUtils.getSpringBeanHoldersForType(
				this.applicationContext, type);
	}
	
	
	public SpringBeanHolder getBeanHolder(String beanName) {
		return
			SpringUtils.getSpringBeanHolder(this.applicationContext, beanName);
	}
	
	
	public String getSingletonObjectIdByName(String beanName) {
		return (String)(this.singletonNameObjectIdMap.get(beanName));
	}
	
	
	/**
	 * Get bean name by actual singleton's object ID 
	 * 
	 * @param objectId Singleton object ID (hex string)
	 * @return Bean name
	 * @throws NoSuchBeanDefinitionException
	 */
	public String getBeanNameBySingletonObjectId(String objectId)
		throws NoSuchBeanDefinitionException {
		
		String beanName = null;
		beanName = (String)
			(this.singletonNameObjectIdMap.inverseBidiMap().get(objectId));
		
		if (beanName != null) {
			return beanName;
		}
		else {
			throw new NoSuchSingletonBeanException(
					"No such singleton bean (object ID: " + objectId
					+ ") is managed by Spring container.");
		}
	}

}
