/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2007 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.velocity.tools.generic;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.WordUtils;
import org.apache.oro.text.perl.Perl5Util;
import org.apache.oro.text.regex.Perl5Compiler;

/**
 * Velocity StringTool
 * 
 * This class's methods are very usrful string modifiers.<br/>
 * If the 1st parameter is not a String object,
 * each method gets String object by toString().<br/>
 * If the 1st parameter is null, each method returns 
 * an empty String ("") without any exception to throw.
 *
 * @author Yomei Komiya
 * @since 0.9.2 
 *
 * @version $Id: StringTool.java 6 2007-06-27 13:49:06Z whitestar $
 */
public class StringTool {

	/**
	 * Jakarta ORO Perl5Util object
	 */
	private Perl5Util perl5Util = null;
	
	
	public StringTool() {
		super();
	}
	
	
	public String capitalize(Object obj) {
		String str = this.defaultString(obj);
		return StringUtils.capitalize(str);
	}
	
	
	public String capitalizeAllWords(Object obj) {
		String str = this.defaultString(obj);
		return WordUtils.capitalize(str);
	}
	
	
	/**
	 * Returns either the passed in String, or if the String is empty or null,
	 * the value of defaultStr.
	 * 
	 * @param obj String or object
	 * @param defaultStr Default string
	 * @return Modified string
	 */
	public String defaultIfEmpty(Object obj, String defaultStr) {
		String str = this.defaultString(obj);
		return StringUtils.defaultIfEmpty(str, defaultStr);
	}
	
	
	/**
	 * Returns either the passed in String, or if the String is null,
	 * an empty String ("").
	 * 
	 * @param obj String or object
	 * @return Modified string
	 */
	public String defaultString(Object obj) {
		String str = null;
		if (obj != null) {
			str = obj.toString();
		}
		else {
			str = null;
		}
		
		return StringUtils.defaultString(str);
	}

	
	/**
	 * Returns either the passed in String, or if the String is null,
	 * the value of defaultStr.
	 * 
	 * @param obj String or object
	 * @param defaultStr Default string
	 * @return Modified string
	 */
	public String defaultString(Object obj, String defaultStr) {
		String str = null;
		if (obj != null) {
			str = obj.toString();
		}
		else {
			str = null;
		}
		
		return StringUtils.defaultString(str, defaultStr);
	}
	
	
	/**
	 * Alias of defaultString() 
	 * 
	 * @param obj String or object
	 * @param defaultStr Default string
	 * @return Modified string
	 */
	public String nvl(Object obj, String defaultStr) {
		return this.defaultString(obj, defaultStr);
	}
	
	
	public String toLowerCase(Object obj) {
		String str = this.defaultString(obj);
		return StringUtils.lowerCase(str);
	}
	
	
	public String toUpperCase(Object obj) {
		String str = this.defaultString(obj);
		return StringUtils.upperCase(str);
	}

	
	/**
	 * Replaces every new line "\n" with "<br/>" tag.
	 * 
	 * @param obj String or object
	 * @return Modified string
	 */
	public String nlToBrTag(Object obj) {
		String str = this.defaultString(obj);
		return StringUtils.replace(str, "\n", "<br/>");
	}
	
	
	public String replace(Object obj, String target, String replacement) {
		String text = this.defaultString(obj);
		return StringUtils.replace(text, target, replacement);
	}
	
	
	/**
	 * Escape meta character of Perl5 regular expressions 
	 * 
	 * @param obj String or object
	 * @return Quoted string
	 */
	public String regexQuoteMeta(Object obj) {
		String str = this.defaultString(obj);
		return Perl5Compiler.quotemeta(str);
	}
	
	
	/**
	 * Replaces strings by Jakarta ORO Perl5Util
	 * 
	 * @param obj String or object
	 * @param perl5Regex Perl5 regular expressions ("s/target/replacement/" ...)
	 * @return Modified string
	 */
	public String regexReplace(Object obj, String perl5Regex) {
		if (this.perl5Util == null) {
			this.perl5Util = new Perl5Util();
		}
		
		String str = this.defaultString(obj);
		return this.perl5Util.substitute(perl5Regex, str);
	}
	
	
	/**
	 * Removes control characters (char <= 32) from both ends of this String
	 * returning an empty String ("") if the String is empty ("") after the trim
	 * or if it is null.
	 * 
	 * @param obj String or object
	 * @return Modified string
	 */
	public String trim(Object obj) {
		String str = this.defaultString(obj);
		return StringUtils.trimToEmpty(str);
	}

	
	public String strip(Object obj) {
		String str = this.defaultString(obj);
		return StringUtils.stripToEmpty(str);
	}
	
	
	public String strip(Object obj, String stripChars) {
		String str = this.defaultString(obj);
		return StringUtils.strip(str, stripChars);
	}
	
	
	/**
	 * Returns truncated string
	 * Default last padding string is "...".
	 * 
	 * @param obj String or object
	 * @param len String length
	 * @return Truncated string
	 */
	public String truncate(Object obj, int len) {
		String str = this.defaultString(obj);
		return this.left(str, len);
	}
	
	
	/**
	 * Returns truncated string
	 * 
	 * @param obj String or object
	 * @param len String length
	 * @param lastPadStr Last padding string (default: "...")
	 * @return Truncated string
	 */
	public String truncate(Object obj, int len, String lastPadStr) {
		String defaultPadStr = "...";
		
		String str = this.defaultString(obj);
		if (lastPadStr == null || lastPadStr.equals("")) {
			lastPadStr = defaultPadStr;
		}
		
		String result = this.left(str, len);
		if (result.length() < str.length()) {
			return result + lastPadStr;
		}
		else {
			return result;
		}
	}
	
	
	/**
	 * Returns the 1st token string split by delimiter
	 * 
	 * @param obj String or object
	 * @param delim Delimiter
	 * @return 1st token
	 */
	public String firstToken(Object obj, String delim) {
		String str = this.defaultString(obj);
		String[] tokens = StringUtils.splitPreserveAllTokens(str, delim);
		
		if (tokens.length != 0) {
			return tokens[0];
		}
		else {
			return str;
		}
	}
	
	
	/**
	 * Returns the last token string split by delimiter
	 * 
	 * @param obj String or object
	 * @param delim Delimiter
	 * @return Last token
	 */
	public String lastToken(Object obj, String delim) {
		String str = this.defaultString(obj);
		String[] tokens = StringUtils.splitPreserveAllTokens(str, delim);
		
		if (tokens.length != 0) {
			return tokens[tokens.length -1];
		}
		else {
			return str;
		}
	}
	
	
	public String left(Object obj, int len) {
		String str = this.defaultString(obj);
		return StringUtils.left(str, len);
	}
	
	
	public String leftPad(Object obj, int size) {
		String str = this.defaultString(obj);
		return StringUtils.leftPad(str, size);
	}
	
	
	public String leftPad(Object obj, int size, String padStr) {
		String str = this.defaultString(obj);
		return StringUtils.leftPad(str, size, padStr);
	}
	
	
	public String right(Object obj, int len) {
		String str = this.defaultString(obj);
		return StringUtils.right(str, len);
	}
	
	
	public String rightPad(Object obj, int size) {
		String str = this.defaultString(obj);
		return StringUtils.rightPad(str, size);
	}
	
	
	public String rightPad(Object obj, int size, String padStr) {
		String str = this.defaultString(obj);
		return StringUtils.rightPad(str, size, padStr);
	}

	
	public String join(Object[] array) {
		if (array != null) {
			return StringUtils.join(array);
		}
		else {
			return "";
		}
	}

	
	public String join(Object[] array, String separator) {
		if (array != null) {
			return StringUtils.join(array, separator);
		}
		else {
			return "";
		}
	}

	
	public String wordwrap(Object obj, int wrapLength) {
		String str = this.defaultString(obj);
		return WordUtils.wrap(str, wrapLength);
	}
	
	
	public String wordwrap(Object obj, int wrapLength, String newLineStr) {
		String str = this.defaultString(obj);
		return WordUtils.wrap(str, wrapLength, newLineStr, false);
	}
	
}
