/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.web.filter;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.servlet.FilterChain;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.struts.Globals;
import org.easymock.EasyMock;

import jp.sourceforge.webframe.test.AbstractEasyMockTestCase;

/**
 * LocaleFilterTest
 *
 * @author Yomei Komiya
 *
 * @version $Id: LocaleFilterTest.java 8 2007-06-27 13:50:16Z whitestar $
 */
public class LocaleFilterTest extends AbstractEasyMockTestCase {

	private String langParameter = "lang";

	private String[] acceptableLanguages = new String[] {"en", "ja"};

	private String defaultLanguage = "en";
	
	private String defaultContentType = "text/html;charset=UTF-8";
	
	private HttpServletRequest requestMock = null;
	
	private HttpServletResponse responseMock = null;
	
	private HttpSession sessionMock = null;

	private FilterChain filterChainMock = null;

	private LocaleFilter localeFilter = null;
	
	
	protected void setUp() throws Exception {
		super.setUp();
		
		this.requestMock = (HttpServletRequest)this.mocksControl
			.createMock(HttpServletRequest.class);
		this.responseMock = (HttpServletResponse)this.mocksControl
			.createMock(HttpServletResponse.class);
		this.filterChainMock = (FilterChain)this.mocksControl
			.createMock(FilterChain.class);
		this.sessionMock = (HttpSession)this.mocksControl
			.createMock(HttpSession.class);
		
		this.localeFilter = new LocaleFilter();
		this.localeFilter.setLangParameter(this.langParameter);
		this.localeFilter.setAcceptableLanguages(this.acceptableLanguages);
		this.localeFilter.setDefaultLanguage(this.defaultLanguage);
		this.localeFilter.setDefaultContentType(this.defaultContentType);
	}

	
	protected void tearDown() throws Exception {
		super.tearDown();
	}

	
	/*
	 * 'jp.sourceforge.webframe.web.filter.LocaleFilter.init(FilterConfig)' test method
	 */
	public void testInit() {
		// nothing.
	}

	
	/*
	 * 'jp.sourceforge.webframe.web.filter.LocaleFilter.doFilter(ServletRequest, ServletResponse, FilterChain)' test method
	 */
	public void testDoFilter() throws Exception {
		/* 1.1 request parameter (acceptable language) ********************** */
		this.mocksControl.reset();

		EasyMock.expect(this.requestMock.getSession())
			.andReturn(this.sessionMock);
		
		// request parameter exists.
		Map parameterMapMock = new HashMap();
		parameterMapMock.put(this.langParameter, "ja");
		
		EasyMock.expect(this.requestMock.getParameterMap())
			.andReturn(parameterMapMock);
		EasyMock.expect(this.requestMock.getParameter(this.langParameter))
			.andReturn("ja");
		this.sessionMock.setAttribute(this.langParameter, "ja");

		this.responseMock.setLocale(new Locale("ja"));
		this.responseMock.setContentType(this.defaultContentType);
		this.sessionMock.setAttribute(Globals.LOCALE_KEY, new Locale("ja"));
		this.filterChainMock.doFilter(this.requestMock, this.responseMock);		
		
		this.mocksControl.replay();
		
		this.localeFilter.doFilter(
				this.requestMock, this.responseMock, this.filterChainMock);
		
		this.mocksControl.verify();
		
		/* 1.2 request parameter (unacceptable language) ******************** */
		this.mocksControl.reset();

		EasyMock.expect(this.requestMock.getSession())
			.andReturn(this.sessionMock);
		
		// request parameter exists.
		parameterMapMock = new HashMap();
		parameterMapMock.put(this.langParameter, "unknown");
		
		EasyMock.expect(this.requestMock.getParameterMap())
			.andReturn(parameterMapMock);
		EasyMock.expect(this.requestMock.getParameter(this.langParameter))
			.andReturn("unknown");
		this.sessionMock.setAttribute(
				this.langParameter, this.defaultLanguage);

		this.responseMock.setLocale(new Locale(this.defaultLanguage));
		this.responseMock.setContentType(this.defaultContentType);
		this.sessionMock.setAttribute(
				Globals.LOCALE_KEY, new Locale(this.defaultLanguage));
		this.filterChainMock.doFilter(this.requestMock, this.responseMock);		
		
		this.mocksControl.replay();
		
		this.localeFilter.doFilter(
				this.requestMock, this.responseMock, this.filterChainMock);
		
		this.mocksControl.verify();

		/* 2. session attribute ********************************************* */
		this.mocksControl.reset();
		EasyMock.expect(this.requestMock.getSession())
			.andReturn(this.sessionMock);

		// no request parameter
		parameterMapMock.clear();
		EasyMock.expect(this.requestMock.getParameterMap())
			.andReturn(parameterMapMock);
		// session attribute exists.
		EasyMock.expect(this.sessionMock.getAttribute(this.langParameter))
			.andReturn("ja").times(2);
		
		this.responseMock.setLocale(new Locale("ja"));
		this.responseMock.setContentType(this.defaultContentType);
		this.sessionMock.setAttribute(Globals.LOCALE_KEY, new Locale("ja"));
		this.filterChainMock.doFilter(this.requestMock, this.responseMock);		
	
		this.mocksControl.replay();
	
		this.localeFilter.doFilter(
				this.requestMock, this.responseMock, this.filterChainMock);
	
		this.mocksControl.verify();
		
		/* 3.1 request header (acceptable language) ************************* */
		this.mocksControl.reset();
		EasyMock.expect(this.requestMock.getSession())
			.andReturn(this.sessionMock);

		// no request parameter
		parameterMapMock.clear();
		EasyMock.expect(this.requestMock.getParameterMap())
			.andReturn(parameterMapMock);
		// no session attribute
		EasyMock.expect(this.sessionMock.getAttribute(this.langParameter))
			.andReturn(null);
		// header locale exists.
		Locale localeMock = new Locale("ja", "JP");
		EasyMock.expect(this.requestMock.getLocale()).andReturn(localeMock);
		
		this.responseMock.setLocale(new Locale("ja"));
		this.responseMock.setContentType(this.defaultContentType);
		this.sessionMock.setAttribute(Globals.LOCALE_KEY, new Locale("ja"));
	
		this.filterChainMock.doFilter(this.requestMock, this.responseMock);		
	
		this.mocksControl.replay();
	
		this.localeFilter.doFilter(
				this.requestMock, this.responseMock, this.filterChainMock);
	
		this.mocksControl.verify();

		/* 3.2 request header (unacceptable language) *********************** */
		this.mocksControl.reset();
		EasyMock.expect(this.requestMock.getSession())
			.andReturn(this.sessionMock);

		// no request parameter
		parameterMapMock.clear();
		EasyMock.expect(this.requestMock.getParameterMap())
			.andReturn(parameterMapMock);
		// no session attribute
		EasyMock.expect(this.sessionMock.getAttribute(this.langParameter))
			.andReturn(null);
		// header locale exists. Esperanto!
		localeMock = new Locale("eo", "JP");
		EasyMock.expect(this.requestMock.getLocale()).andReturn(localeMock);
		
		this.responseMock.setLocale(new Locale(this.defaultLanguage));
		this.responseMock.setContentType(this.defaultContentType);
		this.sessionMock.setAttribute(
				Globals.LOCALE_KEY, new Locale(this.defaultLanguage));
	
		this.filterChainMock.doFilter(this.requestMock, this.responseMock);		
	
		this.mocksControl.replay();
	
		this.localeFilter.doFilter(
				this.requestMock, this.responseMock, this.filterChainMock);
	
		this.mocksControl.verify();

		/* 4. default ******************************************************* */
		this.mocksControl.reset();
		EasyMock.expect(this.requestMock.getSession())
			.andReturn(this.sessionMock);

		// no request parameter
		parameterMapMock.clear();
		EasyMock.expect(this.requestMock.getParameterMap())
			.andReturn(parameterMapMock);
		// no session attribute
		EasyMock.expect(this.sessionMock.getAttribute(this.langParameter))
			.andReturn(null);
		// no header locale
		localeMock = null;
		EasyMock.expect(this.requestMock.getLocale()).andReturn(localeMock);
		
		this.responseMock.setLocale(new Locale(this.defaultLanguage));
		this.responseMock.setContentType(this.defaultContentType);
		this.sessionMock.setAttribute(
				Globals.LOCALE_KEY, new Locale(this.defaultLanguage));
		this.filterChainMock.doFilter(this.requestMock, this.responseMock);		
	
		this.mocksControl.replay();
	
		this.localeFilter.doFilter(
				this.requestMock, this.responseMock, this.filterChainMock);
	
		this.mocksControl.verify();
	}

	
	/*
	 * 'jp.sourceforge.webframe.web.filter.LocaleFilter.destroy()' test method
	 */
	public void testDestroy() {
		// nothing
	}

}
