/*  ESmol - Molecular Viewer for Android

     (C) Copyright 2011, biochem_fan

     This file is part of ESmol.

     ESmol is free software: you can redistribute it and/or modify
     it under the terms of the GNU Lesser General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU Lesser General Public License for more details.

     You should have received a copy of the GNU Lesser General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>. */

package jp.sfjp.webglmol.ESmol;

import java.util.ArrayList;
import java.util.HashMap;

import javax.microedition.khronos.egl.EGLConfig;
import javax.microedition.khronos.opengles.GL10;
import javax.microedition.khronos.opengles.GL11;


import android.opengl.GLSurfaceView;
import android.opengl.GLU;
import android.util.Log;

class GLView implements GLSurfaceView.Renderer {
	Renderable scene;
	
	public float objX, objY, objZ;
	public float cameraZ;
	public Quaternion rotationQ;
	private Atom [] atoms;
	private Protein protein;
	public int width, height;
	
	public int proteinMode = 0; // 0 - ribbon, 1 - trace
	public int hetatmMode = 2; // 0 - sphere, 1 - stick, 2 - line
	public boolean showSidechain = false;
	public boolean showUnitcell = false;
	public int symmetryMode = 0; // 0 - none, 1 - biomt, 2 - packing
	public int colorMode = 0; // 0 - chainbow, 1 - chain, 2 - ss
	
	public float sphereRadius = 1.5f; 
	public float cylinderRadius = 0.2f;
	public float lineWidth = 0.5f;
	public float curveWidth = 3.0f;
	
	public boolean isMoving = false;

	
	public GLView() {
		objX = 0; objY = 0; objZ = 0;
		cameraZ = -300;
		rotationQ = new Quaternion(1, 0, 0, 0);
		
		scene = new Renderable();
	}
	
	public void setProtein(Protein p) {
		protein = p;
		atoms = p.atoms;
//		objX = 0; objY = 0; objZ = 0;
		cameraZ = -300;
		objX = -protein.centerx;// * 0.5f;
		objY = -protein.centery;// * 0.5f;
		objZ = -protein.centerz;// * 0.5f;
	}

	public void drawAxis(float len, float radius) {
		Renderable cylinder1 = new Cylinder(0, 0, 0, len, 0, 0, radius, new Color(1, 0, 0, 1)); // red, X
		Renderable cylinder2 = new Cylinder(0, 0, 0, 0, len, 0, radius, new Color(0, 1, 0, 1)); // green, Y
		Renderable cylinder3 = new Cylinder(0, 0, 0, 0, 0, len, radius, new Color(0, 0, 1, 1)); // blue, Z
		scene.children.add(cylinder1);
		scene.children.add(cylinder2);
		scene.children.add(cylinder3);
		Renderable sphere = new VBOSphere(0, 0, 0, radius, new Color(1, 1, 1, 1));
		scene.children.add(sphere);
	} 
	
	public void prepareScene() {
		scene = new Renderable();

		
//		drawAxis(10, 5);
		
//		Renderable test = new MeshTest(); 
//		scene.children.add(test);
		
		if (protein == null) return;

		ArrayList<Integer> all = getAll();
		ArrayList<Integer> hetatm = removeSolvents(getHetatms(all));
		colorByAtom(all, null);
		
		switch (colorMode) {
		case 0:
			colorChainbow(all);
			break;
		case 1:
			colorByChain(all);
			break;
		case 2:
			colorByStructure(all, new Color(0xCC00CC), new Color(0x00CCCC));
		}

		switch (proteinMode) {
		case 0:
			drawCartoon(scene, all);
			break;
		case 1:
			drawMainchainCurve(scene, all, curveWidth, "CA");
		}
		drawMainchainCurve(scene, all, curveWidth, "P");
		
		if (showSidechain) {
			drawBondsAsLine(scene, getSideChain(all), lineWidth);
		}

		switch (hetatmMode) {
		case 0:
			drawAtomsAsVdWSphere(scene, hetatm);
			break;
		case 1:
			drawBondsAsStick(scene, hetatm, cylinderRadius, cylinderRadius);
			break;
		case 2:
			drawBondsAsLine(scene, hetatm, lineWidth * 8);
		}
		
		if (showUnitcell) {
			drawUnitcell(scene, lineWidth);
		}
		
		switch (symmetryMode) {
		case 1:
			drawSymmetryMates(scene, all, protein.biomtMatrices);
			break;
		case 2:
			drawSymmetryMatesWithTranslation(scene, all, protein.symmetryMatrices);
		}
	}
	
	public ArrayList<Integer> getAll() {
		ArrayList<Integer> ret = new ArrayList<Integer>();
		for (int i = 0; i < 100001; i++) {
			if (atoms[i] != null) ret.add(i);
		}
		return ret;   
	}

	public ArrayList<Integer> getHetatms(ArrayList<Integer> atomlist) {
		ArrayList<Integer> ret = new ArrayList<Integer>();
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;
			
			if (atom.hetflag)	ret.add(atom.serial);
		}
		return ret;   
	}
	
	public ArrayList<Integer> removeSolvents(ArrayList<Integer> atomlist) {
		ArrayList<Integer> ret = new ArrayList<Integer>();
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;
			
			if (!atom.resn.equals("HOH")) ret.add(atom.serial);
		}
		return ret;   
	}
	
	public ArrayList<Integer> getResiduesById(ArrayList<Integer> atomlist, ArrayList<Integer> resi) {
		ArrayList<Integer> ret = new ArrayList<Integer>();
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;
			
			if (resi.indexOf(atom.resi) != -1) ret.add(atom.serial);
		}
		return ret;   
	}
	
	public ArrayList<Integer> getSideChain(ArrayList<Integer> atomlist) {
		ArrayList<Integer> ret = new ArrayList<Integer>();
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;
			
			if (atom.hetflag) continue;
			if (atom.atom.equals("O") || atom.atom.equals("N") || atom.atom.equals("C")) continue;
			ret.add(atom.serial);
		}
		
		return ret;   
	}

	
	public ArrayList<Integer> getChain(ArrayList<Integer> atomlist, ArrayList<String> chain) {
		ArrayList<Integer> ret = new ArrayList<Integer>();
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;
			
			if (chain.indexOf(atom.chain) != -1) ret.add(atom.serial);
		}
		return ret;   
	}

	public void colorByAtom(ArrayList<Integer> atomlist, HashMap<String, Color> colors) {
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;
			
			Color c = null;
			if (colors != null ) c= colors.get(atom.elem);
			if (c == null) c = ChemDatabase.getColor(atom.elem);
			atom.color = c;
		}
	}
	
	public void colorByStructure(ArrayList<Integer> atomlist, Color helixColor, Color sheetColor) {
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;

			if (!atom.atom.equals("CA") || atom.hetflag) continue;
			if (atom.ss.charAt(0) == 's') atom.color = sheetColor;
			else if (atom.ss.charAt(0) == 'h') atom.color = helixColor;
		}
	}
	
	public void colorByChain(ArrayList<Integer> atomlist) {
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;

			if (!atom.atom.equals("CA") || atom.hetflag) continue;
			atom.color = new Color().setHSV((float)(atom.chain.charAt(0) % 15) / 15, 1, 0.9f);
		}
	}
	
	public void colorChainbow(ArrayList<Integer> atomlist) {
		int cnt = 0;
		
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;

			if (!atom.atom.equals("CA") || atom.hetflag) continue;
			cnt++;
		}

		int total = cnt;
		cnt = 0;

		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;

			if (!atom.atom.equals("CA") || atom.hetflag) continue;
			atom.color = new Color().setHSV((float)240 / 360 * cnt / total, 1, 0.9f);
			cnt++;
		}
	}
	
	public void drawMainchainCurve(Renderable scene, ArrayList<Integer> atomlist, float curveWidth, String atomName) {
		ArrayList<Vector3> points = new ArrayList<Vector3>();
		ArrayList<Color> colors = new ArrayList<Color>();
		
		String currentChain = "A";
		int currentResi = -1;
		
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;

			if ((atom.atom.equals(atomName)) && !atom.hetflag) {
				if (!currentChain.equals(atom.chain) || currentResi + 1 != atom.resi) {
					scene.children.add(new SmoothCurve(points, colors, curveWidth));
					points = new ArrayList<Vector3>();
					colors  = new ArrayList<Color>();
				}
				points.add(new Vector3(atom.x, atom.y, atom.z));
				colors.add(atom.color);
				currentChain = atom.chain;
				currentResi = atom.resi;
			}
		}
		scene.children.add(new SmoothCurve(points, colors, curveWidth));

		// TODO: Arrow heads
	}

	public void drawCartoon(Renderable scene, ArrayList<Integer> atomlist) {
		ArrayList<Vector3> points1 = new ArrayList<Vector3>();
		ArrayList<Vector3> points2 = new ArrayList<Vector3>();
		ArrayList<Color> colors = new ArrayList<Color>();

		String currentChain = "A";
		int currentResi = -1;
		// Just to satisfy our poor compiler...
		Vector3 prevCO = null, currentCA =null;

		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;

			if ((atom.atom.equals("O") || atom.atom.equals("CA")) && !atom.hetflag) {
				if (atom.atom.equals("CA")) {
					if (!currentChain.equals(atom.chain) || currentResi + 1 != atom.resi) {
						scene.children.add(new RibbonStrip(points1, points2, colors));
						//colors.unshift(0); colors.pop(); // FIXME: ad hoc.  
						scene.children.add(new SmoothCurve(points1, colors, curveWidth));
						scene.children.add(new SmoothCurve(points2, colors, curveWidth));
						points1 = new ArrayList<Vector3>(); points2 = new ArrayList<Vector3>();
						colors = new ArrayList<Color>();
						prevCO = null;
					}
					currentCA = new Vector3(atom.x, atom.y, atom.z);
					currentChain = atom.chain;
					currentResi = atom.resi;
					colors.add(atom.color);
				} else { // O
					float ox = atom.x - currentCA.x, oy = atom.y - currentCA.y, oz = atom.z - currentCA.z;
					float width = (atom.ss.charAt(0) == 'c') ? 0.1f : 0.5f;
					ox *= width; oy *= width; oz *= width; 
					if (prevCO != null && (ox * prevCO.x + oy * prevCO.y + oz * prevCO.z) < 0) {
						ox *= -1; oy *= -1; oz *= -1;
					}
					prevCO = new Vector3(ox, oy, oz);
					points1.add(new Vector3(currentCA.x + prevCO.x, currentCA.y + prevCO.y, currentCA.z + prevCO.z));
					points2.add(new Vector3(currentCA.x - prevCO.x, currentCA.y - prevCO.y, currentCA.z - prevCO.z));
				}
			}
		}
		scene.children.add(new RibbonStrip(points1, points2, colors));//		colors.unshift(0); colors.pop(); // FIXME! ad hoc.
		scene.children.add(new SmoothCurve(points1, colors, curveWidth));
		scene.children.add(new SmoothCurve(points2, colors, curveWidth));
	}

	public boolean isIdentity(float[] mat) {
		for (int i = 0; i < 4; i++) {
			for (int j = 0; j < 4; j++) {
				if (i == j && Math.abs(mat[i * 4 + j] - 1) > 0.001) return false;
				if (i != j && Math.abs(mat[i * 4 + j]) > 0.001) return false;
			}
		}
		return true;
	}


	public void drawBondsAsStick(Renderable scene, ArrayList<Integer> atomlist, float bondR, float atomR) {
		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			Atom atom1 = atoms[atomlist.get(i)];
			if (atom1 == null) continue;
			for (int j = i + 1; j < i + 40 && j < lim; j++) { // TODO: Check if 40 is sufficient for DNA and HETATMs., port to GLmol
				Atom atom2 = atoms[atomlist.get(j)];
				if (atom2 == null) continue;
				if (!atom1.isConnected(atom2)) continue;

				Renderable cylinder = new VBOCylinder(atom1.x, atom1.y, atom1.z,
						(atom1.x + atom2.x) / 2, (atom1.y + atom2.y) / 2, (atom1.z + atom2.z) / 2, bondR, atom1.color);
				scene.children.add(cylinder);
				
				cylinder = new VBOCylinder((atom1.x + atom2.x) / 2, (atom1.y + atom2.y) / 2, (atom1.z + atom2.z) / 2,
						atom2.x, atom2.y, atom2.z, bondR, atom2.color);
				scene.children.add(cylinder);
			}
			Renderable sphere = new VBOSphere(atom1.x, atom1.y, atom1.z, atomR, atom1.color);
			scene.children.add(sphere);
		}
	}

	// ????????????ArrayList<Integer> ????????��??��?????��???????????????
	public void drawBondsAsLine(Renderable scene, ArrayList<Integer> atomlist, float lineWidth) {
		ArrayList<Vector3> points = new ArrayList<Vector3>();
		ArrayList<Color> colors = new ArrayList<Color>();

		for (int i = 0, lim = atomlist.size(); i < lim; i++) {
			for (int j = i + 1; j < i + 40 && j < lim; j++) { // TODO: Check if 40 is sufficient for DNA and HETATMs., port to GLmol
				Atom atom1 = atoms[atomlist.get(i)], atom2 = atoms[atomlist.get(j)];
				if (atom1 == null || atom2 == null) continue;
				if (!atom1.isConnected(atom2)) continue;

				Vector3 mid = new Vector3((atom1.x + atom2.x) / 2, (atom1.y + atom2.y) / 2, (atom1.z + atom2.z) / 2);
				points.add(mid);
				points.add(new Vector3(atom1.x, atom1.y, atom1.z));
				colors.add(atom1.color); colors.add(atom1.color);

				points.add(mid);
				points.add(new Vector3(atom2.x, atom2.y, atom2.z));
				colors.add(atom2.color); colors.add(atom2.color);
			}
		}

		Line line = new Line(points, colors);
		line.width = lineWidth;
		line.discrete = true;
		scene.children.add(line);
	}
	
	public void drawAtomsAsVdWSphere(Renderable scene, ArrayList<Integer> atomlist) {
		ArrayList<Vector3> points = new ArrayList<Vector3>();
		ArrayList<Color> colors = new ArrayList<Color>();
		ArrayList<Float> radii = new ArrayList<Float>();
		for (int i = 1, lim = atomlist.size(); i < lim; i++) {
			Atom atom = atoms[atomlist.get(i)];
			if (atom == null) continue;

//			Renderable sphere = new VBOSphere(atom.x, atom.y, atom.z, ChemDatabase.getVdwRadius(atom.elem), atom.color);
//			scene.children.add(sphere);
			points.add(new Vector3(atom.x, atom.y, atom.z));
			colors.add(atom.color);
			radii.add(ChemDatabase.getVdwRadius(atom.elem));
		}
		Renderable sphere = new VBOSpheres(points, colors, radii);
		scene.children.add(sphere);
	}
	public void drawSymmetryMates(Renderable scene, ArrayList<Integer> atomlist, HashMap<Integer, float[]> biomtMatrices) {
		if (biomtMatrices == null) return;
 
		for (float mat[]: biomtMatrices.values()) {
			if (mat == null || isIdentity(mat)) continue;

//			Log.d("drawSymmetryMates", "{(" + mat[0] + ", " + mat[1] + ", " +  mat[2] + ", " +  mat[3] + ")\n(" +
//					 mat[4] + ", " + mat[5] + ", " + mat[6] + ", " + mat[7] + ")\n(" +
//					 mat[8] + ", " + mat[9] + ", " + mat[10] + ", " + mat[11] + ")\n(" + 
//					 mat[12] + ", " + mat[13] + ", " + mat[14] + ", " + mat[15] + ")}");
			MatRenderable symmetryMate = new MatRenderable();
			drawMainchainCurve(symmetryMate, atomlist, curveWidth, "CA");
			drawMainchainCurve(symmetryMate, atomlist, curveWidth, "P");
			symmetryMate.setupMatrix(mat);
			scene.children.add(symmetryMate);
		}
	}

	public void drawSymmetryMatesWithTranslation(Renderable scene, ArrayList<Integer> atomlist, HashMap<Integer, float[]> matrices) {
		if (matrices == null) return;

		for (float mat[]: matrices.values()) {
			if (mat == null) continue;

			for (int a = -1; a <= 0; a++) {
				for (int b = -1; b <= 0; b++) {
					for (int c = -1; c <= 0; c++) {
						MatRenderable symmetryMate = new MatRenderable();
						float[] translated = mat.clone();
						translated[3] += protein.ax * a + protein.bx * b + protein.cx * c;
						translated[7] += protein.ay * a + protein.by * b + protein.cy * c;
						translated[11] += protein.az * a + protein.bz * b + protein.cz * c;
						if (isIdentity(translated)) continue;
						symmetryMate.setupMatrix(translated);
						
						drawMainchainCurve(symmetryMate, atomlist, curveWidth, "CA");
						drawMainchainCurve(symmetryMate, atomlist, curveWidth, "P");
						
						scene.children.add(symmetryMate);
					}
				}
			}
		}
	}

	public void drawUnitcell(Renderable scene, float width) {
		if (protein.a == 0) return;

		float vertices[][] = {{0, 0, 0}, 
				{protein.ax, protein.ay, protein.az}, 
				{protein.bx, protein.by, protein.bz},
				{protein.ax + protein.bx, protein.ay + protein.by, protein.az + protein.bz},
				{protein.cx, protein.cy, protein.cz},
				{protein.cx + protein.ax, protein.cy + protein.ay,  protein.cz + protein.az}, 
				{protein.cx + protein.bx, protein.cy + protein.by, protein.cz + protein.bz}, 
				{protein.cx + protein.ax + protein.bx, protein.cy + protein.ay + protein.by, protein.cz + protein.az + protein.bz}};
		int edges[] = {0, 1, 0, 2, 1, 3, 2, 3, 4, 5, 4, 6, 5, 7, 6, 7, 0, 4, 1, 5, 2, 6, 3, 7};    

		float points[] = new float[24 * 3];
		for (int i = 0; i < edges.length; i++) {
			points[i * 3] = vertices[edges[i]][0];
			points[i * 3 + 1] = vertices[edges[i]][1];
			points[i * 3 + 2] = vertices[edges[i]][2];
		}
		Line line = new Line(points);
		line.objectColor = new Color(0.8f, 0.8f, 0.8f, 1);
		line.discrete = true;
		line.width = width;
		scene.children.add(line);
	}

	public void onDrawFrame(GL10 gl){
		Log.d("test", "OnDrawFrame");
		gl.glClear(GL10.GL_COLOR_BUFFER_BIT|GL10.GL_DEPTH_BUFFER_BIT);

		gl.glMatrixMode(GL10.GL_MODELVIEW);
		gl.glLoadIdentity();
		gl.glTranslatef(0, 0, cameraZ);
		
		Vector3 axis = rotationQ.getAxis();
		gl.glRotatef(180 * rotationQ.getAngle() / (float)Math.PI, axis.x, axis.y, axis.z);
		
		gl.glTranslatef(objX, objY, objZ);
		
		if (scene != null)
			scene.render(gl, this);
	}

	public void onSurfaceChanged(GL10 gl, int width, int height) {
		Log.d("test", "OnSurfaceChanged");
		
		this.width = width; this.height = height;
		
		gl.glViewport(0, 0, width, height);
		gl.glMatrixMode(GL10.GL_PROJECTION);
		gl.glLoadIdentity();    
		GLU.gluPerspective(gl, 20f,(float) width / height, 1f, 800f); // FIXME: Z-fighting?

		VBOCylinder.prepareVBO((GL11)gl);
		VBOSphere.prepareVBO((GL11)gl, this);
	}

	public void onSurfaceCreated(GL10 gl, EGLConfig config) {
		gl.glClearColor(0, 0, 0, 1);
		gl.glEnable(GL10.GL_DEPTH_TEST);
//		gl.glEnable(GL10.GL_NORMALIZE);
		gl.glShadeModel(GL10.GL_SMOOTH);
//		gl.glHint(GL10.GL_POINT_SMOOTH_HINT, GL10.GL_NICEST);
		gl.glEnable(GL10.GL_POINT_SMOOTH);
		gl.glEnable(GL10.GL_BLEND);
		gl.glBlendFunc(GL10.GL_SRC_ALPHA, GL10.GL_ONE_MINUS_SRC_ALPHA);
		//		gl.glEnable(GL10.GL_LINE_SMOOTH); // FIXME: Check if this is working.
		gl.glLightModelx(GL10.GL_LIGHT_MODEL_TWO_SIDE, 1); // double sided
		gl.glEnable(GL10.GL_COLOR_MATERIAL); // glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE) ?
		gl.glDepthFunc(GL10.GL_LEQUAL);
		gl.glDisable(GL10.GL_DITHER);
		gl.glEnable(GL10.GL_LIGHTING);
		gl.glEnable(GL10.GL_LIGHT0);
		gl.glLightfv(GL10.GL_LIGHT0, GL10.GL_AMBIENT, Geometry.getFloatBuffer(new float[] {0.8f, 0.8f, 0.8f, 1}));
		gl.glLightfv(GL10.GL_LIGHT0, GL10.GL_POSITION, Geometry.getFloatBuffer(new float[] {0, 0, 1, 0}));
		gl.glLightfv(GL10.GL_LIGHT0, GL10.GL_DIFFUSE, Geometry.getFloatBuffer(new float[] {0.4f, 0.4f, 0.4f, 1}));
		gl.glLightfv(GL10.GL_LIGHT0, GL10.GL_SPECULAR, Geometry.getFloatBuffer(new float[] {0.4f, 0.4f, 0.4f, 1}));
		gl.glEnable(GL10.GL_LIGHT1);
//		gl.glLightfv(GL10.GL_LIGHT0, GL10.GL_AMBIENT, Geometry.getFloatBuffer(new float[] {0.4f, 0.4f, 0.4f, 1}));
		gl.glLightfv(GL10.GL_LIGHT1, GL10.GL_POSITION, Geometry.getFloatBuffer(new float[] {1, 1, 1, 0}));
		gl.glLightfv(GL10.GL_LIGHT1, GL10.GL_DIFFUSE, Geometry.getFloatBuffer(new float[] {0.1f, 0.1f, 0.1f, 1}));
		gl.glLightfv(GL10.GL_LIGHT1, GL10.GL_SPECULAR, Geometry.getFloatBuffer(new float[] {0.1f, 0.1f, 0.1f, 1}));
//		gl.glEnable(GL10.GL_LIGHT1);
//		gl.glLightfv(GL10.GL_LIGHT1, GL10.GL_POSITION, Geometry.getFloatBuffer(new float[] {1, 1, 0, 0}));
//		gl.glLightfv(GL10.GL_LIGHT1, GL10.GL_SPECULAR, Geometry.getFloatBuffer(new float[] {1f, 1f, 1f, 1}));
//		gl.glEnable(GL10.GL_LIGHT1);
//		gl.glLightfv(GL10.GL_LIGHT1, GL10.GL_POSITION, FloatBuffer.wrap(new float[] {-0.6f, -0.8f, 0.2f, 0}));
		((GL11)gl).glPointParameterfv(GL11.GL_POINT_DISTANCE_ATTENUATION, Geometry.getFloatBuffer(new float[] {0, 0, 1}));
	}
}