/*  NDKmol - Molecular Viewer on Android NDK

     (C) Copyright 2011 - 2012, biochem_fan

     This file is part of NDKmol.

     NDKmol is free software: you can redistribute it and/or modify
     it under the terms of the GNU Lesser General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU Lesser General Public License for more details.

     You should have received a copy of the GNU Lesser General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#include "RibbonStrip.hpp"
#include "Geometry.hpp"
#include <cmath>
#include <android/log.h>

RibbonStrip::RibbonStrip(std::vector<Vector3> &_points1, std::vector<Vector3> &_points2, std::vector<Color> &colors): Renderable() {
	if (_points1.size() < 1) return; // 줬뤫顢nFaces ν˺ʤ!
	vertexColors = true;

	float *points1 = subdivide(_points1, div);
	float *points2 = subdivide(_points2, div);

	int size = ((_points1.size() - 1) * div + 1) * 3; // number of elements.
	__android_log_print(ANDROID_LOG_DEBUG,"RibbonStrip","nPoints = %d, size = %d", _points1.size(), size);

	vertexBuffer = new float[size * 2];
	vertexNormalBuffer = new float[size * 2];
	faceBuffer = new unsigned short[size * 2];
	nFaces = size * 2 - 6;

	int vertexOffset = 0, faceOffset = 2;
	float *vb = vertexBuffer;
	unsigned short *fb = faceBuffer;

	for (int i = 0, lim = size / 3; i < lim; i++) {
		*(vb++) = points1[vertexOffset]; // face 2 * i
		*(vb++) = points1[vertexOffset + 1];
		*(vb++) = points1[vertexOffset + 2];
		*(vb++) = points2[vertexOffset]; // face 2 * i + 1
		*(vb++) = points2[vertexOffset + 1];
		*(vb++) = points2[vertexOffset + 2];
		vertexOffset += 3;

		if (i == 0) continue;
		*(fb++) = (short)(faceOffset - 2);
		*(fb++) = (short)(faceOffset - 1);
		*(fb++) = (short)faceOffset;
		*(fb++) = (short)faceOffset;
		*(fb++) = (short)(faceOffset - 1);
		*(fb++) = (short)(faceOffset + 1);
		faceOffset += 2;
	}

	float *vnb = vertexNormalBuffer;
	for (int i = 0, lim = size / 3; i < lim; i++) {
		vertexOffset = 3 * i;
		float ax = 0, ay = 0, az = 0; // for ribbon direction, take average
		if (i > 0) {
			ax += points1[vertexOffset] - points1[vertexOffset - 3];
			ay += points1[vertexOffset + 1] - points1[vertexOffset - 2];
			az += points1[vertexOffset + 2] - points1[vertexOffset - 1];
		}
		if (i < lim - 1) {
			ax += points1[vertexOffset + 3] - points1[vertexOffset];
			ay += points1[vertexOffset + 4] - points1[vertexOffset + 1];
			az += points1[vertexOffset + 5] - points1[vertexOffset + 2];
		}
		float bx = points2[vertexOffset] - points1[vertexOffset];
		float by = points2[vertexOffset + 1] - points1[vertexOffset + 1];
		float bz = points2[vertexOffset + 2] - points1[vertexOffset + 2];

		float nx = ay * bz - az * by;
		float ny = az * bx - ax * bz;
		float nz = ax * by - ay * bx;

		float norm = (float)std::sqrt(nx * nx + ny * ny + nz * nz);
		nx /= norm; ny /= norm; nz /= norm;

		*(vnb++) = nx;
		*(vnb++) = ny;
		*(vnb++) = nz;
		*(vnb++) = nx;
		*(vnb++) = ny;
		*(vnb++) = nz;
	}

	colorBuffer = colorVectorToFloatArray(colors, div * 2);

	delete points1;
	delete points2;
}

