/*
 * $Id: WildcardTest.java,v 1.3 2004/04/01 05:48:15 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests;

import java.util.*;
import java.util.regex.Pattern;

import junit.framework.TestCase;

import com.narucy.webpub.core.Wildcard;

/**
 * 
 */
public class WildcardTest extends TestCase {

	static String[] sampleFilelist = {
		"001.java", "002.java", "003.java",
		"001.html", "002.html", "003.html",
		"101.java", "102.java", "103.java",

		"a/", "a/x/", "a/y/",
		"a/001.java", "a/002.java", "a/003.java",
		"a/001.html", "a/002.html", "a/003.html",
		"a/101.java", "a/102.java", "a/103.java",

		"a/b/", "a/b/x/", "a/b/y/",
		"a/b/001.java", "a/b/002.java", "a/b/003.java",
		"a/b/001.html", "a/b/002.html", "a/b/003.html",
		"a/b/101.java", "a/b/102.java", "a/b/103.java",

		"a/b/c/", "a/b/c/x/", "a/b/c/y/",
		"a/b/c/001.java", "a/b/c/002.java", "a/b/c/003.java",
		"a/b/c/001.html", "a/b/c/002.html", "a/b/c/003.html",
		"a/b/c/101.java", "a/b/c/102.java", "a/b/c/103.java",
	};

	public WildcardTest(String name) {
		super(name);
	}
	
	public void testWildcard(){
		String[] ms = matches(new Wildcard("*/*/*/001.java"));
		assertTrue( Arrays.equals(new String[]{"a/b/c/001.java"}, ms) );
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}
	
	public void testMatch(){
		Wildcard pattern = new Wildcard(".publish");
		assertTrue( pattern.match(".publish") );

		pattern = new Wildcard("*.tmp");
		assertTrue( pattern.match("test.tmp") );
	}
	
	public void testDirMatch(){
		Wildcard pattern = new Wildcard("**/y/");
		String[] ms = matches(pattern);
		assertEquals( "a/y/", ms[0]);
		assertEquals( "a/b/y/", ms[1]);
		assertEquals( "a/b/c/y/", ms[2]);
		assertEquals( 3, ms.length );

		pattern = new Wildcard("a/**/x/");
		ms = matches(pattern);
		assertEquals( "a/b/x/", ms[0]);
		assertEquals( "a/b/c/x/", ms[1]);
		assertEquals( 2, ms.length );
	}
	
	public void testRandom(){
		Wildcard pattern = new Wildcard("**/b/10?.*");
		String[] ms = matches(pattern);
		for (int i = 0; i < ms.length; i++) {
//			p(ms[i]);
			assertTrue( Pattern.matches( "a/b/10\\d\\.java$", ms[i]) );
		}
		assertEquals( 3, ms.length );
	}
	
	public void testSearchAll(){
		Wildcard pattern = new Wildcard("**/*.java");
		String[] ms = matches(pattern);
		for (int i = 0; i < ms.length; i++) {
			assertTrue( Pattern.matches( ".+\\.java$", ms[i]) );
		}
		assertEquals(18, ms.length );
		
		pattern = new Wildcard("a/**/*.java");
		ms = matches(pattern);
		assertEquals(12, ms.length);
		
		pattern = new Wildcard("**/c/*.html");
		ms = matches(pattern);
		for (int i = 0; i < ms.length; i++) {
			assertTrue( Pattern.matches( "a/b/c/00\\d.html$", ms[i]) );
		}
		assertEquals(3, ms.length);
	}

	public void testWildcardCompilerWizDir(){
		Wildcard pattern = new Wildcard("a/*.java");
		String[] ms = matches(pattern);
		assertEquals( 6, ms.length );
		assertEquals("a/001.java", ms[0]);
		assertEquals("a/103.java", ms[5]);

		pattern = new Wildcard("a/b/*.java");
		ms = matches(pattern);
		assertEquals( 6, ms.length );
		assertEquals("a/b/001.java", ms[0]);
		assertEquals("a/b/103.java", ms[5]);
	}
	
	public void testWildcardCompiler(){
		Wildcard pattern = new Wildcard("*.java");
		assertEquals( 6, matches(pattern).length );
		
		pattern = new Wildcard("*.html");
		assertEquals( 3, matches(pattern).length );

		pattern = new Wildcard("*.html");
		assertEquals( 3, matches(pattern).length );

		pattern = new Wildcard("10?.java");
		String[] ms = matches(pattern);
		assertEquals( "101.java", ms[0]);
		assertEquals( "102.java", ms[1]);
		assertEquals( "103.java", ms[2]);
		assertEquals( 3, ms.length );
		
		pattern = new Wildcard("?0?.java");
		ms = matches(pattern);
		assertEquals( 6, ms.length );
		
		pattern = new Wildcard("?02.*");
		ms = matches(pattern);
		assertEquals( "002.java", ms[0]);
		assertEquals( "002.html", ms[1]);
		assertEquals( "102.java", ms[2]);
		
		assertEquals( 3, ms.length );
	}

	static String[] matches(Wildcard pattern){
		ArrayList list = new ArrayList();
		for (int i = 0; i < sampleFilelist.length; i++) {
			String f = sampleFilelist[i];
			if( pattern.match(f) ){
				list.add(f);
			}
		}
		return (String[])list.toArray(new String[list.size()]);
	}

}
