/*
 * $Id: PublisherBuilderTest.java,v 1.8 2004/04/20 06:57:04 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests.publish;

import java.io.File;

import junit.framework.TestCase;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.Path;

import com.narucy.webpub.core.*;
import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.core.tests.WebpubTestPlugin;

/**
 * 
 */
public class PublisherBuilderTest extends TestCase {

	public PublisherBuilderTest(String name) {
		super(name);
	}
	
	public void testErrorOutput() throws Exception {
		File tmpFile = WebpubTestPlugin.createTempFile("PublisherBuilderTest.testStdErrOutput");
		try{
			String arg = "-e 'def x; @x.x; end; $stderr = open(%Q!" + tmpFile.getAbsolutePath().replaceAll("\\\\", "/") + "!, \"w\") ;for i in 0...6000; $stdout << %Q!It is STDOUT #{i}\\n!;end; x()";
			
			Process proc = WebpubPlugin.getDefault().rubyExec(arg, null);
			
			TextReader r = new TextReader(proc.getInputStream());
			int c;
			for(c=0; r.hasNext(); c++){
				r.next();
			}
			assertEquals(6000, c);
			assertEquals(1, proc.exitValue());
			
			RubyException[] e = RubyException.createFromLines(new TextReader(tmpFile.toURL()));
			assertEquals(1, e.length);
			assertEquals("in `x': private method `x' called for nil:NilClass (NoMethodError)", e[0].getMessage());
			
			RubyException.TraceItem[] items = e[0].getTraceItems();
			for (int i = 0; i < items.length; i++) {
				assertEquals("-e", items[i].script);
			}
			assertEquals(2, items.length);
		}finally{
			tmpFile.delete();
		}
	}
		
	public void testStdErrOutput() throws Exception {
		File tmpFile = WebpubTestPlugin.createTempFile("PublisherBuilderTest.testStdErrOutput");
		try{
			String arg = "-e '$stderr = open(%Q!" + tmpFile.getAbsolutePath().replaceAll("\\\\", "/") + "!, \"w\") ;for i in 0...6000; $stdout << %Q!It is STDOUT #{i}\\n!; $stderr << %Q!It is STDERR #{i}\\n!; end";
			Process proc = WebpubPlugin.getDefault().rubyExec(arg, null);
			
			TextReader r = new TextReader(proc.getInputStream());
			int c;
			for(c=0; r.hasNext(); c++){
				r.next();
			}
			assertEquals(6000, c);
			
			assertEquals(0, proc.exitValue());
			assertTrue(tmpFile.exists());
			
			r = new TextReader(tmpFile.toURL());
			for(c=0; r.hasNext(); c++){
				r.next();
			}
			assertEquals(6000, c);
		}finally{
			tmpFile.delete();
		}
	}

	public void testBuildToOtherProject() throws Exception {
		IProject pubProj = ResourcesPlugin.getWorkspace().getRoot().getProject("pub_proj");
		pubProj.create(null);
		pubProj.open(null);
		
		pubProj.getFolder("a").create(true, true, null);
		pubProj.getFolder("a/b").create(true, true, null);
		pubProj.getFolder("a/b/c").create(true, true, null);
		
		WebProject webProj = WebpubTestPlugin.createTestProject("test_scripts/webpub/test_workspace/web_project");
		IProject proj = webProj.getProject();
		
		try{
			String[] testFolderTargets = {"/pub_proj", "/pub_proj/a", "/pub_proj/a/b/c"};
			
			for(int i=0; i<testFolderTargets.length; i++){
				// distribute to other project
				webProj.setValue(WebProject.KEY_PUBLISH_FOLDER, testFolderTargets[i]);
				webProj.storePreferences();
				
				IContainer pubFolder = webProj.getFolder(WebProject.KEY_PUBLISH_FOLDER);
				
				assertTrue(pubFolder.isAccessible());
				proj.build(IncrementalProjectBuilder.FULL_BUILD, null);
				
				PublishDescriptionFactory descFac = webProj.getPublishDescriptionFactory();
				
				IFile from = proj.getFile("ht_sources/home.html");
				PublishDescription desc = descFac.create(from);
				assertEquals(
						pubFolder.getFile(new Path("home.html")),
						desc.getPublishTo());
				assertEquals(from, findSource(desc));
				
				from = proj.getFile("ht_sources/folder/subfolder/subsubfolder/s-1.html");
				desc = descFac.create(from);
				assertEquals(pubFolder.getFile(
						new Path("folder/subfolder/subsubfolder/s-1.html")),
						desc.getPublishTo());
				assertEquals(from, findSource(desc));
				
				from = proj.getFile("erbs/a.erb");
				desc = descFac.create(from);
				assertEquals(
						pubFolder.getFile(new Path("erbs/a.html")),
						desc.getPublishTo());
				assertEquals(from, findSource(desc));
				
				from = proj.getFile("erbs/c/a/c-a-b.erb");
				desc = descFac.create(from);
				assertEquals(
						pubFolder.getFile(new Path("erbs/c-a-b.html")),
						desc.getPublishTo());
				assertEquals(from, findSource(desc));

				from = proj.getFile("rd/a.rd");
				desc = descFac.create(from);
				assertEquals(
						pubFolder.getFile(new Path("pub_rd/a.html")),
						desc.getPublishTo());
				assertEquals(from, findSource(desc));
				
				from = proj.getFile("rd/with_tmpl/c.rd");
				desc = descFac.create(from);
				assertEquals(
						pubFolder.getFile(new Path("pub_rd/amrita_wz_rd.html")),
						desc.getPublishTo());
				assertEquals(from, findSource(desc));
			}
		}finally{
			WebpubTestPlugin.forceDelete(proj);
			WebpubTestPlugin.forceDelete(pubProj);
		}
	}
	
	IResource findSource(PublishDescription desc) throws Exception{
		File file = desc.getPublishTo().getLocation().toFile();
		assertTrue(file.exists());
		return SourceFileSeeker.findSource(file.toURL());
	}

}
