#
# $Id: publish_manager.rb,v 1.1 2004/01/17 12:02:26 hn Exp $
# Copyright Narushima Hironori. All rights reserved.
#

require 'fileutils'
require 'webpub/publisher'

module Webpub

class PublisherEnvironmentException < StandardError
end

class PublishManager
	
	attr_accessor :result_output, :scripts_cache
	attr_reader :plugin_registory, :workspace, :plugins_entry_filepath

	def initialize
		@plugin_registory = PublisherRegistory.instance
		@workspace = EclipseWorkspace.instance
		
		@scripts_cache = true
		@default_ignores = ['.publish', 'CVS', 'Thumbs.db']
		
		@factory = PublishDescriptionFactory.new
		@result_output = $stdout
	end
	
	def plugins_entry_filepath
		raise PublisherEnvironmentException, "can't define publish plugin cache file location: workspace path is not specified" unless @workspace.workspace_path
		@workspace.workspace_path + '/.metadata/.plugins/com.narucy.webpub.core/.cache'
	end
	
	def check_env
		raise PublisherEnvironmentException, "publishers not found" if @plugin_registory.ids.empty?
		raise PublisherEnvironmentException, "workspace is not specified" unless @workspace.workspace_path
	end
	
	def store_plugins_entry
		file = plugins_entry_filepath()
		FileUtils.mkdir_p(File.dirname(file))
		@plugin_registory.store_publisher_entry_file(file)
	end
	
	def load_plugins_entry
		@plugin_registory.regist_plugins_from_entry_file(plugins_entry_filepath())
	end
	
	def check_files(files)
		ws = EclipseWorkspace.instance
		ws.project_pathes.each { |name, path|
			if /^#{path}/i === files[0]
				files.each { |f|
					raise ArgumentError, "invalid publish target files, must be specify same project: #{f}" unless /^#{path}/ === f
				}
				return ws.web_project(name)
			end
		}
		
		raise ArgumentError, %Q!specify files aren't included project: #{files.join("\n")}!
	end
	
	def publish(files, by_orders = nil)
		check_env()
		
		files = [files] if files.instance_of?(String)
		files.map! { |f|
			File.expand_path(f)
		}
		files.reject! { |f|
			@default_ignores.any? { |pattern| f.split('/').any? { |p| File::fnmatch?(pattern, p) } }
		}
		wp = check_files(files)
		$: << wp[:scripts_dir]
		
		requests = {}
		files.each { |f|
			if desc = @factory.create(f)
				by = desc.by
				if !by_orders or by_orders.include?(by)
					requests[by] = [] unless requests[by]
					requests[by] << desc
				end
			end
		}
		
		ok_descs = []
		failed_descs = []
		
		requests.each { |by, descriptions|
			publisher = @plugin_registory[by]
			
			unless publisher
				descriptions.each { |desc|
					
					ex = StandardError.new("publisher not found `#{by}'")
					ex.set_backtrace([])
					@result_output << "<<<FAILED,#{desc.publish_from},#{by},#{desc.publish_to},\n" +
						Webpub::to_fail_str(ex) +
						">>>\n"
					failed_descs << desc
				}
				next
			end
			
			descriptions.each { |desc|
				result = nil
				
				begin
					to = desc.publish_to
					to = File.dirname(to) unless /\/$/ === to
					FileUtils.mkdir_p(to)
					publisher.publish(desc)
					
					result = "<<<OK,#{desc.publish_from},#{desc.by},#{desc.publish_to},>>>\n"
					ok_descs << desc
				rescue Exception
					result =
						"<<<FAILED,#{desc.publish_from},#{desc.by},#{desc.publish_to},\n" +
						Webpub::to_fail_str($!) +
						">>>\n"
					
					failed_descs << desc
				end
				
				@result_output << result
			}
		}
		
		$:.delete(wp[:scripts_dir])
		
		[ok_descs, failed_descs]
	end
	
	def PublishManager::result_parser(lines)
		ok = []
		failed = []
		lines.each { |l|
			if /^<<<(.+?),(.+?),(.+?),(.+?),/ === l
				case $1
					when 'OK'
						ok << [$2, $3, $4]
					when 'FAILED'
						failed << [$2, $3, $4, []]
				end
			else
				failed.last[3] << l if failed.last and !(/^>>>$/ === l)
			end
		}
		
		[ok, failed]
	end
	
end

end
