/*
 * $Id: WebProject.java,v 1.11 2004/04/24 05:04:51 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core;

import java.io.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.core.template.*;
import com.narucy.webpub.core.toc.*;

/**
 * WebProject represents web resources keeps web project.
 */
public class WebProject extends Preferences implements IProjectNature {
	
	/**
	 * The web project nature id.
	 */
	final public static String ID_NATURE = WebpubPlugin.ID_PLUGIN + ".publishNature";
	
	/**
	 * Reference key for html source folder.
	 */
	final public static String KEY_HTSOURCES_FOLDER = "htsources_folder";
	
	/**
	 * Reference key for scripts folder.
	 */
	final public static String KEY_SCRIPTS_FOLDER = "scripts_folder";
	
	/**
	 * Reference key for publish folder.
	 */
	final public static String KEY_PUBLISH_FOLDER = "publish_folder";
	
	/**
	 * Referencekey for template synchronization extensions, and
	 * file specify publish definication featture (&gt;?publish by="foo"?&lt;) enabled.
	 */
	final public static String KEY_HT_EXTENSIONS = "ht_extensions";
	
	/**
	 * Reference key for HTML publish and template synchironiation process using
	 * encoding.
	 */
	final public static String KEY_ENCODING = "encoding";
	
	/**
	 * Refernce key for toc file pathes.
	 */
	final public static String KEY_TOC_PATHES = "toc_pathes";
	
	/**
	 * Key for list of web contents mapping url.
	 */
	final public static String KEY_MAPPED_URL = "mapped_url";

	/**
	 * Key for list of directory index (for specified url omitted only directory name).
	 */
	final public static String KEY_DIRECTORY_INDEX = "directory_index";
	
	/**
	 * Ignore publish target file.
	 */
	final public static String KEY_IGNORE_FILE = "ignore_publish";
	
	final static String[] mustBuilderIds = {
		TemplateSynchronizeBuilder.ID_BUILDER,
		PublisherBuilder.ID_BUILDER,
	};
	static {
		Arrays.sort(mustBuilderIds);
	}
	
	IProject project;
	TemplateManager tmplManager;
	PublishDescriptionFactory descFactory;
	PublishLogStore logStore;

	HashMap tocCache = new HashMap();

	public WebProject(){
		setDefault(KEY_HTSOURCES_FOLDER, "");
		setDefault(KEY_SCRIPTS_FOLDER, "");
		setDefault(KEY_PUBLISH_FOLDER, "");
		
		setDefault(KEY_HT_EXTENSIONS, new String[]{"html", "htm", "erb"} );
		setDefault(KEY_ENCODING, "UTF-8");
		setDefault(KEY_TOC_PATHES, "");
		setDefault(KEY_MAPPED_URL, "");
		setDefault(KEY_IGNORE_FILE, new String[]{".publish", "CSV", "Thumbs.db"});
		setDefault(KEY_DIRECTORY_INDEX, new String[]{"index.html", "index.htm", "home.html", "home.htm", "default.html", "default.htm"});
	}
	
	public TemplateManager getTemplateManager(){
		return tmplManager;
	}

	public PublishLogStore getLogStore() {
		return logStore;
	}
	
	public PublishDescriptionFactory getPublishDescriptionFactory() {
		return descFactory;
	}
	
	/**
	 * Project owner explicit specified toc files.
	 * (return files are no check: file exist and specify file is valid for toc.
	 */
	public IFile[] getTocFiles(){
		String[] pathes = getArray(KEY_TOC_PATHES);
		IFile[] files = new IFile[pathes.length];
		for (int i = 0; i < pathes.length; i++) {
			files[i] = project.getFile(pathes[i]);
		}
		return files;
	}
	
	void removeTocCache(IFile f) throws CoreException{
		tocCache.remove(f);
	}

	public void setProject(IProject proj) {
		if(proj.equals(project)){
			return;
		}
		project = proj;
		try {
			load();
			
			tmplManager = new TemplateManager(this);
			descFactory = new PublishDescriptionFactory(this);
			logStore = new PublishLogStore(proj);
		} catch (CoreException e) {
			WebpubPlugin.handleException(e);
		}
	}
	
	public void setDefault(String name, String[] vals){
		setDefault(name, toLine(vals));
	}
	
	public void setValue(String name, String[] vals){
		setValue(name, toLine(vals));
	}
	
	public String[] getArray(String name){
		return toArray(getString(name));
	}
	
	public String[] getDefaultArray(String name){
		return toArray(getDefaultString(name));
	}
	
	public static String[] toArray(String s){
		s = s.trim();
		return s.length() > 0 ? s.split("\\s+") : new String[0];
	}

	public static String toLine(String[] val){
		StringBuffer buff = new StringBuffer();
		for(int i=0; i<val.length; i++){
			buff.append(val[i]);
			if(i != val.length - 1){
				buff.append(' ');
			}
		}
		return buff.toString();
	}
	
	public boolean isHTExtension(String extension){
		if( extension != null){
			String[] exts = getArray(WebProject.KEY_HT_EXTENSIONS);
			Arrays.sort(exts);
			return Arrays.binarySearch(exts, extension) >= 0;
		}
		return false;
	}

	public void load() throws CoreException{
		IFile descFile = project.getFile(".webproject");
		if(!descFile.isAccessible()){
			return;
		}
		try {
			load(descFile.getContents(true));
			// compatibility for old version.
			if(getString("ht_charset").length() > 0){
				setValue(KEY_ENCODING, getString("ht_charset"));
				storePreferences();
			}
			if( getString(KEY_HTSOURCES_FOLDER).equals( getString(KEY_PUBLISH_FOLDER)) ){
				System.out.println("Convert web project nature configuration file:" + WebProjectConfigurationLoader052.reloadDescription(this) );
				storePreferences();
			}
		} catch (IOException e) {
			throw new CoreException(
				new Status(
					IStatus.ERROR,
					WebpubPlugin.ID_PLUGIN,
					IStatus.OK,
					"Can not read web project configuration file:" + descFile,
					e));
		}
	}

	public void configure() throws CoreException {
		IProjectDescription desc = project.getDescription();
		ICommand cmds[] = desc.getBuildSpec();
		
		ArrayList newCmds = new ArrayList();
		
		String[] cmdIds = new String[cmds.length];
		for (int i = 0; i < cmds.length; i++) {
			ICommand cmd = cmds[i];
			cmdIds[i] = cmd.getBuilderName();
			newCmds.add(cmd);
		}
		Arrays.sort(cmdIds);
		
		for(int i=0; i<mustBuilderIds.length; i++){
			String mustId = mustBuilderIds[i];
			if(Arrays.binarySearch(cmdIds, mustId) < 0){
				ICommand newCmd = desc.newCommand();
				newCmd.setBuilderName(mustId);
				newCmds.add(newCmd);
			}
		}
		
		desc.setBuildSpec((ICommand[])newCmds.toArray(new ICommand[newCmds.size()]));
		project.setDescription(desc, null);
	}
	
	public void deconfigure() throws CoreException {
		IProjectDescription projDesc = project.getDescription();
		ICommand[] cmds = projDesc.getBuildSpec();
		ICommand[] newCmds = new ICommand[cmds.length - mustBuilderIds.length];
		for (int i=0, j=0; i<cmds.length; i++) {
			ICommand cmd = cmds[i];
			if(Arrays.binarySearch(mustBuilderIds, cmd.getBuilderName()) < 0){
				newCmds[j++] = cmd;
			}
		}
		projDesc.setBuildSpec(newCmds);
		project.setDescription(projDesc, null);
		
		Object[] ls = listeners.getListeners();
		for (int i = 0; i < ls.length; i++) {
			listeners.remove(ls[i]);
		}
		
		logStore.dispose();
	}

	/**
	 * <p>
	 * Returns web project specify resources (these are base html document
	 * folder, for html generation scripts folder, public document folder)
	 * path.
	 * 
	 * <p>
	 * @param key specify HTSOURCES_FOLDER, SCRIPTS_FOLDER, PUBLIC_FOLDER
	 */
	public IContainer getFolder(String key){
		return pathToContainer(getString(key));
	}

	/**
	 * <p>
	 * Argument string folderPath represetnts path of web base container.
	 *
	 * Return values:
	 * <pre>
	 *   /base     : IProject /base
	 *   foo       : IFolder  /(this project)/foo
	 *   /base/foo : IFolder /base/foo
	 * </pre>
	 * 
	 * <p>
	 * Does not check exist, sync, open container.
	 */
	public IContainer pathToContainer(String folderPath){
		if( folderPath.length() == 0){
			return project;
		}
		if( folderPath.charAt(0) == '/'){
			Path path = new Path(folderPath);
			IWorkspaceRoot root = project.getWorkspace().getRoot();
			IProject proj = root.getProject(path.segment(0));
			
			if(path.segmentCount() <= 1){
				return proj;
			}else{
				return proj.getFolder(path.removeFirstSegments(1));
			}
		}else{
			return project.getFolder(folderPath);
		}
	}

	public IProject getProject() {
		return project;
	}
	
	/**
	 * Return null load failed.
	 */
	public Toc getToc(IFile f){
		Toc toc = (Toc)tocCache.get(f);
		if(toc == null){
			try{
				toc = TocFactory.createToc(f);
				tocCache.put(f, toc);
			} catch (CoreException e){
				WebpubPlugin.handleException(e);
			}
		}
		return toc;
	}

	public void storePreferences() throws CoreException {
		try{
			IFile descFile = project.getFile(".webproject");
			FileOutputStream out = null;
			try{
				out = new FileOutputStream( descFile.getLocation().toFile() );
				store(out, null);
			}finally{
				if(out != null){
					out.close();
				}
			}
		}catch(IOException e){
			WebpubPlugin.handleException(e);
		}
	}

	void distributeFolders(IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("Create Web Project Folders", 3);
			// create folders
			String[] keys = {
				KEY_HTSOURCES_FOLDER,
				KEY_SCRIPTS_FOLDER,
				KEY_PUBLISH_FOLDER
			};
			for (int i = 0; i < keys.length; i++) {
				IContainer folder = getFolder(keys[i]);
				if(!folder.exists()){
					if(folder instanceof IFolder){
						((IFolder)folder).create(true, true, new SubProgressMonitor(monitor, 1));
					}else{
						throw new CoreException(new Status(IStatus.ERROR, WebpubPlugin.ID_PLUGIN, IStatus.OK, "Specify container is not exist:" + folder, null));
					}
				}
			}
		} finally{
			monitor.done();
		}
	}

	/**
	 * Search file members from specify folder key name.
	 * If specify key is null, returns all of project files.
	 */
	public IFile[] getFileMembers(String key) throws CoreException {
		IContainer folder = getFolder(key);
		ArrayList dist = new ArrayList();
		if(folder.isAccessible()){
			collectFileMembers(folder, dist);
		}
		return (IFile[])dist.toArray(new IFile[dist.size()]);
	}
	
	static void collectFileMembers(IContainer folder, ArrayList distlist) throws CoreException{
		IResource[] files = folder.members();
		for (int i = 0; i < files.length; i++) {
			IResource res = files[i];
			if(res instanceof IFolder){
				collectFileMembers( (IFolder)res, distlist);
			} else if(res instanceof IFile){
				distlist.add(res);
			}
		}
	}

	/**
	 * Distrubte new project.
	 */
	public static WebProject distributeNewProject(IProject proj, String htSourceFolder, String scriptsFolder, String publicFolder, IProgressMonitor monitor) throws CoreException {
		if(monitor == null){
			monitor = new NullProgressMonitor();
		}
		try {
			monitor.beginTask("store new web project", 4);
			
			// sets nature
			IProjectDescription desc = proj.getWorkspace().newProjectDescription(proj.getName());
			desc.setNatureIds(new String[]{ID_NATURE});
			
			// project initialize
			boolean created = proj.exists();
			if(!created){
				proj.create(desc, new SubProgressMonitor(monitor,1));
			}
			if(!proj.isOpen()){
				proj.open(new SubProgressMonitor(monitor, 1));
			}
			if(created){
				proj.setDescription(desc, IResource.FORCE, new SubProgressMonitor(monitor,1) );
			}
			
			// store configuration file
			WebProject wp = (WebProject)proj.getNature(ID_NATURE);
			wp.setProject(proj);
			wp.setValue(KEY_HTSOURCES_FOLDER, htSourceFolder);
			wp.setValue(KEY_SCRIPTS_FOLDER, scriptsFolder);
			wp.setValue(KEY_PUBLISH_FOLDER, publicFolder);
			wp.storePreferences();

			// create folders.
			wp.distributeFolders(new SubProgressMonitor(monitor,1));
			
			// TODO: why call manually configure() method?
			wp.configure();
			
			return wp;
		} finally{
			monitor.done();
		}
	}

	public static WebProject[] getWebProjects() throws CoreException {
		ArrayList dist = new ArrayList();
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		for (int i = 0; i < projects.length; i++) {
			IProject project = projects[i];
			if (project.isAccessible()) {
				WebProject wp = (WebProject) project.getNature(ID_NATURE);
				if (wp != null) {
					dist.add(wp);
				}
			}
		}
		return (WebProject[]) dist.toArray(new WebProject[dist.size()]);
	}

}
