/*
 * $Id: SourceFileSeeker.java,v 1.7 2004/04/14 23:19:08 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;
import java.io.File;
import java.net.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;
/**
 * PublishFromSeeker provides search publish from content file function.
 */
public class SourceFileSeeker {
	
	public static IFile findSource(String urlStr) throws CoreException {
		IFile f = urlToFile(urlStr);
		return f != null ? findSource(f) : null;
	}

	public static IFile findSource(URL url) throws CoreException {
		IFile f = urlToFile(url);
		return f != null ? findSource(f) : null;
	}
	
	public static IFile findSource(IFile f) throws CoreException {
		return new SourceFileSeeker(f).getPublishFrom();
	}
	
	IFile publishFile;
	WebProject webProject = null;
	
	public static IFile urlToFile(String urlStr) throws CoreException {
		// for cgi value.
		urlStr = urlStr.replaceFirst("\\?.*", "");
		URL url = null;
		try {
			File f = new File(urlStr);
			if (f.exists()) {
				url = f.toURL();
			}
			url = new URL(urlStr);
		} catch (MalformedURLException e) {
		}
		return url != null ? urlToFile(url) : null;
	}
	
	/**
	 * URL to ht resource location. returns publish file.
	 */
	public static IFile urlToFile(URL url) throws CoreException {
		// to workspace file
		Path path = new Path(new File(url.getFile()).toString());
		IFile file = ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(path);
		if (file != null && file.isAccessible()) {
			return file;
		}
		// find from mapped url.
		String urlStr = url.toString();
		WebProject[] wps = WebProject.getWebProjects();
		for (int i = 0; i < wps.length; i++) {
			WebProject wp = wps[i];
			IContainer pubFolder = wp.getFolder(WebProject.KEY_PUBLISH_FOLDER);
			String[] urls = wp.getArray(WebProject.KEY_MAPPED_URL);
			for (int j = 0; j < urls.length; j++) {
				String u = urls[j];
				if (urlStr.toLowerCase().startsWith(u.toLowerCase())) {
					String relPath = urlStr.substring(u.length());
					if (relPath.charAt(relPath.length() - 1) == '/') {
						String[] indexFileNames = wp.getArray(WebProject.KEY_DIRECTORY_INDEX);
						for (int k = 0; k < indexFileNames.length; k++) {
							IFile f = pubFolder.getFile(new Path(indexFileNames[k]));
							if (f.isAccessible()) {
								return f;
							}
						}
					} else {
						IFile f = pubFolder.getFile(new Path(relPath));
						if (f.isAccessible()) {
							return f;
						}
					}
				}
			}
		}
		return null;
	}
	
	public SourceFileSeeker(IFile pubFile) throws CoreException {
		this.publishFile = pubFile;
		WebProject[] wps = WebProject.getWebProjects();
		for (int i = 0; i < wps.length; i++) {
			WebProject wp = wps[i];
			IContainer pubFolder = wp.getFolder(WebProject.KEY_PUBLISH_FOLDER);
			if (pubFolder.getProject().equals(publishFile.getProject()) && isParentFolder(publishFile, pubFolder)) {
				webProject = wp;
				break;
			}
		}
	}
	
	/**
	 * <p>
	 * Returns publish source file of constructer specified url published file.
	 * 
	 * <ul>
	 * <li>If specify file is not inclulded in publish folder of web project.
	 * <li>Next, search way to simple mapping "ht source folder"/"public
	 * folder".
	 * <li>If source file not found in this method work, search publish
	 * descriptor, and map a file.
	 * </ul>
	 */
	IFile getPublishFrom() throws CoreException {
		if (webProject == null) {
			return null;
		}
		IContainer pubFolder = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER);
		if (pubFolder == null || !isParentFolder(publishFile, pubFolder)) {
			return null;
		}
		IContainer htFolder = webProject.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
		IPath publishPath = pubFolder.getFullPath();
		IPath relPath = publishFile.getFullPath().removeFirstSegments(publishPath.segmentCount());
		IFile htSrc = htFolder.getFile(relPath);
		if (htSrc.exists()) {
			return htSrc;
		}
		return getPublishFromAsPublishDescriptions();
	}
	
	IFile getPublishFromAsPublishDescriptions() throws CoreException {
		// if not found ht source mapping, search from publish descriptors.
		IFile[] publishPropFiles = findPublishPropertyFiles();
		for (int i = 0; i < publishPropFiles.length; i++) {
			IFile propFile = publishPropFiles[i];
			IContainer fromBase = propFile.getParent();

			PublishDescriptionDefinition def = PublishDescriptionDefinitionStore.getInstance().getDescriptionDefinication(propFile);
			for (int j = 0; j < def.getEntryCount(); j++) {
				Wildcard wildcard = def.getPattern(j);
				String pubTo = (String) def.getArguments(j).get("publish_to");
				if (pubTo != null) {
					IFile file = isPublishFileEntried(fromBase, wildcard, pubTo, publishFile);
					if (file != null) {
						return file;
					}
				}
			}
		}
		return null;
	}
	
	/**
	 * Find with wild card pattern string.
	 */
	IFile isPublishFileEntried(IContainer fromContainer, Wildcard wildcard, String publishTo, IFile publishFile) throws CoreException {
		// direct match
		IFile tobePublishedFile = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFile(new Path(publishTo));

		IFile[] fromFiles = getMatchedFile(fromContainer, wildcard);
		if (publishFile.equals(tobePublishedFile)) {
			return fromFiles[0];
		} else if (publishTo.charAt(publishTo.length() - 1) == '/') {
			for (int i = 0; i < fromFiles.length; i++) {
				IFile fromFile = fromFiles[i];
				if (fromFile.getName().equals(publishFile.getName())) {
					return fromFile;
				}
			}
		} else if (publishTo.indexOf('*') != -1) {
			// refer publish location path.
			IPath publishLocation = new Path(publishTo).removeLastSegments(1);
			IFolder parentFolder = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFolder(publishLocation);
			if (publishFile.getParent().equals(parentFolder)) {
				// do find as from container.
				for (int i = 0; i < fromFiles.length; i++) {
					IFile fromFile = fromFiles[i];
					if (getBaseName(fromFile.getName()).equals(getBaseName(publishFile.getName()))) {
						return fromFile;
					}
				}
			}
		}
		return null;
	}
	
	IFile[] getMatchedFile(IContainer fromContainer, Wildcard wildcard) throws CoreException{
		ArrayList list = new ArrayList();
		
		IFile[] fromFiles = getFiles(fromContainer);
		for (int i = 0; i < fromFiles.length; i++) {
			IFile f = fromFiles[i];
			String relPath = f.getFullPath().removeFirstSegments(fromContainer.getFullPath().segmentCount()).toString();
			if (wildcard.match(relPath)){
				list.add(f);
			}
		}
		
		return (IFile[]) list.toArray(new IFile[list.size()]);
	}
	
	static IFile[] getFiles(IContainer container) throws CoreException {
		ArrayList dist = new ArrayList();
		collectFiles(container, dist);
		return (IFile[]) dist.toArray(new IFile[dist.size()]);
	}
	
	static void collectFiles(IContainer container, List dist) throws CoreException {
		IResource[] members = container.members();
		for (int i = 0; i < members.length; i++) {
			IResource r = members[i];
			if (r instanceof IFile) {
				dist.add(r);
			} else if (r instanceof IContainer) {
				collectFiles((IContainer) r, dist);
			}
		}
	}
	
	static String getBaseName(String file) {
		int separatorIndex = file.lastIndexOf('/');
		if (separatorIndex != -1) {
			file = file.substring(separatorIndex + 1);
		}
		int dotIndex = file.lastIndexOf('.');
		if (dotIndex != -1) {
			file = file.substring(0, dotIndex);
		}
		return file;
	}
	
	static boolean isParentFolder(IFile file, IContainer target) {
		IResource res = file;
		while (!((res = res.getParent()) instanceof IWorkspaceRoot)) {
			if (target.equals(res)) {
				return true;
			}
		}
		return false;
	}
	
	IFile[] findPublishPropertyFiles() throws CoreException {
		ArrayList dist = new ArrayList();
		collectPublishPropertyFiles(webProject.getProject(), dist);
		IFile[] files = (IFile[]) dist.toArray(new IFile[dist.size()]);
		final IContainer htFolder = webProject.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
		Arrays.sort(files, new Comparator() {
			public boolean equals(Object obj) {
				return false;
			}
			public int compare(Object a, Object b) {
				IFile af = (IFile) a;
				IFile bf = (IFile) b;
				boolean inHtSourceA = isParentFolder(af, htFolder);
				boolean inHtSourceB = isParentFolder(bf, htFolder);
				if (inHtSourceA != inHtSourceB) {
					return inHtSourceA ? 1 : -1;
				}
				int as = af.getFullPath().segmentCount();
				int bs = bf.getFullPath().segmentCount();
				return (as != bs) ? (as < bs ? 1 : -1) : 0;
			}
		});
		return files;
	}
	
	void collectPublishPropertyFiles(IContainer folder, List dist) throws CoreException {
		WebProject[] wps = WebProject.getWebProjects();
		for (int i = 0; i < wps.length; i++) {
			if (wps[i].getFolder(WebProject.KEY_PUBLISH_FOLDER).equals(folder)) {
				return;
			}
		}
		IResource[] resources = folder.members();
		for (int i = 0; i < resources.length; i++) {
			IResource r = resources[i];
			if (r instanceof IFile && ((IFile) r).getName().equals(".publish")) {
				dist.add(r);
			} else if (r instanceof IContainer) {
				collectPublishPropertyFiles((IContainer) r, dist);
			}
		}
	}
	
}
