/*
 * $Id: TemplateParser.java,v 1.7 2004/04/24 05:04:51 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.template;

import java.io.*;
import java.util.*;
import java.util.regex.*;

import com.narucy.webpub.core.TextReader;

/**
 * TemplateParser role is to create root templates.
 */
public class TemplateParser {

	final static String
		NAME_ROOT_TEMPLATE = "Root";
	
	final static Pattern
		beginPattern = Pattern.compile("^\\s*<!--\\s*\\$begin:(.+)\\$\\s*-->\\s*$"),
		endPattern = Pattern.compile("^\\s*<!--\\s*\\$end:(.+)\\$\\s*-->\\s*$");

	/**
	 * Template property value of find line number.
	 * value type of Integer that represents find line number.
	 */
	final public static String
		PROP_OFFSET_LINE = "offset_line",
		PROP_PARENT = "parent";
		
	public Template parse(InputStream stream, String encoding) throws TemplateParseException, IOException {
		TextReader reader = new TextReader(stream, encoding);
		try{
			return parse(reader);
		} finally {
			if( reader != null){
				reader.close();
			}
		}
	}

	public Template parse(Iterator iter) throws TemplateParseException {
		return parse(iter, NAME_ROOT_TEMPLATE);
	}
	
	public Template parse(Iterator iter, String rootTmplName) throws TemplateParseException {
		Template root = new Template(rootTmplName);
		
		Stack stack = new Stack();
		Template template = root;
		int ln;
		for (ln=0; iter.hasNext(); ln++) {
			String line = (String)iter.next();
			
			String name = matchBeginTmplateLine(line);
			if(name != null){
				// check specify template name is already added in parent
				for(int i=0; i<stack.size(); i++){
					if( ((Template)stack.get(i)).getName().equals(name) ){
						throw new TemplateParseException("illegal template hierarchy, duplicate template name:" + name, ln);
					}
				}
				Template newTemplate = new Template(name);
				newTemplate.setProperty(PROP_OFFSET_LINE,  new Integer(ln));
				newTemplate.setProperty(PROP_PARENT, template);
				
				stack.push(template);
				template.addTemplate(newTemplate);
				template = newTemplate;
			}
			template.addLine(line);
			String endName = matchEndTemplateLine(line);
			if( endName != null){
				if(template == null || stack.isEmpty()){
					throw new TemplateParseException(
						"illegal enclose, not beginning template '" + endName + "'", ln);
				}
				if(!endName.equals(template.getName())){
					throw new TemplateParseException(
						"illegal enclose, begin template name is '" + template.getName() + "' but was '" + endName + "'", ln);
				}
				template = (Template)stack.pop();
			}
		}
		if(template != root){
			StringBuffer buff = new StringBuffer(template.getName());
			for(int i=stack.size()-1; i>=1; i--){
				buff.append(',');
				buff.append( ((Template)stack.get(i)).getName() );
			}
			throw new TemplateParseException(
					"illegal enclose, not closed '" + buff.toString() + "'", ln);
		}
		
		return root;
	}
	
	public static String matchBeginTmplateLine(String line){
		Matcher beginMatch = beginPattern.matcher(line);
		if(beginMatch.matches()){
			return beginMatch.group(1);
		}
		return null;
	}
	
	public static String matchEndTemplateLine(String line){
		Matcher endMatch = endPattern.matcher(line);
		if(endMatch.matches()){
			return endMatch.group(1);
		}
		return null;
	}


}
