 /*
 * $Id: HTEditor.java,v 1.3 2004/04/24 05:04:45 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.editors.html;

import java.util.Arrays;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.text.*;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.util.*;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.ui.*;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.texteditor.*;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.ui.views.properties.IPropertySheetPage;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.editors.ColorManager;

public class HTEditor extends TextEditor implements IPropertyChangeListener {

	final public static String
		ID_KEYBIND_SCOPE = "com.narucy.webpub.ui.editors.html.HTEditorScope";

	final static String[] relationalPreferenceKeys = {
		WebpubUIPlugin.HT_EDITOR_COLOR_DEFAULT,
		WebpubUIPlugin.HT_EDITOR_COLOR_TAG,
		WebpubUIPlugin.HT_EDITOR_COLOR_COMMENT,
		WebpubUIPlugin.HT_EDITOR_COLOR_ERB,
		WebpubUIPlugin.HT_EDITOR_COLOR_ATTR,
		WebpubUIPlugin.HT_EDITOR_COLOR_PROCESSOR,
	};
	
	static {
		Arrays.sort(relationalPreferenceKeys);
	}

	ColorManager colorManager;
	HTDocumentProvider docProvider;
	HTOutlinePage outlinePage;
	HTPropertySheetPage propPage;
	
	int lastCursorLineNum = -1;
	
	protected void handleCursorPositionChanged() {
		super.handleCursorPositionChanged();
		if(outlinePage != null){
			ISourceViewer sourceViewer = getSourceViewer();
			IDocument doc = sourceViewer.getDocument();
			int position = widgetOffset2ModelOffset( sourceViewer, sourceViewer.getTextWidget().getCaretOffset());
			try{
				int lineNum = doc.getLineOfOffset(position);
				if(lineNum != lastCursorLineNum){
					lastCursorLineNum = lineNum;
					outlinePage.handleCursorLineChanged(lineNum);
				}
			}catch(BadLocationException e){
				WebpubUIPlugin.handleException(e);
			}
		}
	}
	
	protected void initializeEditor() {
		// configure editor
		setEditorContextMenuId("#TextEditorContext");
		setRulerContextMenuId("#TextRulerContext");	
		
		configureInsertMode(SMART_INSERT, false);
		setInsertMode(INSERT);
		
		setKeyBindingScopes(new String[]{ID_KEYBIND_SCOPE});
		setRangeIndicator(new DefaultRangeIndicator());
		
		docProvider = new HTDocumentProvider();
		setDocumentProvider(docProvider);

		// initialize instance values.
		colorManager = new ColorManager();

		HTViewerConfiguration conf = new HTViewerConfiguration(colorManager);
		setSourceViewerConfiguration(conf);

		// init preferenancesss
		IPreferenceStore prefStore = WebpubUIPlugin.getDefault().getPreferenceStore();
		setPreferenceStore(prefStore);
		WorkbenchChainedTextFontFieldEditor.startPropagate(prefStore, JFaceResources.TEXT_FONT);
		
		prefStore.addPropertyChangeListener(this);
	}
	
	public void propertyChange(PropertyChangeEvent event){
		ISourceViewer sourceViewer = getSourceViewer();
		if(sourceViewer == null){
			return;
		}
		
		HTViewerConfiguration conf = (HTViewerConfiguration)getSourceViewerConfiguration();
		String prop = event.getProperty();
		IPreferenceStore store = getPreferenceStore();

		StyledText styledText = sourceViewer.getTextWidget();
		if(prop.equals(WebpubUIPlugin.HT_EDITOR_DISPLAYED_TAB_WIDTH)){
			styledText.setTabs(store.getInt(prop));
		}else if(prop.equals(WebpubUIPlugin.HT_EDITOR_WORDWRAP)){
			styledText.setWordWrap(store.getBoolean(prop));
		}else if(prop.equals(WebpubUIPlugin.HT_EDITOR_COLOR_BACKGROUND)){
			styledText.setBackground( colorManager.getColor(prop));
		}else {
			conf.propertyChange(event);
		}
	}
	
	public Object getAdapter(Class required) {
		if (IContentOutlinePage.class.equals(required)) {
			return getOutlinePage();
		}
		if (required.equals(IPropertySheetPage.class)) {
			if(propPage == null){
				propPage = new HTPropertySheetPage();
			}
			return propPage;
		}
		return super.getAdapter(required);
	}
	
	HTOutlinePage getOutlinePage(){
		if (outlinePage == null) {
			outlinePage = new HTOutlinePage(this);
		}
		return outlinePage;
	}

	protected boolean affectsTextPresentation(PropertyChangeEvent event) {
		return Arrays.binarySearch(relationalPreferenceKeys, event.getProperty()) >= 0;
	}

	protected void createActions() {
		super.createActions();
		installEncodingSupport();
		IEditorInput input = getEditorInput();
		if(input instanceof IFileEditorInput){
			try{
				WebProject wp = (WebProject)((IFileEditorInput)input).getFile().getProject().getNature(WebProject.ID_NATURE);
				if(wp != null){
					fEncodingSupport.setEncoding(wp.getString(WebProject.KEY_ENCODING));
				}
			}catch(CoreException e){
				WebpubUIPlugin.handleException(e);
			}
		}

		ContentAssistAction action = new ContentAssistAction(
			WebpubUIPlugin.getResourceBundle(),
			"ContentAssistProposal.",
			this);

		// This action definition is associated with the accelerator Ctrl+Space
		action.setActionDefinitionId(ITextEditorActionDefinitionIds.CONTENT_ASSIST_PROPOSALS);
		setAction("ContentAssistProposal", action);
	}

	public void dispose() {
		getPreferenceStore().removePropertyChangeListener(this);
		colorManager.dispose();
		super.dispose();
	}

}
