/*
 * $Id: MapEditDialog.java,v 1.3 2004/04/24 05:04:46 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.util.*;

import org.eclipse.jface.dialogs.*;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

import com.narucy.webpub.ui.TableLabelProvider;

/**
 * MapEditDialog roles to edit given Map instance.
 */
public class MapEditDialog extends Dialog {

	final static String[] propNames = { "Key", "Value" };

	ArrayList editTarget = new ArrayList();
	String[] listProperties = new String[]{ "key", "value" };

	int initialWidth = 400, initialHeight = 300;
	TableViewer tableViewer;
	Button addButton, removeButton, upButton, downButton;
	
	public MapEditDialog(Shell shell, Map map) {
		super(shell);
		setShellStyle(getShellStyle() | SWT.MAX | SWT.RESIZE);
		
		Object[] keys = map.keySet().toArray();
		Arrays.sort(keys);
		for (int i = 0; i < keys.length; i++) {
			String k = (String)keys[i];
			editTarget.add(new String[]{k, (String)map.get(k)});
		}
	}
	
	public Control createDialogArea(Composite parent) {
		Composite base = (Composite)super.createDialogArea(parent);
		base.setLayout(new GridLayout(2, false));
		
		new Label(base, SWT.NONE).setText("Maps keys to values:");
		new Label(base, SWT.NONE);
		
		// create table
		tableViewer = new TableViewer(
			base, SWT.BORDER | SWT.FULL_SELECTION | SWT.V_SCROLL | SWT.H_SCROLL);

		// innter table setting
		Table table = tableViewer.getTable();
		
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 4;
		table.setLayoutData(gd);
		for(int i=0; i<listProperties.length; i++){
			TableColumn col = new TableColumn(table, SWT.LEFT);
			col.setText(propNames[i]);
		}
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// cell settings,
		tableViewer.setColumnProperties(listProperties);
		
		// content provider settign
		tableViewer.setContentProvider(new ArrayContentProvider());
		
		// label provider settings.
		tableViewer.setLabelProvider(new TableLabelProvider() {
			public String getColumnText(Object element, int columnIndex) {
				return ((String[])element)[columnIndex];
			}
		});
		
		// cell modifier settings
		tableViewer.setCellModifier(new ICellModifier() {
			public boolean canModify(Object element, String property) {
				return true;
			}

			public Object getValue(Object element, String property) {
				return ((String[])element)[ property.equals("key") ? 0 : 1 ];
			}

			public void modify(Object element, String property, Object value) {
				if (element instanceof Item) {
					element = ((Item) element).getData();
				}
				((String[])element)[ property.equals("key") ? 0 : 1  ] = (String)value;
				tableViewer.refresh();
			}
		});
		tableViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshButtonStatus();
			}
		});
		
		// input data
		tableViewer.setInput(editTarget);
		
		// sets column size
		// minimun width to define refer initial width size. (initial width divide by three).
		TableColumn[] columns = table.getColumns();
		for (int i = 0, minSize = initialWidth / 3; i < columns.length; i++) {
			TableColumn column = columns[i];
			column.pack();
			if( column.getWidth() < minSize ){
				column.setWidth(minSize);
			}
		}
		tableViewer.setCellEditors(new CellEditor[]{
				new TextCellEditor(table),
				new TextCellEditor(table),
		});
		
		// set buttons
		addButton = createButton(base, "&Add", "addButton");
		removeButton = createButton( base, "&Remove", "removeButton");
		upButton = createButton(base, "&Up", "upButton");
		downButton = createButton(base, "&Down", "downButton");
		
		refreshButtonStatus();
		return base;
	}
	
	public void setSelectionIndex(int index){
		tableViewer.setSelection(new StructuredSelection(editTarget.get(index)));
	}
	
	void handleButtonPressed(Button b){
		int i = editTarget.indexOf(getCurrentSelection());
		
		if(addButton == b){
			callAddDialog(i);
		}
		if(removeButton == b){
			editTarget.remove(i);
		}
		if(upButton == b){
			editTarget.add(i-1, editTarget.remove(i));
		}
		if(downButton == b){
			editTarget.add(i+1, editTarget.remove(i));
		}
		tableViewer.refresh();
		refreshButtonStatus();
	}
	
	Button createButton(Composite base, String text, String controlName){
		Button button = new Button(base, SWT.PUSH);
		button.setLayoutData(
			new GridData(
				GridData.HORIZONTAL_ALIGN_FILL |
				GridData.VERTICAL_ALIGN_BEGINNING) );
		
		button.setText(text);
		button.setData(controlName);
		
		button.addListener( SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				handleButtonPressed( (Button)event.widget );
			}
		});
		
		return button;
	}
	
	void refreshButtonStatus(){
		String[] vals = getCurrentSelection();
		removeButton.setEnabled(vals != null);
		
		int i = editTarget.indexOf(vals);
		upButton.setEnabled(i != -1 && i > 0);
		downButton.setEnabled(i != -1 && i < editTarget.size()-1);
	}
	
	String[] getCurrentSelection(){
		IStructuredSelection sel = (IStructuredSelection)tableViewer.getSelection();
		return (String[])sel.getFirstElement();
	}
	
	void callAddDialog(int addIndex){
		String newKey = null;
		InputDialog dialog = new InputDialog(getShell(), "New Item Entry", "Input key of new item", null, null);
		if( dialog.open() ==  Window.OK){
			newKey = dialog.getValue();
			editTarget.add(
				addIndex < 0 ? editTarget.size() : addIndex,
				new String[]{newKey, ""});
		}
	}

	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("Edit Key/Value");
	}

	protected Point getInitialSize() {
		return new Point(initialWidth, initialHeight);
	}
	
	public String[][] getValues(){
		return (String[][])editTarget.toArray(new String[editTarget.size()][2]);
	}

}
