/*
 * $Id: ResourceMatchDialog.java,v 1.1 2004/04/11 08:23:38 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.util.*;
import java.util.List;
import java.util.regex.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.IDecoratorManager;
import org.eclipse.ui.dialogs.*;
import org.eclipse.ui.model.WorkbenchLabelProvider;

import com.narucy.webpub.core.Wildcard;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * ResourceMatchDialog.
 */
public class ResourceMatchDialog extends TwoPaneElementSelector {

	String
		initialPattern,
		baseMessage = "&Resource match (? = any charactor, * = any string, ** = any directory):",
		errorMessage = null;
	
	Label messageLabel;
	IContainer workingContainer;
	
	Button useRegexp;
	
	static ILabelProvider createLabelProvider(final IContainer workingContainer){
		
		ILabelProvider labelProvider = new WorkbenchLabelProvider(){
			protected String decorateText(String input, Object element){
				IResource res = (IResource)element;
				return getRelativePath(res, workingContainer);
			}
		};
		
		IDecoratorManager decoratorManager =
			WebpubUIPlugin.getDefault().getWorkbench().getDecoratorManager();
		return new DecoratingLabelProvider(labelProvider,
			decoratorManager.getLabelDecorator() );
	}
	
	static String getRelativePath(IResource target, IContainer base){
		String fullPath = base.getFullPath().toString();
		int dirIndex = fullPath.length();
		String relPath = target.getFullPath().toString().substring(dirIndex);
		if( relPath.length() > 0 && relPath.charAt(0) == '/'){
			relPath = relPath.substring(1);
		}
		if( target instanceof IFolder){
			relPath += '/';
		}
		return relPath;
	}

	static ILabelProvider createQualifierLabelProvider(){
		return new LabelProvider(){
			public String getText(Object element){
				return element.toString();
			}
		};
	}

	class WildcardMatcher implements FilteredList.FilterMatcher {
		Pattern pattern;
		Wildcard wildcard;

		public void setFilter(String ptn, boolean ignoreCase, boolean ignoreWildCards) {
			pattern = null;
			wildcard = null;
			errorMessage = null;
			try{
				if(useRegexp.getSelection()){
					pattern = Pattern.compile(ptn);
				}else{
					wildcard = new Wildcard(ptn);
				}
				errorMessage = null;
			} catch(PatternSyntaxException e){
				errorMessage = "Pattern syntax error (" + ptn + "):" + e.getMessage();
			}
			messageLabel.setText( errorMessage != null ? errorMessage.replaceAll("\n", "\\n") : baseMessage );
		}

		public boolean match(Object element) {
			IResource res = (IResource)element;
			String relPath = getRelativePath(res, workingContainer);
			
			boolean match =
				errorMessage == null && (
					(pattern != null && pattern.matcher(relPath).matches()) ||
					(wildcard != null && wildcard.match(relPath))
				);
			
			return match;
		}
	}

	public ResourceMatchDialog(Shell parent, String initialPattern, IContainer workingContainer) {
		super(parent, createLabelProvider(  workingContainer ), createQualifierLabelProvider() );
		this.initialPattern = initialPattern;
		this.workingContainer = workingContainer;
		try {
			setElements( getAllTargetMembers() );
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
		setTitle("Matching Resources");
		setMessage(baseMessage);
		setUpperListLabel("Match resources");
		setLowerListLabel("Resource full description");
		
		String ptn = isInitialPatternUseRegexp() ?
			initialPattern.substring(1, initialPattern.length()-1) : initialPattern;
		
		setFilter(ptn);
	}
	
	boolean isInitialPatternUseRegexp(){
		return initialPattern.charAt(0) == '/' &&
			initialPattern.charAt(initialPattern.length() - 1) == '/';
	}
	
	IResource[] getAllTargetMembers() throws CoreException{
		List list = new ArrayList();
		doFindMembers(workingContainer, list);
		return (IResource[])list.toArray(new IResource[list.size()]);
	}
	
	static void doFindMembers(IContainer container, List list) throws CoreException{
		IResource[] members = container.members();
		for (int i = 0; i < members.length; i++) {
			IResource r = members[i];
			list.add(r);
			if( r instanceof IContainer){
				doFindMembers((IContainer)r, list);
			}
		}
	}
		
	protected Text createFilterText(Composite parent) {
		Text text = super.createFilterText(parent);
		
		// create regexp expansion widget
		useRegexp = new Button(parent, SWT.CHECK);
		useRegexp.setText("Use Regular Expression");
		useRegexp.setSelection( isInitialPatternUseRegexp() );
		useRegexp.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				setFilter(getFilter()); // reload
				handleSelectionChanged();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		return text;
	}

	protected FilteredList createFilteredList(Composite parent) {
		FilteredList list = super.createFilteredList(parent);
		list.setFilterMatcher(new WildcardMatcher() );
		return list;
	}

	protected Label createMessageArea(Composite composite) {
		messageLabel = super.createMessageArea(composite);
		
		return messageLabel;
	}

}
