/*
 * $Id: HTPreView.java,v 1.6 2004/04/17 22:56:19 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.views;

import java.io.File;
import java.net.*;
import java.util.Arrays;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.*;
import org.eclipse.jface.resource.StringConverter;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.publish.SourceFileSeeker;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.actions.RepublishAction;

/**
 * HTPreView role is to define of browser selection 
 * and indication a HTML document and display Title.
 */
public class HTPreView extends ViewPart {

	final static String
		LAST_BROWSE_URL = "last_browse_url",
		KEY_URL_HISTORY = "url_history",
		LINK_WITH_EDITOR = "link_with_editor";

	IMemento memento;
	
	int maxUrlHistoryCount = 10;
	
	Browser browser;
	String lastSetUrl;
	int loadingProgress = -1;

	Combo addressBar;
	
	Action
		republishAction,
		openEditorAction,
		backAction,
		forwardAction,
		stopAction;
	
	String pageTitle = null;
	HTPreviewLinkWithEditorManager linkWithEditorMgr;
	
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		this.memento = memento;
	}

	/**
	 * Returns publish original file if url file is publishable.
	 * If other case, return null.
	 */
	public IFile getCurrentPublishFrom() {
		IFile from = null;
		try{
			if (lastSetUrl != null) {
				from = SourceFileSeeker.findSource(lastSetUrl);
			}
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
		return from;
	}
	
	/**
	 * Call back from eclipse resources. Reload web browser when changed resource is
	 * current browse file reloational project.
	 */
	IResourceChangeListener resourceListener = new IResourceChangeListener() {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			if (delta != null) {
				runReloadProc(delta);
			}
		}
		
		boolean isRequireReload(IResource r) throws CoreException{
			final IFile from = getCurrentPublishFrom();
			if(r instanceof IFile && from != null){
				WebProject wp = (WebProject)from.getProject().getNature(WebProject.ID_NATURE);
				if( wp == null){
					return from.getProject().equals(r.getProject());
				}else{
					IContainer pubFolder = wp.getFolder(WebProject.KEY_PUBLISH_FOLDER);
					for(IContainer c = r.getParent(); !(c instanceof IWorkspaceRoot); c = c.getParent() ){
						if(c.equals(pubFolder)){
							return true;
						}
					}
				}
			}
			return false;
		}

		void runReloadProc(final IResourceDelta delta) {
			IWorkbenchPartSite site = getSite();
			if(site != null && linkWithEditorMgr != null){
				Shell sh = site.getShell();
				if(sh != null){
					sh.getDisplay().asyncExec(new Runnable() {
						public void run() {
							reload(delta);
						}
					});
				}
			}
		}

		void reload(IResourceDelta delta){
			final boolean[] needReload = new boolean[]{false};
			
			try {
				delta.accept(new IResourceDeltaVisitor() {
					public boolean visit(IResourceDelta d) throws CoreException {
						if ( isRequireReload(d.getResource()) ) {
							needReload[0] = true;
							return false;
						}
						return true;
					}
				});
			} catch (CoreException e) {
				WebpubUIPlugin.handleException(e);
			}
			if(needReload[0]){
				browser.refresh();
			}
		}
	};

	/**
	 * short cut:
	 * getViewSite().getActionBars().getMenuManager()
	 */
	public IMenuManager getMenuManager() {
		return getViewSite().getActionBars().getMenuManager();
	}
	
	/**
	 * Create controls.
	 */
	public void createPartControl(Composite parent) {
		// create control
		Composite base = new Composite(parent, SWT.NONE);
		GridLayout gl = new GridLayout();
		gl.marginWidth = 0;
		gl.marginHeight = 0;
		base.setLayout(gl);
		
		// create address bar
		GridData gd = new GridData(GridData.VERTICAL_ALIGN_CENTER);
		gd.horizontalIndent = 2;
		
		addressBar = new Combo(base, SWT.BORDER | SWT.DROP_DOWN);
		addressBar.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		// create browser
		browser = new Browser(base, SWT.NONE);
		browser.setLayoutData(new GridData(GridData.FILL_BOTH));
				
		// init menu.
		IMenuManager menuMgr = getMenuManager();
		menuMgr.add(new Action("&Deselect") {
			public void run() {
				browse(null);
			}
		});
		
		linkWithEditorMgr = new HTPreviewLinkWithEditorManager(this);
		
		// create actions
		republishAction = new Action("Publish", WebpubUIPlugin.getImageDescriptor("clcl16/republish.gif")) {
			public void run() {
				IFile republishTarget = getCurrentPublishFrom();
				if (republishTarget != null) {
					RepublishAction act = new RepublishAction();
					act.init(getSite().getWorkbenchWindow());
					act.selectionChanged(this, new StructuredSelection(republishTarget));
					act.run(this);
				}
			}
		};
		republishAction.setToolTipText("Publish");
		
		openEditorAction = new Action("Open Editor", WebpubUIPlugin.getImageDescriptor("clcl16/open_editor.gif")) {
			public void run() {
				IFile f = getCurrentPublishFrom();
				try {
					IDE.openEditor(getSite().getPage(), f, true);
				} catch (PartInitException e) {
					WebpubUIPlugin.handleException(e);
				}
			}
		};
		openEditorAction.setToolTipText("Open Editor");
		
		backAction = new Action("Back", WebpubUIPlugin.getImageDescriptor("clcl16/backward_nav.gif")){
			public void run(){
				browser.back();
			}
		};
		backAction.setDisabledImageDescriptor(WebpubUIPlugin.getImageDescriptor("dlcl16/backward_nav.gif"));
		backAction.setToolTipText("Back");
		
		forwardAction = new Action("Forward", WebpubUIPlugin.getImageDescriptor("clcl16/forward_nav.gif")){
			public void run(){
				browser.forward();
			}
		};
		forwardAction.setDisabledImageDescriptor(WebpubUIPlugin.getImageDescriptor("dlcl16/forward_nav.gif"));
		forwardAction.setToolTipText("Foward");
		
		stopAction = new Action("Stop", WebpubUIPlugin.getImageDescriptor("clcl16/stop.gif")){
			public void run(){
				browser.stop();
			}
		};
		stopAction.setToolTipText("Stop");
		
		final IActionBars bars = getViewSite().getActionBars();

		// toolbar
		IToolBarManager toolBar = bars.getToolBarManager();
		toolBar.add(republishAction);
		toolBar.add(backAction);
		toolBar.add(forwardAction);
		toolBar.add(stopAction);
		toolBar.add(openEditorAction);
		toolBar.add(linkWithEditorMgr.getLinkWithEditorAction());
		
		// handle global action.
		bars.setGlobalActionHandler(ActionFactory.CUT.getId(), new Action() {
			public void run() {
				if( addressBar.isFocusControl()){
					addressBar.cut();
				}
			}
		});
		bars.setGlobalActionHandler(ActionFactory.COPY.getId(), new Action() {
			public void run() {
				if( addressBar.isFocusControl()){
					addressBar.copy();
				}
			}
		});
		bars.setGlobalActionHandler(ActionFactory.PASTE.getId(), new Action() {
			public void run() {
				if( addressBar.isFocusControl()){
					addressBar.paste();
				}
			}
		});
		
		// set selection for external web browser from "Run > External Tools".
		getSite().setSelectionProvider(new HTPreviewSelectionProvider(this));
		
		// listener setting
		addressBar.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				if (e.character == '\r' || e.character == '\n') {
					enterUrl();
				}
			}
		});
		addressBar.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				enterUrl();
			}
		});

		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceListener);
		browser.addProgressListener(new ProgressListener() {
			public void changed(ProgressEvent event) {
				if (0 < event.total){
					loadingProgress = event.current * 100 / event.total;
					if(loadingProgress < 100){
						setTitle("Loading... " + loadingProgress + "%");
					}else{
						// MS OLE or SWT Browser widget bug? not called completed method.
						completed(event);
					}
				}
			}

			public void completed(ProgressEvent event) {
				loadFinished();
			}
		});
		browser.addTitleListener(new TitleListener() {
			public void changed(TitleEvent event) {
				pageTitle = event.title;
			}
		});
		
		browser.addLocationListener(new LocationAdapter() {
			public void changed(LocationEvent event) {
				browse(event.location);
			}
		});
		
		browser.addStatusTextListener(new StatusTextListener() {
			public void changed(StatusTextEvent event) {
				setWorkbenchPageStatusMessage(event.text);
			}
		});

		// set memento values.
		if(memento != null){
			addressBar.setItems(memento.getString(KEY_URL_HISTORY).split("\\s+"));
			browse(memento.getString(LAST_BROWSE_URL));
			linkWithEditorMgr.getLinkWithEditorAction().setChecked(
				memento.getString(LINK_WITH_EDITOR).equals("true") );
		}
		
		refreshActionsState();
	}
	
	synchronized void loadFinished(){
		setWorkbenchPageStatusMessage(null);
		refreshActionsState();
		
		// to provide selection for external tools.
		getSite().getSelectionProvider().setSelection(new StructuredSelection(addressBar.getText()));
		setTitle(pageTitle != null ? pageTitle.replaceAll("&", "&&") : "Untitled");
		loadingProgress = -1;
	}
	
	void setWorkbenchPageStatusMessage(String message){
		getViewSite().getActionBars().getStatusLineManager().setMessage(message);
	}
	
	public boolean isLoading(){
		return loadingProgress != -1;
	}
	
	void enterUrl(){
		enterUrl(addressBar.getText());
	}
	
	/**
	 * Browse specify url and add to url history.
	 */
	public void enterUrl(String url){
		if(!isContainURLHistory(url) ){
			addressBar.add(url.toString(), 0);
			if( addressBar.getItemCount() > maxUrlHistoryCount){
				addressBar.remove(addressBar.getItemCount()-1);
			}
		}
		browse(url);
	}

	void refreshActionsState() {
		if( lastSetUrl != null && lastSetUrl.length() > 0){
			try{
				IFile pubFrom = getCurrentPublishFrom();
				republishAction.setEnabled(pubFrom != null && new URL(lastSetUrl).getProtocol().equals("file"));
				openEditorAction.setEnabled(pubFrom != null);
			}catch(MalformedURLException e){
				republishAction.setEnabled(false);
				openEditorAction.setEnabled(false);
			}
		}
		backAction.setEnabled(browser.isBackEnabled());
		forwardAction.setEnabled(browser.isForwardEnabled());
		
		linkWithEditorMgr.handleWebBorwserUrlChanged();
	}
	
	public String[] getHistory(){
		return addressBar.getItems();
	}

	boolean isContainURLHistory(String url){
		String[] history = getHistory();
		Arrays.sort(history);
		return Arrays.binarySearch(history, url) >= 0;
	}

	public boolean isPreviewble(IFile file) {
		String ext = file.getFileExtension();
		if(ext == null){
			return false;
		}
		String[] previewExts = StringConverter.asArray(
			WebpubUIPlugin.
				getDefault().
				getPreferenceStore().
				getString(WebpubUIPlugin.HT_PREVIEW_EXTENSIONS));
		Arrays.sort(previewExts);
		return Arrays.binarySearch(previewExts, ext) >= 0;
	}

	public void browse(String urlStr) {
		boolean available = urlStr != null;
		browser.setEnabled(available);
		
		if(available){
			urlStr = adjustUrlString(urlStr);
			if(!urlStr.equals(lastSetUrl)){
				lastSetUrl = urlStr;
				loadingProgress = 0;
				browser.setUrl(urlStr);
			}
			if(!urlStr.equals(addressBar.getText())){
				addressBar.setText(urlStr);
			}
		} else {
			lastSetUrl = null;
			browser.setText("");
			setTitle("An control is not an available");
			refreshActionsState();
		}
	}
	
	public String adjustUrlString(String urlStr){
		URL u = null;
		try {
			File f = new File(urlStr);
			if(f.exists()){
				u = f.toURL();
			}else{
				u = new URL(urlStr);
			}
		} catch (MalformedURLException e1) {
			try {
				u = new URL("http://" + urlStr);
			} catch (MalformedURLException x) {
			}
		}
		return u != null ? u.toString() : urlStr;
	}

	public void setFocus() {
	}

	public String getUrl(){
		return lastSetUrl;
	}
	
	public HTPreviewLinkWithEditorManager getLinkWithEditorManager(){
		return linkWithEditorMgr;
	}

	public void saveState(IMemento memento) {
		memento.putString(
			LINK_WITH_EDITOR,
			String.valueOf(linkWithEditorMgr.getLinkWithEditorAction().isChecked()));

		memento.putString(LAST_BROWSE_URL, lastSetUrl);

		String[] urls = addressBar.getItems();
		memento.putString(KEY_URL_HISTORY, WebpubUIPlugin.join(urls));
	}

	public void dispose() {
		linkWithEditorMgr.dispose();
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
		super.dispose();
	}


}