package com.narucy.webpub.ui.views;

import java.io.*;
import java.text.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.*;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.ViewPart;

import com.narucy.webpub.core.*;
import com.narucy.webpub.core.publish.PublishLogStore;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * 
 */
public class PublishConsoleView extends ViewPart {

	final static String CUSTOM_ENCODING = "custom_encoding";
	
	final static String
		KEY_CONSOLE_FONT = "consoleFont",
		ID_SELECTED_FILE = "selectedFile";

	File selectedFile;

	Text consoleText;
	Action logSelectionAction, pinAction;
	String customEncoding = null;
	
	String[] reloadPrioritySuffix = {"stderr", "stdout"};
	DateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
	
	IMenuCreator menuCreator = new IMenuCreator() {
	
		Image
			stdoutIcon = WebpubUIPlugin.createImage("clcl16/stdout.gif"),
			stderrIcon = WebpubUIPlugin.createImage("clcl16/stderr.gif");
	
		public Menu getMenu(Control parent) {
			Menu menu = new Menu(parent);
			Map logEntries = PublishLogStore.getLogEntries();
			
			Date[] dates = (Date[])logEntries.keySet().toArray(new Date[logEntries.size()]);
			Arrays.sort(dates);
			
			for (int i=dates.length-1; i>=0; i--) {
				Date date = dates[i];
				WebProject wp = (WebProject)logEntries.get(date);
				
				PublishLogStore logStore = wp.getLogStore();
				
				createShowLogAction(
						menu,
						logStore.getLogFile(date, "stderr"),
						stderrIcon);
				
				createShowLogAction(
						menu,
						logStore.getLogFile(date, "stdout"),
						stdoutIcon);
			}
			
			new MenuItem(menu, SWT.SEPARATOR);
			MenuItem item = new MenuItem(menu, SWT.NONE);
			item.setText("Clear History");
			item.addListener(SWT.Selection, new Listener() {
				public void handleEvent(Event event) {
					PublishLogStore.clearAllLogs();
					browse(null, false);
				}
			});

			return menu;
		}
		
		public Menu getMenu(Menu parent) {
			return null;
		}
		
		void createShowLogAction(Menu menu, final File file, Image img){
			if(file == null || !file.exists()){
				return;
			}
			MenuItem item = new MenuItem(menu, SWT.RADIO);
			item.setImage(img);
			item.setText(toLabelString(file));
			item.addListener(SWT.Selection, new Listener() {
				public void handleEvent(Event event) {
					browse(file, false);
				}
			});
				
			if(file.equals(selectedFile)){
				item.setSelection(true);
			}
		}
		
		public void dispose() {
			stderrIcon.dispose();
			stdoutIcon.dispose();
		}

	};
	
	IResourceChangeListener resourceListener = new IResourceChangeListener() {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			if(delta == null){
				return;
			}
			
			final WebProject[] webProject = {null};
			try{
				delta.accept(new IResourceDeltaVisitor() {
					public boolean visit(IResourceDelta delta) throws CoreException {
						IResource r = delta.getResource();
						if( r instanceof IFile && (delta.getKind() == IResourceDelta.CHANGED || delta.getKind() == IResourceDelta.ADDED) ){
							WebProject wp = (WebProject)r.getProject().getNature(WebProject.ID_NATURE);
							if(wp != null){
								webProject[0] = wp;
								return false;
							}
						}
						return true;
					}
				});
			}catch(CoreException e){
				WebpubUIPlugin.handleException(e);
			}
			if(webProject[0] != null){
				final Display disp = getSite().getShell().getDisplay();
				disp.asyncExec(new Runnable(){
					public void run(){
						if(disp != null){
							showLastLog(webProject[0]);
						}
					}
				});
			}
		}
		
	};
	
	IPropertyChangeListener fontChangeListener = new IPropertyChangeListener() {
		public void propertyChange(PropertyChangeEvent event) {
			if( event.getProperty().equals(KEY_CONSOLE_FONT) ){
				revalidateFont();
			}
		}
	};

	public File getBrowseLogFile() {
		return selectedFile;
	}
	
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		if(memento != null){
			customEncoding = memento.getString(CUSTOM_ENCODING);
		}
	}
	
	public String getText(){
		return consoleText != null ? consoleText.getText() : null;
	}

	public void createPartControl(Composite parent) {
		consoleText = new Text(parent, SWT.READ_ONLY | SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
		consoleText.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
		revalidateFont();
		
		IActionBars bars = getViewSite().getActionBars();
		bars.setGlobalActionHandler(ActionFactory.COPY.getId(), new Action() {
			public void run() {
				if( consoleText.isFocusControl()){
					consoleText.copy();
				}
			}
		});
		
		IToolBarManager toolBar = bars.getToolBarManager();
		// pin action role is lock the auto change on incremental build log.
		pinAction = new Action("Pin Console", Action.AS_CHECK_BOX){
			public void run(){
				if( pinAction.isChecked() ){
					ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
				}else{
					ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceListener);
				}
			}
		};
		pinAction.setToolTipText("Pin Console");
		pinAction.setImageDescriptor(WebpubUIPlugin.getImageDescriptor("clcl16/pin.gif"));
		pinAction.setDisabledImageDescriptor(WebpubUIPlugin.getImageDescriptor("dlcl16/pin.gif"));
		toolBar.add(pinAction);
		
		// pull down menu includes old log view.
		logSelectionAction = new Action("Display Selected Logs", Action.AS_DROP_DOWN_MENU){
			public void run(){
				Map logEntries = PublishLogStore.getLogEntries();
				if(!logEntries.isEmpty()){
					Date[] dates = (Date[])logEntries.keySet().toArray(new Date[logEntries.size()]);
					Arrays.sort(dates);
					Date lastDate = dates[dates.length-1];
					showLastLog( (WebProject)logEntries.get(lastDate) );
				}
			}
		};
		logSelectionAction.setToolTipText("Display Selected Logs");
		logSelectionAction.setMenuCreator(menuCreator);
		logSelectionAction.setImageDescriptor(WebpubUIPlugin.getImageDescriptor("clcl16/console_history.gif"));
		toolBar.add(logSelectionAction);
		
		JFaceResources.getFontRegistry().addListener(fontChangeListener);
		pinAction.run();
		
		browse(null, true);
	}
	
	public void browse(File file, boolean force){
		if(!force && (file == selectedFile || (file != null && file.equals(selectedFile))) ){
			return;
		}
		selectedFile = file;
		String text = null;
		try {
			if(file != null && file.canRead()){
				String encoding = (customEncoding != null) ? customEncoding : ResourcesPlugin.getPlugin().getPluginPreferences().getString(ResourcesPlugin.PREF_ENCODING);
				if(encoding.length() == 0) {
					encoding = null;
				}
				text = new TextReader(file.toURL(), encoding).joinedLines();
			}
		} catch (FileNotFoundException e) {
			WebpubUIPlugin.handleException(e);
		} catch (IOException e) {
			WebpubUIPlugin.handleException(e);
		}
		boolean enable = text != null;
		
		setTitle(enable ? toLabelString(file) : "A publish console is not an available.");
		consoleText.setText(enable ? text : "");
		consoleText.setBackground(consoleText.getDisplay().getSystemColor(enable ? SWT.COLOR_LIST_BACKGROUND : SWT.COLOR_WIDGET_BACKGROUND));
		consoleText.setEnabled(enable);
		getSite().getPage().bringToTop(PublishConsoleView.this);
	}

	void showLastLog(WebProject webProject){
		PublishLogStore logStore = webProject.getLogStore();
		logStore.deleteExcessFiles();
		
		Date[] dates = logStore.getLogDates();
		if(dates.length > 0){
			Date date = dates[dates.length-1];
			for(int i=0; i<reloadPrioritySuffix.length; i++){
				File f = logStore.getLogFile(date, reloadPrioritySuffix[i]);
				if(f != null){
					browse(f, false);
					break;
				}
			}
		}
	}
	
	String toLabelString(File f){
		Date date = PublishLogStore.dateFromFileName(f);
		String str = dateFormat.format(date);
		
		String fileName = f.toString();
		int dotIndex = fileName.lastIndexOf('.');
		if(dotIndex > 0){
			str += ' ' + f.toString().substring(dotIndex+1);
		}
		return str + ' ' + f.getParentFile().getName();
	}

	void revalidateFont(){
		consoleText.setFont( JFaceResources.getFontRegistry().get(KEY_CONSOLE_FONT) );
	}

	public void setFocus() {
	}
	
	public void saveState(IMemento memento) {
		memento.putString(CUSTOM_ENCODING, customEncoding);
		super.saveState(memento);
	}

	public void dispose() {
		consoleText = null;
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
		JFaceResources.getFontRegistry().removeListener(fontChangeListener);
		super.dispose();
	}

}
