#
# $Id$
# Copyright Narushima Hironori. All rights reserved.
#

require 'webpub/publisher'
require 'webpub/publish_description'

module Webpub

class PublishDescriptionFactory

	def initialize(web_project)
		@mappings = {}
		@web_project = web_project
		
		@error_out = STDOUT
	end

	attr_accessor :error_out

	def create(source)
		source = File.expand_path(source)
		unless /^#{@web_project.path}/i === source
			@error_out << "specify file is out of project file: #{source}\n"
			return nil
		end
		
		map, pub_to = choose_prop_from_file(source) if File.file?(source)
		unless map
			map, pub_to = choose_prop_from_dir(source)
			unless map
				@error_out << "can not publish: #{source}\n"
				return nil
			end
		end
		
		by = map['by']
		unless by
			@error_out << "can not get publiing code (#{map}): #{source}\n"
			return nil
		end
		
		desc = PublishDescription.new(by, @web_project)
		desc.publish_from = source
		desc.publish_to = pub_to
		map.each { |k, v|
			desc.arguments[k] = v unless %w!publish_to by!.include?(k)
		}
		if desc.ht_type?
			desc.ht_filters << AbsToRelFilter.new
		end
		
		desc
	end

	def choose_prop_from_file(file)
		content = nil
		IO.foreach(file) { |l|
			if content = /^<\?publish(.+)\?>$/.match(l)
				attrs = {}
				content[1].scan(/(.+?)\s*=\s*"(.+?)"/) { |m|
					attrs[m[0].strip] = m[1]
				}
				pub_to = file.sub(@web_project[:htsources_dir], @web_project[:publish_dir])
				return attrs, pub_to
			end
		}
		nil
	end

	def publish_mapping(mapfile)
		return @mappings[mapfile] if @mappings.key?(mapfile)
		
		root = REXML::Document.new(IO.read(mapfile)).root
		@mappings[mapfile] = root.get_elements('**/mapping').map do |e|
			pattern = e.attributes['pattern']
			attrs = {}
			e.elements['publish'].attributes.each { |k, v|
				attrs[k] = v
			}
			{ :pattern => pattern, :attributes => attrs }
		end
	end

	def choose_prop_from_dir(source)
		dir = File.dirname(source)
		in_htdir = /^#{@web_project[:htsources_dir]}/i === dir
		
		begin
			next unless File.exist?(mapfile = dir + '/.publish')
			
			publish_mapping(mapfile).each { |e|
				rel_path = source.sub(/^#{dir}\//, '')
				
				match_group = PublishDescriptionFactory.match_group(e[:pattern], rel_path)
				next unless match_group
				
				base_dir = ''
				if in_htdir
					base_dir = dir[@web_project[:htsources_dir].length .. -1]
				end
				
				pub_to = e[:attributes]['publish_to']
				
				if pub_to
					if /\/$/ === pub_to
						pub_to = pub_to + File.basename(source)
					end
					if pub_to.gsub!(/^\//, '')
						base_dir = ''
					end
					
					pub_to = PublishDescriptionFactory.make_path(pub_to, match_group)
				elsif in_htdir
					pub_to = rel_path
				else
					return e[:attributes], nil
				end
				
				pub_to = File.join( @web_project[:publish_dir], base_dir, pub_to)
				return e[:attributes], File.expand_path(pub_to)
			}
		end while dir.sub!(%r!/[^/]+?$!, '')
		
		if in_htdir
			pub_to = source.sub(/^#{@web_project[:htsources_dir]}/i, @web_project[:publish_dir])
			return {'by' => 'copy'}, pub_to
		end
		
		nil
	end

	def PublishDescriptionFactory.make_path(wildcard, results)
		path = []
		wc_segs = wildcard.split('/')
		
		len = wc_segs.size
		while path.size < len - results.size
			path << wc_segs.shift
		end
		
		if len < results.size
			results = results[-len, len]
		end
		
		while !wc_segs.empty? and !results.empty?
			res = results.shift
			res.shift
			
			p = wc_segs.shift
			while (a = p['*'] or p['?']) and !res.empty?
				if a
					p.sub!('*', res.shift)
				else
					p.sub!('?', res.shift[0, 1])
				end
			end
			path << p
		end
		
		path.join('/')
	end

	def PublishDescriptionFactory.match_group(pattern, path)
		results = []
		ptn_segs = pattern.split('/')
		path_segs = path.split('/')
		
		return nil if ptn_segs.reject { |seg| seg == '**' }.size > path_segs.size
		
		until path_segs.empty?
			path_seg = path_segs.shift
			ptn_seg = ptn_segs.shift
			if ptn_seg == '**'
				until path_segs.size == ptn_segs.size - 1
					results << [path_seg, nil]
					path_seg = path_segs.shift
				end
				ptn_seg = ptn_segs.shift
			end
			
			ptn = ptn_seg.gsub('.', '\.').gsub('?', '(.)').gsub('*', '(.*?)')
			if m = /^#{ptn}$/.match(path_seg)
				results << m.to_a
			else
				return nil
			end
		end
		
		results
	end
	
end

end
