/*
 * $Id: RubyExceptionTest.java,v 1.6 2004/06/07 06:46:49 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests;

import java.io.*;
import java.net.URL;
import java.util.Arrays;
import java.util.regex.Matcher;

import junit.framework.TestCase;

import com.narucy.webpub.core.*;

/**
 * 
 */
public class RubyExceptionTest extends TestCase {

	String[] lines;
	
	protected void setUp() throws Exception {
		lines = new TextReader(getClass().getResource("ruby_backtrace")).toLines();
	}
	
	protected void tearDown() throws Exception {
		lines = null;
	}
	
	public RubyExceptionTest(String name) {
		super(name);
	}
	
	public void testSyntaxErrorHandler() throws Exception {
		URL[] urls = new URL[]{
				getClass().getResource("ruby_backtrace_syntax_error"),
				getClass().getResource("ruby_backtrace_syntax_error-unc"),
		};
		String[] pathes = new String[2];
		for(int i=0; i<urls.length; i++){
			TextReader reader = new TextReader(urls[i]);
			RubyException[] ex = RubyException.createFromLines(reader);
			assertEquals(1, ex.length);
			
			RubyException.TraceItem info = ex[0].getSyntaxErrorInfo();
			pathes[i] = info.script;
			assertEquals("odd number list for Hash", info.occasionIn);
			assertEquals(5, info.lineNumber);
		}
		assertEquals("/eclipse-SDK-3.0M8-win32/eclipse/junit-workbench-workspace/ErrorHandling/scripts/syntax_error.rb", pathes[0]);
		assertEquals("C:/eclipse-SDK-3.0M8-win32/eclipse/junit-workbench-workspace/ErrorHandling/scripts/syntax_error.rb", pathes[1]);
		
		urls = new URL[]{
				getClass().getResource("ruby_backtrace_syntax_error2"),
				getClass().getResource("ruby_backtrace_syntax_error2-unc"),
		};
		for (int i = 0; i < urls.length; i++) {
			TextReader reader = new TextReader(urls[i]);
			RubyException[] ex = RubyException.createFromLines(reader);
			assertEquals(2, ex.length);
			RubyException.TraceItem info = ex[0].getSyntaxErrorInfo();
			pathes[i] = info.script;
			assertEquals(5, info.lineNumber);
			assertEquals("odd number list for Hash", info.occasionIn);
			
			assertEquals(1, ex[0].getTraceItems().length);
		}
		assertEquals("/eclipse/junit-workbench-workspace/ErrorHandling/scripts/syntax_error.rb", pathes[0]);
		assertEquals("C:/eclipse-SDK-3.0M8-win32/eclipse/junit-workbench-workspace/ErrorHandling/scripts/syntax_error.rb", pathes[1]);
	}

	public void testPattern(){
		Matcher m = RubyException.firstLinePattern.matcher(lines[0]);
		assertTrue( m.matches() );
		assertEquals("D:/mydoc/com.narucy.webpub.core/scripts/lib/webpub/amrita_publisher.rb", m.group(1));
		assertEquals("6", m.group(2));
		assertEquals("in `require': No such file to load -- amrita/template (LoadError)", m.group(3));
		
		m = RubyException.traceLinePattern.matcher(lines[1]);
		assertTrue( m.matches() );
		assertEquals("D:/mydoc/com.narucy.webpub.core/scripts/lib/webpub/amrita_publisher.rb", m.group(1));
		assertEquals("6", m.group(2));
		assertEquals( null, m.group(3));

		m = RubyException.traceLinePattern.matcher(lines[2]);
		assertTrue( m.matches() );
		assertEquals("D:/mydoc/com.narucy.webpub.core/scripts/lib/webpub/publish_manager.rb", m.group(1));
		assertEquals("70", m.group(2));
		assertEquals( "in `require'", m.group(3));
		
		m = RubyException.traceLinePattern.matcher(lines[6]);
		assertTrue( m.matches() );
		assertEquals("D:/mydoc/com.narucy.webpub.core/scripts/bin/pubtool.rb", m.group(1));
		assertEquals("42", m.group(2));
		assertEquals( null, m.group(3));
		
		for(int i=1; i<lines.length; i++){
			assertFalse( RubyException.firstLinePattern.matcher(lines[i]).matches() );
		}
	}

	public void testParse(){
		RubyException[] exceptions = RubyException.createFromLines( Arrays.asList(lines).iterator() );
		assertEquals(1, exceptions.length);
		
		RubyException e = exceptions[0];
		assertEquals(
			"in `require': No such file to load -- amrita/template (LoadError)",
			e.getMessage());

		assertEquals(6, e.getTraceItems()[0].lineNumber);
		assertEquals(7, e.getTraceItems().length);
	}
	
	public void testPrintBacktrace() throws Exception {
		RubyException ex = RubyException.createFromLines( Arrays.asList(lines).iterator() )[0];
		
		File tmpFile = WebpubTestPlugin.createTempFile("RubyBacktraceTest.printBackTrace");
		try{
			PrintWriter writer = new PrintWriter(new FileWriter(tmpFile));
			ex.printStackTrace(writer);
			writer.close();
			
			String[] ls = new TextReader(tmpFile.toURL()).toLines();
			assertTrue(Arrays.equals(ls, lines));
		}finally{
			tmpFile.delete();
		}
	}
	
	public void testErbException(){
		String[] lines = {
			"(erb):3:in `z': undefined local variable or method `zzz' for main:Object (NameError)",
			"\tfrom (erb):7:in `y'",
			"\tfrom (erb):11:in `x'",
			"\tfrom (erb):14",
		};
		
		RubyException[] errors = RubyException.createFromLines(Arrays.asList(lines).iterator());
		assertEquals(1, errors.length);
	}
	
}
