/*
 * $Id: WildcardTest.java,v 1.10 2004/06/07 06:46:49 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests;

import java.util.*;
import java.util.regex.Pattern;

import junit.framework.TestCase;

import org.eclipse.core.runtime.Path;

import com.narucy.webpub.core.Wildcard;

/**
 * 
 */
public class WildcardTest extends TestCase {

	static String[] sampleFilelist = {
		"001.java", "002.java", "003.java",
		"001.html", "002.html", "003.html",
		"101.java", "102.java", "103.java",

		"a/", "a/x/", "a/y/",
		"a/001.java", "a/002.java", "a/003.java",
		"a/001.html", "a/002.html", "a/003.html",
		"a/101.java", "a/102.java", "a/103.java",

		"a/b/", "a/b/x/", "a/b/y/",
		"a/b/001.java", "a/b/002.java", "a/b/003.java",
		"a/b/001.html", "a/b/002.html", "a/b/003.html",
		"a/b/101.java", "a/b/102.java", "a/b/103.java",

		"a/b/c/", "a/b/c/x/", "a/b/c/y/",
		"a/b/c/001.java", "a/b/c/002.java", "a/b/c/003.java",
		"a/b/c/001.html", "a/b/c/002.html", "a/b/c/003.html",
		"a/b/c/101.java", "a/b/c/102.java", "a/b/c/103.java",

		"d/b/", "d/b/x/", "d/b/y/",
		"d/b/001.erb", "d/b/002.erb", "d/b/003.erb",
		"d/b/001.rb", "d/b/002.rb", "d/b/003.rb",
		"d/b/101.erb", "d/b/102.erb", "d/b/103.erb",
	};

	public WildcardTest(String name) {
		super(name);
	}
	
	public void testMakePathByMatchResult(){
		// simple
		Wildcard wildcard = new Wildcard("a");
		assertEquals(
				new Path("a"),
				wildcard.makePathByMatchResult(new String[][]{}));

		assertEquals(
				new Path("a"),
				wildcard.makePathByMatchResult(new String[][]{
						{"x"},
				}));

		assertEquals(
				new Path("a"),
				wildcard.makePathByMatchResult(new String[][]{
						{"x"},
						{"y"},
				}));
		
		assertEquals(
				new Path("a"),
				wildcard.makePathByMatchResult(new String[][]{
						{"x", null},
						{"y", null},
						{"z"},
				}));
	
		// one question mark
		wildcard = new Wildcard("?.html");
		String[][] result = wildcard.matchWithGroup("a.html");
		assertEquals(
				new Path("a.html"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("a/*.html").matchWithGroup("a/bbb.html");
		assertEquals(
				new Path("b.html"),
				wildcard.makePathByMatchResult(result));

		
		result = new Wildcard("a/b/*/d/*").matchWithGroup("a/b/c/d/e");
		assertEquals(
				new Path("e.html"),
				wildcard.makePathByMatchResult(result));
		
		result = new Wildcard("**/*.html").matchWithGroup("a/b/c/d/e.html");
		assertEquals(
				new Path("e.html"),
				wildcard.makePathByMatchResult(result));

		// one asterisk mark
		wildcard = new Wildcard("*.html");
		result = wildcard.matchWithGroup("a.html");
		assertEquals(
				new Path("a.html"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("a/*.html").matchWithGroup("a/bbb.html");
		assertEquals(
				new Path("bbb.html"),
				wildcard.makePathByMatchResult(result));

		
		result = new Wildcard("a/b/*/d/*").matchWithGroup("a/b/c/d/eee");
		assertEquals(
				new Path("eee.html"),
				wildcard.makePathByMatchResult(result));
		
		
		// double question mark
		wildcard = new Wildcard("a?.?");
		result = wildcard.matchWithGroup("ax.x");
		assertEquals(
				new Path("ax.x"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("a/*.*").matchWithGroup("a/bbb.html");
		assertEquals(
				new Path("ab.h"),
				wildcard.makePathByMatchResult(result));

		
		result = new Wildcard("a/b/*/d/*").matchWithGroup("a/b/c/d/e");
		assertEquals(
				new Path("ae.?"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("**/?.*").matchWithGroup("a/b/c/d/e.html");
		assertEquals(
				new Path("ae.h"),
				wildcard.makePathByMatchResult(result));

		// triple asterisk mark
		wildcard = new Wildcard("*.*.*");
		result = wildcard.matchWithGroup("a.bb.ccc.dddd");
		assertEquals(
				new Path("a.bb.ccc.dddd"),
				wildcard.makePathByMatchResult(result));

		
		result = new Wildcard("*/*??.html").matchWithGroup("a/bbb01.html");
		assertEquals(
				new Path("bbb.0.1"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("a/b/*/d/*_*_*").matchWithGroup("a/b/c/d/ee_ff_gg");
		assertEquals(
				new Path("ee.ff.gg"),
				wildcard.makePathByMatchResult(result));

	}
	
	public void testMakePathByMatchResult2(){
		// dir match
		Wildcard wildcard = new Wildcard("*/*/*.html");
		String[][] result = new Wildcard("*.html").matchWithGroup("bbb.html");
		assertEquals(
				new Wildcard("*/*/bbb.html"),
				wildcard.makePathByMatchResult(result)); 

		result = new Wildcard("*/*.html").matchWithGroup("aaa/bbb.html");
		assertEquals(
				new Wildcard("*/aaa/bbb.html"),
				wildcard.makePathByMatchResult(result));
		
		result = new Wildcard("*/*/*.html").matchWithGroup("aaa/bbb/ccc.html");
		assertEquals(
				new Path("aaa/bbb/ccc.html"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("*/*/*.html").matchWithGroup("aaa/bbb/ccc.html");
		assertEquals(
				new Path("aaa/bbb/ccc.html"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("**/*.html").matchWithGroup("aaa/bbb/ccc.html");
		assertEquals(new String[][]{
				{"aaa", null},
				{"bbb", null},
				{"ccc.html", "ccc"},
		}, result);
		assertEquals(
				new Path("aaa/bbb/ccc.html"),
				wildcard.makePathByMatchResult(result));

		result = new Wildcard("*/**/*.html").matchWithGroup("aaa/bbb/ccc.html");
		assertEquals(new String[][]{
				{"aaa", "aaa"},
				{"bbb", null},
				{"ccc.html", "ccc"},
		}, result);
		assertEquals(
				new Path("aaa/bbb/ccc.html"),
				wildcard.makePathByMatchResult(result));

		// more segument match
		wildcard = new Wildcard("*/*/*/*/*/*.html");
		result = new Wildcard("*/*.html").matchWithGroup("aaa/bbb.html");
		assertEquals(
				new Wildcard("*/*/*/*/aaa/bbb.html"),
				wildcard.makePathByMatchResult(result));
		
		result = new Wildcard("*/*/*/*/*.html").matchWithGroup("aaa/bbb/ccc/ddd/eee.html");
		assertEquals(
				new Path("*/aaa/bbb/ccc/ddd/eee.html"),
				wildcard.makePathByMatchResult(result));

		// double asterisk match
		wildcard = new Wildcard("**/z.html");
		result = new Wildcard("**/*.html").matchWithGroup("a/b/c/index.html");
		
		assertEquals(
				new Path("a/b/c/z.html"),
				wildcard.makePathByMatchResult(result));
	}
	
	public void testWithGroup(){
		Map entry = matchWithGroup("a/*/*.java");
		Object[] es = entry.keySet().toArray();
		
		String[][] e = (String[][])entry.get(es[0]);
		assertEquals("001", e[2][1]);
		
		e = (String[][])entry.get(es[1]);
		assertEquals("002", e[2][1]);
		
		e = (String[][])entry.get(es[2]);
		assertEquals("003", e[2][1]);
		
		e = (String[][])entry.get(es[3]);
		assertEquals("101", e[2][1]);
		
		e = (String[][])entry.get(es[4]);
		assertEquals("102", e[2][1]);
		
		assertEquals(6, es.length);
		
		entry = matchWithGroup("**/b/10?.*");
		es = entry.keySet().toArray(); 
		Arrays.sort(es);
		assertTrue(Arrays.equals(es, new Object[]{
			"a/b/101.java",
			"a/b/102.java",
			"a/b/103.java",
			"d/b/101.erb",
			"d/b/102.erb",
			"d/b/103.erb",
		}));
		for (int i = 0; i < es.length; i++) {
			e = (String[][])entry.get(es[i]);
			// multi pattern match represented by second index is null.
			assertNull(e[0][1]);
		}
	}
	
	SortedMap matchWithGroup(String pattern){
		TreeMap matches = new TreeMap();
		Wildcard wildcard = new Wildcard(pattern);
		for (int i = 0; i < sampleFilelist.length; i++) {
			String p = sampleFilelist[i];
			String[][] group = wildcard.matchWithGroup(p);
			if(group != null){
				matches.put(p, group);
			}
		}
		return matches;
	}
	
	public void testDirMatch(){
		Map matchEntries = matchWithGroup("**/y/");
		
		Object[] ms = matchEntries.keySet().toArray();
		assertTrue(Arrays.equals(ms, new Object[]{
			"a/b/c/y/",
			"a/b/y/",
			"a/y/",
			"d/b/y/",
		}));
		String[][] group = (String[][])matchEntries.get("a/b/c/y/");
		assertEquals(new String[][]{
				{"a", null},
				{"b", null},
				{"c", null},
				{"y"},
		}, group);
		
		matchEntries = matchWithGroup("a/**/x/");
		assertTrue(Arrays.equals( matchEntries.keySet().toArray(), new Object[]{
			"a/b/c/x/",
			"a/b/x/",
		}));
		
		group = (String[][])matchEntries.get("a/b/c/x/");
		assertEquals(new String[][]{
				{"a"},
				{"b", null},
				{"c", null},
				{"x"},
		}, group);
		
	}
	
	void assertEquals(Object[][] a, Object[][] b){
		assertEquals(a.length, b.length);
		for (int i = 0; i < a.length; i++) {
			assertEquals(a[i].length, b[i].length);
			assertTrue( Arrays.equals(a[i], b[i]) );
		}
	}
	
	public void testDoubleAsterisk(){
		String[] ms = matches("**/b/10?.*");
		assertEquals(6, ms.length );
		assertTrue(Arrays.equals(new String[]{
				"a/b/101.java",
				"a/b/102.java",
				"a/b/103.java",
				"d/b/101.erb",
				"d/b/102.erb",
				"d/b/103.erb",
		}, ms));

		ms = matches("**/*.java");
		for (int i = 0; i < ms.length; i++) {
			assertTrue( Pattern.matches( ".+\\.java$", ms[i]) );
		}
		assertEquals(18, ms.length );
		
		ms = matches("a/**/*.java");
		assertEquals(12, ms.length);
		
		ms = matches("**/c/*.html");
		for (int i = 0; i < ms.length; i++) {
			assertTrue( Pattern.matches( "a/b/c/00\\d.html$", ms[i]) );
		}
		assertEquals(3, ms.length);

		ms = matches("**/b/10?.j*");
		for (int i = 0; i < ms.length; i++) {
			assertTrue( Pattern.matches( "a/b/10\\d\\.java$", ms[i]) );
		}
		assertEquals( 3, ms.length );
		ms = matches("**/b/c/?03.java");
		assertTrue(Arrays.equals(ms, new Object[]{
				"a/b/c/003.java",
				"a/b/c/103.java",
		}));
	}
	
	void p(Object o){
		System.out.println(o);
	}
	
	void p(Object[] ms){
		for (int i = 0; i < ms.length; i++) {
			p(ms[i]);
		}
	}

	public void testAsteriskMatch(){
		String[] ms = matches("a/*.java");
		assertEquals(6, ms.length );
		assertEquals("a/001.java", ms[0]);
		assertEquals("a/103.java", ms[5]);

		ms = matches("a/001.*");
		assertEquals(2, ms.length );
		assertEquals("a/001.java", ms[0]);
		assertEquals("a/001.html", ms[1]);
		
		ms = matches("a/?01.*");
		assertEquals(3, ms.length );
		assertEquals("a/001.java", ms[0]);
		assertEquals("a/001.html", ms[1]);
		assertEquals("a/101.java", ms[2]);

		ms = matches("a/b/*.java");
		assertEquals( 6, ms.length );
		assertEquals("a/b/001.java", ms[0]);
		assertEquals("a/b/103.java", ms[5]);
	}
	
	public void testFileMatch(){
		assertEquals( 6, matches("*.java").length );
		assertEquals( 3, matches("*.html").length );

		String[] ms = matches("10?.java");
		assertEquals( "101.java", ms[0]);
		assertEquals( "102.java", ms[1]);
		assertEquals( "103.java", ms[2]);
		assertEquals( 3, ms.length );
		
		ms = matches("?0?.java");
		assertEquals( 6, ms.length );
		
		ms = matches("?02.*");
		assertEquals( "002.java", ms[0]);
		assertEquals( "002.html", ms[1]);
		assertEquals( "102.java", ms[2]);
		
		assertEquals( 3, ms.length );
	}
	
	public void testIllegalPattern(){
		Exception exception = null;
		try{
			new Wildcard("**/**/foo");
		}catch(Exception e){
			exception = e;
		}
		assertNotNull(exception);
		
		exception = null;
		try{
			new Wildcard("foo**");
		}catch(Exception e){
			exception = e;
		}
		assertNotNull(exception);
	}

	String[] matches(String ptn){
		Wildcard wildcard = new Wildcard(ptn);
		ArrayList list = new ArrayList();
		for (int i = 0; i < sampleFilelist.length; i++) {
			String f = sampleFilelist[i];
			if( wildcard.match(f) ){
				list.add(f);
			}
		}
		return (String[])list.toArray(new String[list.size()]);
	}

}
