/*
 * $Id: ZippedPresetTest.java,v 1.5 2004/06/07 06:46:50 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests.presets;

import java.util.Map;
import java.util.zip.ZipEntry;

import junit.framework.TestCase;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.Path;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.preset.ZippedPreset;
import com.narucy.webpub.core.tests.WebpubTestPlugin;

/**
 * Tests distribute zipped web project preseet.
 */
public class ZippedPresetTest extends TestCase {
	
	IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();

	public void testDistribute() throws Exception {
		ZippedPreset preset = new ZippedPreset(
				getClass().getResource("PresetTest-EUC-JP.zip"),
				"ZipWebProject",
				"For testing zipped project");
		
		// tests getDistributeEntries
		IProject proj = root.getProject("a");
		try{
			WebProject wp = WebProject.distributeNewProject(proj, "x_ht_sources", "x_scripts", "x_public_html", null);
			Map distTargets = preset.getDistributeEntries(wp);
			ZipEntry[] entries = preset.getDistributeZipEntries(distTargets);
			assertEquals(4, entries.length);
			
			assertEquals("PresetTest-EUC-JP/ht_sources/a.html", entries[0].getName());
			assertEquals("PresetTest-EUC-JP/ht_sources/b.htm", entries[1].getName());
			assertEquals("PresetTest-EUC-JP/ht_sources/c.erb", entries[2].getName());
			assertEquals("PresetTest-EUC-JP/ht_sources/x.xhtml", entries[3].getName());
			
			assertEquals(proj.getFile("x_ht_sources/a.html"), distTargets.get(entries[0]));
			assertEquals(proj.getFile("x_ht_sources/b.htm"), distTargets.get(entries[1]));
			assertEquals(proj.getFile("x_ht_sources/c.erb"), distTargets.get(entries[2]));
			assertEquals(proj.getFile("x_ht_sources/x.xhtml"), distTargets.get(entries[3]));
		}finally{
			WebpubTestPlugin.forceDelete(proj);
		}
		
		proj = root.getProject("b");
		try{
			// tests distribute
			preset.distribute(proj, "x", "y", "z", null);
			
			assertEquals(0, proj.findMarkers(null, true, IResource.DEPTH_INFINITE).length);
			
			assertTrue( proj.getFolder("x").exists() );
			assertTrue( proj.getFile("x/a.html").exists() );
			assertTrue( proj.getFile("x/b.htm").exists() );
			assertTrue( proj.getFile("x/c.erb").exists() );
			assertTrue( proj.getFile("x/x.xhtml").exists() );
			
			assertTrue( proj.getFolder("y").exists() );
			
			assertTrue( proj.getFolder("z").exists() );
			assertTrue( proj.getFile("z/a.html").exists() );
			assertTrue( proj.getFile("z/b.htm").exists() );
			assertTrue( proj.getFile("z/c.erb").exists() );
			assertTrue( proj.getFile("z/x.xhtml").exists() );
			
			assertEquals("EUC-JP", ((WebProject)proj.getNature(WebProject.ID_NATURE)).getString(WebProject.ENCODING));
		}finally{
			WebpubTestPlugin.forceDelete(proj);
		}
	}
	
	public void testDistributeScript() throws Exception {
		ZippedPreset preset = new ZippedPreset(
				getClass().getResource("PublishLogStorePresetTest.zip"),
				"Publish Log Store",
				"test publish log store in distribute web project as preset");
		
		IProject proj = root.getProject("c");
		try{
			preset.distribute(proj, "x", "y", "z", null);
			
			assertTrue( proj.getFile("y/am_page.rb").exists() );
			
			assertTrue( proj.getFile("z/am_page.html").exists() );
			assertTrue( proj.getFile("z/erb_page.html").exists() );
			assertEquals("UTF-8" ,((WebProject)proj.getNature(WebProject.ID_NATURE)).getString(WebProject.ENCODING));
		}finally{
			WebpubTestPlugin.forceDelete(proj);
		}
	}

	/**
	 * Tests require folder entry included.
	 */
	public void testRequireFolder() throws Exception {
		ZippedPreset preset = new ZippedPreset(
				getClass().getResource("PresetTest-Shift_JIS.zip"),
				"Publish Log Store",
				"test publish log store in distribute web project as preset");
		
		IProject proj = root.getProject("testProject");
		try{
			WebProject wp = preset.distribute(proj, "ht_sources", "scripts", "public_html", null);
			
			// test distribute require folders
			assertTrue( proj.getFolder("a").exists());
			assertTrue( proj.getFolder("a/aa").exists());
			assertTrue( proj.getFile("a/aa/aax").exists());
			
			assertFalse( proj.getFolder("a/ab").exists());
			assertFalse( proj.getFolder("a/ac").exists());

			assertTrue( proj.getFolder("b").exists());
			assertTrue( proj.getFolder("b/ba").exists());
			assertTrue( proj.getFile("b/ba/bax").exists());
			assertFalse( proj.getFolder("b/ba/baa").exists());
			
			// test distribute script folders
			IContainer scriptsFolder = wp.getFolder(WebProject.SCRIPTS_FOLDER);
			assertTrue( scriptsFolder.getFile(new Path("script_a/a_x.rb")).exists() );
			assertTrue( scriptsFolder.getFile(new Path("script_a/script_aa/aa_x.rb")).exists() );
			assertTrue( scriptsFolder.getFile(new Path("script_a/script_aa/script_aaa/aaa_x.rb")).exists() );
			assertTrue( scriptsFolder.getFile(new Path("script_a/script_aa/script_aaa/aaa_y.rb")).exists() );
			
			assertTrue( scriptsFolder.getFile(new Path("script_a/script_ab/ab_x.rb")).exists() );
			assertFalse( scriptsFolder.getFolder(new Path("script_a/script_ab/script_aba")).exists() );
			
			// test ht sources distribute 
			IContainer sourcesFolder = wp.getFolder(WebProject.HTSOURCES_FOLDER);
			assertTrue( sourcesFolder.getFile(new Path("x.erb")).exists() );
			assertTrue( sourcesFolder.getFile(new Path("a/x.htm")).exists() );
			assertTrue( sourcesFolder.getFile(new Path("a/aa/x.html")).exists() );
			assertTrue( sourcesFolder.getFile(new Path("a/ab/x.xhtml")).exists() );
			assertFalse( sourcesFolder.getFolder(new Path("a/ac")).exists() );
		}finally{
			WebpubTestPlugin.forceDelete(proj);
		}
	}
}
