/*
 * $Id: SourceFileSeekerTest.java,v 1.16 2004/06/10 03:17:25 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests.publish;

import java.net.URL;

import junit.framework.TestCase;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.publish.SourceFileSeeker;
import com.narucy.webpub.core.tests.WebpubTestPlugin;

/**
 * SourceFileSeekerTest
 */
public class SourceFileSeekerTest extends TestCase {

	WebProject webProj;
	
	public SourceFileSeekerTest(String name) {
		super(name);
	}
	
	IFile seek(String url) throws CoreException{
		return SourceFileSeeker.findSource(url);
	}

	IFile seek(IFile f) throws Exception{
		return SourceFileSeeker.findSource(f.getLocation().toFile().toURL());
	}
	
	/**
	 * http://aaa.com
	 * http://bbb.com/bbb
	 * http://ccc.gr.jp/ccc/ccc/ccc/
	 * file:/c:/
	 * file:/c:/bbb
	 * file:/c:/ccc/ccc/ccc/
	 * file:/ddd/ddd/ddd
	 */
	public void testMappedURL() throws Exception {
		webProj = WebpubTestPlugin.createTestProject("test_scripts/webpub/wildcard_test_project");
		
		String[] urls = webProj.getArray(WebProject.MAPPED_URL);
		assertEquals(7, urls.length);
		
		String pubPath = webProj.getFolder(WebProject.PUBLISH_FOLDER).getLocation().toOSString();
		
		IProject proj = webProj.getProject();
		IFile a = proj.getFile("ht_sources/a.html");
		IFile b = proj.getFile("ht_sources/b.html");
		IFile c = proj.getFile("ht_sources/c.html");
		IFile xa = proj.getFile("ht_sources/x/a.html");
		IFile xyb = proj.getFile("ht_sources/x/y/b.html");
		IFile xyzc = proj.getFile("ht_sources/x/y/z/c.html");
		
		if(System.getProperty("file.separator").equals("\\")){
			// System.out.println("test for UNC path");
			assertEquals(a, seek(pubPath + "\\a.html") );
			assertEquals(b, seek(pubPath + "\\b.html") );
			assertEquals(c, seek(pubPath + "\\c.html") );
			assertEquals(xa, seek(pubPath + "\\x\\a.html") );
			assertEquals(xyb, seek(pubPath + "\\x\\y\\b.html") );
			assertEquals(xyzc, seek(pubPath + "\\x\\y\\z\\c.html") );
		}

		assertEquals(a, seek("http://aaa.com/a.html?a=b;c=d&e=f;g=h") );
		assertEquals(b, seek("http://aaa.com/b.html?") );
		assertEquals(c, seek("http://aaa.com/c.html?a") );
		assertEquals(xa, seek("http://aaa.com/x/a.html?xyz") );
		assertEquals(xyb, seek("http://aaa.com/x/y/b.html?a=b&c") );
		assertEquals(xyzc, seek("http://aaa.com/x/y/z/c.html") );
		
		assertEquals(a, seek("http://bbb.com/bbb/a.html"));
		assertEquals(xyzc, seek("http://bbb.com/bbb/x/y/z/c.html") );
		assertEquals(xa, seek("http://ccc.gr.jp/ccc/ccc/ccc/x/a.html"));
		
		assertEquals(a, seek("file:/c:/a.html") );
		assertEquals(b, seek("file:/c:/b.html") );
		assertEquals(c, seek("file:/c:/c.html") );

		assertEquals(xa, seek("file:/c:/x/a.html") );
		assertEquals(xyb, seek("file:/c:/x/y/b.html") );
		assertEquals(xyzc, seek("file:/c:/x/y/z/c.html") );

		assertEquals(a, seek("file:/ddd/ddd/ddd/a.html") );
		assertEquals(c, seek("file:/ddd/ddd/ddd/c.html") );
		assertEquals(xa, seek("file:/ddd/ddd/ddd/x/a.html") );
		assertEquals(xyzc, seek("file:/ddd/ddd/ddd/x/y/z/c.html") );
	}
	
	public void testGetPublishFrom() throws Exception {
		webProj = WebpubTestPlugin.createTestProject("test_scripts/webpub/test_workspace/web_project");
		IProject project = webProj.getProject();
		// test findPublishDescriptorFiles
		URL url = project.getFile("public_html/home.html").getLocation().toFile().toURL();
		
		// search simple
		assertEquals(
				project.getFile("ht_sources/home.html"),
				seek(project.getFile("public_html/home.html")));
		
		// search form wildcard published file
		assertEquals(
				project.getFile("rd/a.rd"),
				seek(project.getFile("public_html/pub_rd/a.html")));
		
		assertEquals(
				project.getFile("rd/with_tmpl/c.rd"),
				seek(project.getFile("public_html/pub_rd/amrita_wz_rd.html")));

		assertEquals(
				project.getFile("erbs/b.erb"),
				seek(project.getFile("public_html/erbs/b.html")));

		assertEquals(
				project.getFile("erbs/c/a/c-a-a.erb"),
				seek(project.getFile("public_html/erbs/c-a-a.html")));
		
		assertEquals(
				project.getFile("ht_sources/folder/subfolder/subsubfolder/s-1.html"),
				seek(project.getFile("public_html/folder/subfolder/subsubfolder/s-1.html")));

		assertEquals(
				project.getFile("ht_sources/folder/subfolder/subsubfolder/t-1.html"),
				seek(project.getFile("public_html/foo_folder/t-1.html")));
		
		assertEquals(
				project.getFile("ht_sources/folder/subfolder/subsubfolder/u-1.html"),
				seek(project.getFile("public_html/uuu.html")));

		assertEquals(
				project.getFile("ht_sources/folder/subfolder/subsubfolder/1-1-1-3.html"),
				seek(project.getFile("public_html/folder/3.html")));

		assertEquals(
				project.getFile("ht_sources/folder/subfolder/subsubfolder/1-1-1-2.html"),
				seek(project.getFile("public_html/folder/1/1/1/2.html")));

		assertEquals(
				project.getFile("ht_sources/folder/subfolder/subsubfolder/1-1-1-1.html"),
				seek(project.getFile("public_html/folder/subfolder/subsubfolder/1-1-1-1.html")));

		// wild card in root
		assertEquals(
				project.getFile("ht_sources/home.png"),
				seek(project.getFile("public_html/home.gif")));
	}

	protected void tearDown() throws Exception {
		if(webProj != null){
			WebpubTestPlugin.forceDelete(webProj.getProject());
			webProj = null;
		}
	}
	

}
