/*
 * $Id: TemplateManagerTest.java,v 1.9 2004/06/07 06:46:50 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests.template;

import java.io.*;
import java.util.Arrays;

import junit.framework.TestCase;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;

import com.narucy.webpub.core.*;
import com.narucy.webpub.core.template.*;
import com.narucy.webpub.core.tests.WebpubTestPlugin;

/**
 * 
 */
public class TemplateManagerTest extends TestCase {

	TemplateParser parser = new TemplateParser();

	public TemplateManagerTest(String arg0) {
		super(arg0);
	}
	
	WebProject[] webProjects;

	WebProject prepare(String testProjPath) throws Exception {
		WebProject wp = WebpubTestPlugin.createTestProject(testProjPath);
		rebuild(wp);
		wp.getTemplateManager().researchAll(null);
		return wp;
	}
	
	void rebuild(WebProject wp) throws CoreException {
		IProject proj = wp.getProject();
		proj.refreshLocal(IResource.DEPTH_INFINITE, null);		
		proj.build(IncrementalProjectBuilder.INCREMENTAL_BUILD, null);
	}
	
	protected void tearDown() throws Exception {
		for (int i=0; webProjects != null && i<webProjects.length; i++) {
			WebpubTestPlugin.forceDelete(webProjects[i].getProject());
		}
		webProjects = null;
	}
	
	public void testJapaneseSupport() throws Exception {
		webProjects = new WebProject[]{
			prepare("test_projects/TemplateTest-EUC-JP"),
			prepare("test_projects/TemplateTest-Shift_JIS"),
			prepare("test_projects/TemplateTest-UTF-8"),
		};
		
		String[] filePathes = {
			"ht_sources/a.html",
			"ht_sources/b.htm",
			"ht_sources/c.erb",
			"ht_sources/x.xhtml",
		};
		
		Template[][][] tmpls = new Template[3][4][0];
		for (int i = 0; i < webProjects.length; i++) {
			WebProject wp = webProjects[i];
			IProject proj = wp.getProject();
			
			for (int j = 0; j < filePathes.length; j++) {
				String p = filePathes[j];
				IFile f = proj.getFile(p);
				tmpls[i][j] = wp.getTemplateManager().getTemplates(f);
			}
		}
		
		assertEquals(1, tmpls[0][0].length);
		assertEquals(2, tmpls[0][1].length);
		assertEquals(3, tmpls[0][2].length);
		assertEquals(0, tmpls[0][3].length);
		
		assertEquals(1, tmpls[1][0].length);
		assertEquals(2, tmpls[1][1].length);
		assertEquals(3, tmpls[1][2].length);
		assertEquals(0, tmpls[1][3].length);
		
		assertEquals(1, tmpls[2][0].length);
		assertEquals(2, tmpls[2][1].length);
		assertEquals(3, tmpls[2][2].length);
		assertEquals(0, tmpls[2][3].length);
		
		assertEquals("a", tmpls[0][0][0].getName());
		assertEquals(tmpls[0][0][0], tmpls[1][0][0]);
		assertEquals(tmpls[0][0][0], tmpls[2][0][0]);

		assertEquals("b", tmpls[0][1][1].getName());
		assertEquals(tmpls[0][1][1], tmpls[1][1][1]);
		assertEquals(tmpls[0][1][1], tmpls[2][1][1]);

		assertEquals("c", tmpls[0][2][1].getName());
		assertEquals(tmpls[0][2][1], tmpls[1][2][1]);
		assertEquals(tmpls[0][2][1], tmpls[2][2][1]);
	}

	public void testSynchronizeTemplate() throws Exception {
		webProjects = new WebProject[]{
			prepare("test_projects/TemplateTest-UTF-8"),
			prepare("test_projects/TemplateTest-EUC-JP"),
			prepare("test_projects/TemplateTest-Shift_JIS"),
		};
		
		assertEquals("UTF-8", webProjects[0].getString(WebProject.ENCODING));
		assertEquals("EUC-JP", webProjects[1].getString(WebProject.ENCODING));
		assertEquals("Shift_JIS", webProjects[2].getString(WebProject.ENCODING));
		
		for (int i = 0; i < webProjects.length; i++) {
			WebProject wp = webProjects[i];
			assertTrue(Arrays.equals(
				new String[]{"html", "htm", "erb"},
				wp.getArray(WebProject.HT_EXTENSIONS) ));
			
			IProject proj = wp.getProject();
			
			IFile aFile = proj.getFile("ht_sources/a.html");
			IFile bFile = proj.getFile("ht_sources/b.htm");
			IFile cFile = proj.getFile("ht_sources/c.erb");
			
			TemplateManager tmplMgr = wp.getTemplateManager();
	
			assertEquals( tmplMgr.getTemplates(aFile)[0], tmplMgr.getGlobalTemplate("a") );
			assertEquals( tmplMgr.getTemplates(bFile)[1], tmplMgr.getGlobalTemplate("b") );
			assertEquals( tmplMgr.getTemplates(cFile)[1], tmplMgr.getGlobalTemplate("c") );
			
			assertTrue( read(aFile).indexOf("A_") != -1 );
			
			replaceLines(aFile, "A_", "X_", 1);
			assertTrue( read(aFile).indexOf("A_") == -1 );
			assertTrue( read(bFile).indexOf("A_") == -1 );
			assertTrue( read(cFile).indexOf("A_") == -1 );
			assertTrue( read(aFile).indexOf("X_") != -1 );
			assertTrue( read(bFile).indexOf("X_") != -1 );
			assertTrue( read(cFile).indexOf("X_") != -1 );
			
			replaceLines(bFile, "I_", "Y_", 1);
			assertTrue( read(aFile).indexOf("Y_") == -1 ); // none define b template
			
			assertTrue( read(bFile).indexOf("I_") == -1 );
			assertTrue( read(cFile).indexOf("I_") == -1 );

			assertTrue( read(bFile).indexOf("Y_") != -1 );
			assertTrue( read(cFile).indexOf("Y_") != -1 );
			
			replaceLines(cFile, "U_", "Z_", 1);
			
			assertTrue( read(aFile).indexOf("Z_") == -1 ); // none define c template
			assertTrue( read(bFile).indexOf("Z_") == -1 ); // none define c template
	
			assertTrue( read(cFile).indexOf("U_") == -1);
			assertTrue( read(cFile).indexOf("Z_") != -1);
		}
	}
	
	void replaceLines(IFile f, String from, String to) throws Exception {
		replaceLines(f, from, to, Integer.MAX_VALUE);
	}

	void replaceLines(IFile f, String from, String to, int replaceCount) throws Exception {
		WebProject wp = (WebProject)f.getProject().getNature(WebProject.ID_NATURE);
		
		String encoding = wp.getString(WebProject.ENCODING);
		String[] ls = new TextReader(f.getContents(), encoding).toLines();
		
		PrintWriter writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream(f.getLocation().toFile()), encoding));
		for (int i = 0, rc = 0; i < ls.length; i++) {
			String l = ls[i];
			if( l.indexOf(from) != -1 && rc++ < replaceCount){
				l = l.replaceAll(from, to);
			}
			writer.println(l);
		}
		writer.close();
		rebuild(wp);
	}

	String read(IFile f) throws IOException, CoreException{
		return new TextReader(f.getContents()).joinedLines();
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}

}
