#
# $Id$
# Copyright Narushima Hironori. All rights reserved.
#

require 'test/unit'
require 'webpub/publish_description_factory'

=begin

PublishDescriptionFactory main role is create PublishDescription object for
specified file that work is a complicate.

(1) When publish target file type is html, seek publish definication property
    in itself file lines. <?publish by="" ...?>
    (html type that is decided to refer file extension default extension are
    html, htm or erb)

(2) Next, find ".publish" file in itself included folder or upon folder as fer
    as project folder. ".publish" defines publish way for entries.
    PublishDescriptionFactory found publishing method in ".publish" file,
    create PublishDescription object in referer publish entry.

(3) If not found publish method entry in ".publish" file, publish target file
    included in html source folder. copy publish folder to html source
    with keep folder constituent.

If not found publish method entry and not included in html source folder for
specified publish target file, nothing to do. return nil.

=end

class PublisherDescriptionFactoryTest < Test::Unit::TestCase

	include Webpub

	def setup
		@workspace = Workspace.instance
		@workspace.path = File.dirname(__FILE__) + '/test_workspace'

		@proj_path = @workspace.path + '/web_project'
		@htsrc_path = @proj_path + '/ht_sources'
		@pubdir_path = @proj_path + '/public_html'
		
		@web_project = @workspace.web_project('web_project')
		@fac = PublishDescriptionFactory.new(@web_project)
	end
	
	def test_make_path_by_moresegments
		path = PublishDescriptionFactory.make_path('a/b/c/d.html', [])
		assert_equal('a/b/c/d.html', path)

		path = PublishDescriptionFactory.make_path('a/b/c/d.html', [['x.html']])
		assert_equal('a/b/c/d.html', path)

		path = PublishDescriptionFactory.make_path('a/b/c/?.html', [['x.html', 'x']])
		assert_equal('a/b/c/x.html', path)

		path = PublishDescriptionFactory.make_path('*/*/*/*.html', [
			['aaa', 'aaa'],
			['bbb', 'bbb'],
			['ccc', 'ccc'],
			['xxx.html', 'xxx'],
		])
		assert_equal('aaa/bbb/ccc/xxx.html', path)

		path = PublishDescriptionFactory.make_path('aaa/*/*/*.html', [
			['ccc', 'ccc'],
			['xxx', 'xxx'],
		])
		assert_equal('aaa/*/ccc/xxx.html', path)

		path = PublishDescriptionFactory.make_path('?/?/?/?.html', [
			['aaa', 'aaa'],
			['bbb', 'bbb'],
			['ccc', 'ccc'],
			['xxx.html', 'xxx'],
		])
		assert_equal('a/b/c/x.html', path)

		path = PublishDescriptionFactory.make_path('?/?/?/?.html', [
			['ccc', 'ccc'],
			['xxx', 'xxx'],
		])
		assert_equal('?/?/c/x.html', path)

		path = PublishDescriptionFactory.make_path('?/?/?/?.html', [
			['ccc', 'ccc'],
			['xxx', 'xxx'],
		])
		assert_equal('?/?/c/x.html', path)

		path = PublishDescriptionFactory.make_path('aaa/bbb/?-?-?/*.?.html', [
			['1.2.3', '1', '2', '3'],
			['xxyyzz', 'xx', 'yy', 'zz'],
		])
		assert_equal('aaa/bbb/1-2-3/xx.y.html', path)

		path = PublishDescriptionFactory.make_path('aaa/bbb/?-?-?/*.?.html', [
			['x', 'x'],
			['x', 'x'],
			['x', 'x'],
			['1.2.3', '1', '2', '3'],
			['xxyyzz', 'xx', 'yy', 'zz'],
		])
		assert_equal('aaa/bbb/1-2-3/xx.y.html', path)
	end

	def test_make_path_by_singlesegment
		path = PublishDescriptionFactory.make_path('a.html', [[]])
		assert_equal('a.html', path)

		path = PublishDescriptionFactory.make_path('a.html', [['x.html']])
		assert_equal('a.html', path)

		path = PublishDescriptionFactory.make_path('a.html', [['x.html', 'x']])
		assert_equal('a.html', path)

		path = PublishDescriptionFactory.make_path('?.html', [['x.html', 'x']])
		assert_equal('x.html', path)

		path = PublishDescriptionFactory.make_path('???.html', [['xxx.html', 'x', 'x', 'x']])
		assert_equal('xxx.html', path)
		
		path = PublishDescriptionFactory.make_path('*.html', [['xxx.html', 'xxx']])
		assert_equal('xxx.html', path)
	end

	def test_in_htsource
		desc = @fac.create(@htsrc_path + '/home.png')
		assert_equal('copy', desc.by)
		
		desc = @fac.create(@htsrc_path + '/folder/subfolder/subsubfolder/c.png')
		assert_equal('copy', desc.by)
		assert_equal(@htsrc_path + '/folder/subfolder/subsubfolder/c.png', desc.publish_from)
		assert_equal(@pubdir_path + '/folder/subfolder/subsubfolder/c.png', desc.publish_to)
	end

	def test_external_dir
		desc = @fac.create(@proj_path + '/erbs/a.erb')
		assert_equal( @pubdir_path + '/erbs/a.html', desc.publish_to)
		
		desc = @fac.create(@proj_path + '/erbs/c/a/c-a-a.erb')
		assert_equal( @pubdir_path + '/erbs/c-a-a.html', desc.publish_to)
		
		desc = @fac.create(@proj_path + '/rd/with_tmpl/c.rd')
		assert_equal( @pubdir_path + '/pub_rd/amrita_wz_rd.html', desc.publish_to)
		
		desc = @fac.create(@proj_path + '/rd/with_tmpl/c.rd')
		assert_equal( @pubdir_path + '/pub_rd/amrita_wz_rd.html', desc.publish_to)
		
		desc = @fac.create(@proj_path + '/rd/img.gif')
		assert_nil(desc.publish_to)
		
		@fac.error_out = []
		desc = @fac.create(@proj_path + '/scripts/home.rb')
		assert_nil( desc)
	end

	def test_create_publish_description_from_file
		# <?publish by="amrita" script="home.rb" class="Home"?>
		desc = @fac.create(@htsrc_path + '/home.html')
		assert_equal( 'amrita', desc.by)
		assert_equal( @pubdir_path + '/home.html', desc.publish_to)
		assert_equal( {'script' => 'home.rb', 'class' => 'Home'}, desc.arguments )
		
		desc = @fac.create(@htsrc_path + '/folder/subfolder/subsubfolder/1-1-1-1.html')
		assert_equal('copy', desc.by)
		assert_equal( @pubdir_path + '/folder/subfolder/subsubfolder/1-1-1-1.html', desc.publish_to)
		
		assert_equal('aaa', desc.arguments['opt_a'])
		assert_equal('a b c d e', desc.arguments['opt_b'])
	end
	
	def test_match_group_one_segment
		res = PublishDescriptionFactory.match_group('a.html', 'a.html')
		assert_equal([['a.html']], res)

		res = PublishDescriptionFactory.match_group('*.html', 'a.html')
		assert_equal([['a.html', 'a']], res)

		res = PublishDescriptionFactory.match_group('a?.html', 'ab.html')
		assert_equal([['ab.html', 'b']], res)
		
		res = PublishDescriptionFactory.match_group('??.html', 'a.html')
		assert_nil(res)
	end
	
	def test_match_group_more_segments
		# two segments
		res = PublishDescriptionFactory.match_group('a/b.html', 'a/b.html')
		assert_equal([['a'], ['b.html']], res)
		
		res = PublishDescriptionFactory.match_group('a/?.html', 'a/b.html')
		assert_equal([['a'], ['b.html', 'b']], res)
		
		res = PublishDescriptionFactory.match_group('?/*.html', 'a/b.html')
		assert_equal([['a', 'a'], ['b.html', 'b']], res)
		
		res = PublishDescriptionFactory.match_group('?/*.html', 'aa/b.html')
		assert_nil(res)
		
		# tree segments
		res = PublishDescriptionFactory.match_group('a/b/c.html', 'a/b/c.html')
		assert_equal([['a'], [ 'b'], ['c.html']], res)
		
		res = PublishDescriptionFactory.match_group('a/b/?.html', 'a/b/c.html')
		assert_equal([['a'], ['b'], ['c.html', 'c']], res)
		
		res = PublishDescriptionFactory.match_group('?/?/*.html', 'a/b/c.html')
		assert_equal([['a', 'a'], ['b', 'b'], ['c.html', 'c']], res)
		
		res = PublishDescriptionFactory.match_group('*/*/*.html', 'a/b/c.html')
		assert_equal([['a', 'a'], ['b', 'b'], ['c.html', 'c']], res)
	end
	
	def test_double_asterisk
		# more segments
		res = PublishDescriptionFactory.match_group('**/x.html', 'a/b/c/x.html')
		assert_equal([
			['a', nil],
			['b', nil],
			['c', nil],
			['x.html'] ], res)
		
		res = PublishDescriptionFactory.match_group('a/**/x.html', 'a/b/c/x.html')
		assert_equal([
			['a'],
			['b', nil],
			['c', nil],
			['x.html'] ], res)

		res = PublishDescriptionFactory.match_group('**/c/x.html', 'a/b/c/x.html')
		assert_equal([
			['a', nil],
			['b', nil],
			['c'],
			['x.html'] ], res)

		res = PublishDescriptionFactory.match_group('**/x/x.html', 'a/b/c/x.html')
		assert_nil(res)
		
		res = PublishDescriptionFactory.match_group('**/*.html', 'x.html')
		assert_equal([['x.html', 'x']], res)
	end
	
	def test_create_publish_from_dir
		dir = @htsrc_path + '/folder/subfolder/subsubfolder'
		
		desc = @fac.create(dir + '/1-1-1-2.html')
		assert_equal('amrita', desc.by)
		
		assert_equal(File.expand_path( dir + '/1-1-1-2.html'), desc.publish_from)
		assert_equal('base.rb', desc.arguments['script'] )
		assert_equal('Base', desc.arguments['class'] )
		
		assert_equal(@pubdir_path +  '/folder/1/1/1/2.html', desc.publish_to)
		assert desc.web_project
		
		desc = @fac.create(dir + '/s-1.html')
		assert_equal( @pubdir_path + '/folder/subfolder/subsubfolder/s-1.html', desc.publish_to)
		assert_equal( 's.rb', desc.arguments['script'] )
		assert_equal( 's', desc.arguments['method'] )
		assert_nil( desc.arguments['class'] )
		
		desc = @fac.create(dir + '/t-1.html')
		assert_equal( @pubdir_path +  '/foo_folder/t-1.html', desc.publish_to)
		assert_equal( 't.rb', desc.arguments['script'] )
		assert_nil(desc.arguments['method'] )
		assert_equal( 'T', desc.arguments['class'] )
		
		desc = @fac.create(dir + '/u-1.html')
		assert_equal(@pubdir_path +  '/uuu.html', desc.publish_to)
		assert_equal( 'u.rb', desc.arguments['script'] )
		assert_equal( 'u', desc.arguments['method'] )
		assert_equal( 'U', desc.arguments['class'] )
		
		desc = @fac.create(dir + '/1-1-1-3.html')
		assert_equal( @pubdir_path + '/folder/3.html', desc.publish_to)
		
		desc = @fac.create(dir + '/1-1-1-2.html')
		assert_equal( @pubdir_path + '/folder/1/1/1/2.html', desc.publish_to)
		
		desc = @fac.create(dir + '/1-1-1-1.html')
		assert_equal( @pubdir_path + '/folder/subfolder/subsubfolder/1-1-1-1.html', desc.publish_to)

		desc = @fac.create(dir + '/1-1-1-1.html')
		assert_equal( @pubdir_path + '/folder/subfolder/subsubfolder/1-1-1-1.html', desc.publish_to)
		
		desc = @fac.create(@htsrc_path + '/home.html')
		assert_equal( @pubdir_path + '/home.html', desc.publish_to)
	end
	
end
