/*
 * $Id: WebResourceLinkModel.java,v 1.6 2004/06/07 06:46:36 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core;

import java.net.*;
import java.util.ArrayList;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

/**
 * WebResourceLinkModel roles are:
 * 
 * <ul>
 * <li>Manage relative link and absolute path.
 * <li>Listener archtecture, notify that value changed. (It used to ui widget).
 * </ul>
 */
public class WebResourceLinkModel {

	IContainer baseContainer;
	WebProject webProject;
	
	ArrayList listeners = new ArrayList();

	IStatus status;
	IPath currentPath = null;
	boolean absolute = false;

	public WebResourceLinkModel(IContainer base) throws CoreException {
		this.baseContainer = base;
		webProject = (WebProject)base.getProject().getNature(WebProject.ID_NATURE);
		if(webProject == null){
			throw new IllegalArgumentException("Invalid container need to include web project:" + base);
		}
		if(!isPublishFolderIncluded(base)){
			IPath htSourcePath = webProject.getFolder(WebProject.HTSOURCES_FOLDER).getFullPath();
			if( !htSourcePath.isPrefixOf(base.getFullPath()) ){
				throw new IllegalArgumentException("Invalid location:" + base);
			}
			baseContainer = webProject.getFolder(WebProject.PUBLISH_FOLDER);
			IPath relPath = base.getFullPath().removeFirstSegments( htSourcePath.segmentCount() ); 
			if(!relPath.isEmpty()){
				baseContainer = baseContainer.getFolder(relPath);
			}
		}
	}
	
	boolean isPublishFolderIncluded(IResource res){
		IContainer publicFolder = webProject.getFolder(WebProject.PUBLISH_FOLDER);
		IContainer c = res instanceof IContainer ? (IContainer)res : res.getParent();
		do{
			if(publicFolder.equals(c)){
				return true;
			}
		}while(!((c = c.getParent()) instanceof IWorkspaceRoot));
		
		return false;
	}
	
	public void setAbsolute(boolean b){
		if(absolute == b){
			return;
		}
		absolute = b;
		adjustPath();
	}

	public IPath getPath() {
		return currentPath;
	}

	public void setPath(String path) {
		setPath(new Path(path));
	}

	public void setPath(IPath path) {
		setPath(path, true);
	}
	
	public void setPath(IPath path, boolean withAbsoluteCheck) {
		if(path == currentPath || (currentPath != null && currentPath.equals(path)) ){
			return;
		}
		currentPath = path;
		if(withAbsoluteCheck){
			absolute = path.isAbsolute();
			adjustPath();
		}
		
		fireHTReferenceTextChanged();
		notifyMessage();
	}
	
	void notifyMessage(){
		String path;
		if(currentPath == null || (path = currentPath.toString()).length() == 0){
			setStatus(IStatus.ERROR, "Need to specify link target.");
		} else if(currentPath.toString().indexOf(' ') != -1){
			setStatus(IStatus.ERROR, "Can not conain a char blink.");
		} else if(isURL(path)){
			setStatus(IStatus.OK, "");
		} else {
			if(getWebResource() != null){
				setStatus(IStatus.OK, "");
			}else{
				setStatus(IStatus.ERROR, "Sepcify resource is not exist");
			}
		}
	}

	static boolean isURL( String u){
		try {
			new URL(u);
			return true;
		} catch (MalformedURLException e) {
		}
		return false;	
	}
	
	public boolean isAbsolute() {
		return absolute;
	}
	
	void adjustPath(){
		IResource res = convertPathToWebResource(currentPath);
		
		if(absolute && !currentPath.isAbsolute()){
			// relative to absolute
			IPath path = (res != null) ?
				convertResourceToWebAbsolutePath(res) :
				currentPath.makeAbsolute();
			setPath(path, false);
		} else if (!absolute && currentPath.isAbsolute()){
			// absolute to relative
			if(res != null){
				setPath( makeRelativePath(baseContainer.getFullPath(), res.getFullPath()), false);
			}
		}
	}

	public static IPath makeRelativePath(IPath from, IPath to){
		if( from.equals(to) ){
			return new Path(from.lastSegment());
		}
		int segments = from.matchingFirstSegments(to);
		int up = from.removeFirstSegments(segments).segmentCount();
		
		StringBuffer buff = new StringBuffer();
		for(int i=0; i<up; i++){
			buff.append("../");
		}
		buff.append( to.removeFirstSegments(segments) );
		return new Path(buff.toString()).makeRelative();
	}
	
	IPath convertResourceToWebAbsolutePath(IResource res){
		IPath from = res.getFullPath();
		IPath to = webProject.getFolder(WebProject.HTSOURCES_FOLDER).getFullPath();
		
		int segments = to.matchingFirstSegments(from);
		if( segments != to.segmentCount()){
			to = webProject.getFolder(WebProject.PUBLISH_FOLDER).getFullPath();
			segments = to.matchingFirstSegments(from);
			if( segments != to.segmentCount()){
				return null;
			}
		}
		return new Path('/' + from.removeFirstSegments(segments).toString());
	}
	
	/**
	 * Returns specified path represents web project resources.
	 */
	IResource convertPathToWebResource(IPath path){
		if(path.isAbsolute()){
			return webProject.getFolder(WebProject.PUBLISH_FOLDER).findMember(path);
		}else{
			return baseContainer.findMember(path);
		}
	}
	
	public IResource getWebResource(){
		return currentPath != null ? convertPathToWebResource(currentPath) : null;
	}
	
	public void setWebResource(IResource res) {
		if( !isPublishFolderIncluded(res) ){
			throw new IllegalArgumentException("Not published file specified:" + res);
		}
		if(absolute){
			setPath( convertResourceToWebAbsolutePath(res), false);
		}else{
			setPath( makeRelativePath(baseContainer.getFullPath(), res.getFullPath()), false);
		}
	}

	public void addListener(WebResourceSelectionListener listener){
		listeners.add(listener);
	}
	
	public void removeListener(WebResourceSelectionListener listener){
		listeners.remove(listener);
	}

	void fireHTReferenceTextChanged(){
		for (int i = 0; i < listeners.size(); i++) {
			((WebResourceSelectionListener)listeners.get(i)).resourceChanged( currentPath, getWebResource());
		}
	}

	void setStatus(int type, String msg){
		IStatus s = new Status(type, WebpubPlugin.ID_PLUGIN, IStatus.OK, msg, null);
		if(status == null || !status.toString().equals(s.toString()) ){
			status = s;
			fireMessageChanged();
		}
	}

	void fireMessageChanged(){
		for (int i = 0; i < listeners.size(); i++) {
			((WebResourceSelectionListener)listeners.get(i)).statusChanged(status);
		}
	}

}
