/*
 * $Id: PublishDescriptionFactory.java,v 1.15 2004/06/07 06:46:35 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;

import java.io.IOException;
import java.util.*;
import java.util.regex.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

public class PublishDescriptionFactory {

	final static String
		PUBLISHCODE_BEGIN = "<?publish ",
		PUBLISHCODE_END = "?>";		
	
	final WebProject webProject;
	
	public PublishDescriptionFactory(WebProject webProject){
		this.webProject = webProject;
	}
	
	/**
	 * <p>
	 * To create PublishDescription from iterator. 
	 * This method for on memory document data.
	 * 
	 * <p>
	 * This method can not automatically set a publish location.
	 */
	public PublishDescription create(Iterator ite) {
		while(ite.hasNext()){
			String line = (String)ite.next();
			if(line == null){
				continue;
			}
			int begin = line.indexOf(PUBLISHCODE_BEGIN);
			int end = line.indexOf(PUBLISHCODE_END);
			if( begin != -1 && end != -1){
				return createFromMatcher(line.substring(begin + PUBLISHCODE_BEGIN.length(), end));
			}
		}
		return null;
	}

	boolean checkPublishSource(IResource source) throws CoreException {
		if(!source.isAccessible()){
			return false;
		}
		IContainer pubFolder = webProject.getFolder(WebProject.PUBLISH_FOLDER);
		if( pubFolder.getFullPath().isPrefixOf(source.getFullPath())){
			return false;
		}
		
		String[] ignores = webProject.getArray(WebProject.IGNORE_FILE);
		for (int i = 0; i < ignores.length; i++) {
			if( new Wildcard(ignores[i]).match(source.getName()) ){
				return false;
			}
		}
		return true;
	}
	
	/**
	 * <p>
	 * Create PublishDescription (this method is define publish location from
	 * referer htFile location that process is not exec other method).
	 * 
	 * <p>
	 * Return null if do not found publish description.
	 */
	public PublishDescription create(IFile source) throws IOException, CoreException, IllegalConfigurationException {
		if(!checkPublishSource(source)){
			return null;
		}
		PublishDescription desc = null;
		if(source instanceof IFile){
			desc = createFromFile((IFile)source);
		}
		if(desc == null){
			// if can not create specify html file.
			// search publish mapping in parent folder.
			desc = createFromParent(source);
		}
		if(desc == null){
			desc = createDefaultPublishDescription(source);
		}
		return desc;
	}
	
	public PublishDescription createFromFile(IFile source) throws CoreException, IOException {
		WebProject wp = (WebProject)source.getProject().getNature(WebProject.ID_NATURE);
		
		if( !source.isAccessible() || wp == null || !wp.isHTExtension(source.getFileExtension()) ){
			return null;
		}
		TextReader reader = null;
		try{
			reader = new TextReader(source.getContents());
			PublishDescription desc = create(reader);
			if(desc != null){
				IPath sourceDir = wp.getFolder(WebProject.HTSOURCES_FOLDER).getFullPath();
				desc.setPublishTo(source.getFullPath().removeFirstSegments(sourceDir.segmentCount()));
			}
			return desc;
		} finally{
			if(reader != null){
				reader.close();
			}
		}
	}
	
	/**
	 * Creates publish description from parent folder on reflexive.
	 * If none looking publish description in html source folder, return null.
	 */
	PublishDescription createFromParent(IFile source) throws CoreException {
		PublishMapping mapping = findPublishMapping(source);
		if(mapping == null){
			return null;
		}
		PublishDescription desc = new PublishDescription(mapping.getPublishBy());
		
		Map args = mapping.getArguments();
		Object[] keys = args.keySet().toArray();
		for (int k = 0; k < keys.length; k++) {
			String key = (String)keys[k];
			desc.setArgument(key, (String)args.get(key));
		}
		initPublishPath(source, desc, mapping);
		
		return desc;
	}
	
	void initPublishPath(IResource source, PublishDescription desc, PublishMapping mapping) throws CoreException {
		// define publish to
		IPath publishToPath = mapping.getPublishTo();
		
		IPath publishToBaseFolderPath = new Path("/");
		if(publishToPath != null){
			// specify publish to location
			if (publishToPath.isAbsolute()){
				// base folder is publish folder root.
				publishToPath = publishToPath.makeRelative();
				if(Wildcard.containWildcardSymble(publishToPath)){
					publishToPath = new Wildcard(publishToPath);
				}
			} else {
				// relative path
				IPath path = htSourceToPublish(mapping.getPropertyFile().getParent());
				if(path != null){
					publishToBaseFolderPath = path;
				}
			}
			if(publishToPath instanceof Wildcard){
				// make path by path and wildcard
				IPath matchPattern = mapping.getMatchPattern();
				if(matchPattern instanceof Wildcard){
					IPath srcRelPath = source.getFullPath().removeFirstSegments(mapping.getPropertyFile().getParent().getFullPath().segmentCount());
					String[][] ress = ((Wildcard)matchPattern).matchWithGroup(srcRelPath);
					publishToPath = ((Wildcard)publishToPath).makePathByMatchResult(ress);
				}
			}
			
			desc.setPublishTo(publishToBaseFolderPath.append(publishToPath));
		}else{
			IPath pubTo = htSourceToPublish(source);
			if(pubTo != null){
				desc.setPublishTo(pubTo);
			}
		}
	}

	IPath htSourceToPublish(IResource res) throws CoreException{
		IPath srcFolderPath = webProject.getFolder(WebProject.HTSOURCES_FOLDER).getFullPath();
		IPath srcPath = res.getFullPath();
		if (srcFolderPath.isPrefixOf(srcPath)){
			return srcPath.removeFirstSegments(srcFolderPath.segmentCount()).makeAbsolute();
		}
		return null;
	}
	
	public PublishMapping findPublishMapping(IFile source) throws CoreException {
		IFile[] files = findPublishMappingFiles(source);
		for (int i = 0; files != null && i < files.length; i++) {
			IFile f = files[i];
			PublishMapping[] mappings = PublishMappingStore.getInstance().getMapping(f);
			IPath mapDirPath = f.getParent().getFullPath();
			IPath relPath = source.getFullPath().removeFirstSegments(mapDirPath.segmentCount());
			for (int j=0; j<mappings.length; j++) {
				PublishMapping mapping = mappings[j];
				IPath matchPattern = mapping.getMatchPattern();
				if(matchPattern instanceof Wildcard ? ((Wildcard)matchPattern).match(relPath) : matchPattern.equals(relPath)){
					return mapping;
				}
			}
		}
		return null;
	}
	
	public IFile[] findPublishMappingFiles(IResource source) throws CoreException {
		if(!checkPublishSource(source) ){
			return null;
		}
		ArrayList files = new ArrayList();
		for(IContainer c = source.getParent(); c instanceof IFolder; c = c.getParent()){
			IResource mappingFile = ((IFolder)c).findMember(".publish");
			if(mappingFile instanceof IFile && mappingFile.exists()){
				files.add(mappingFile);
			}
		}
		return files.size() > 0 ? (IFile[]) files.toArray(new IFile[files.size()]) : null;
	}
	
	/**
	 * Return null if can not publish location as specify file.
	 */
	PublishDescription createDefaultPublishDescription(IResource targetFile) throws CoreException{
		if(targetFile.isAccessible()){
			WebProject webProj = (WebProject)targetFile.getProject().getNature(WebProject.ID_NATURE);
			if( webProj != null){
				IContainer htSourceFolder = webProj.getFolder(WebProject.HTSOURCES_FOLDER);
				IPath srcPath = htSourceFolder.getFullPath();
				if(srcPath.isPrefixOf(targetFile.getFullPath())){
					IPath pubTo = htSourceToPublish(targetFile);
					if(pubTo != null){
						PublishDescription desc = new PublishDescription("copy");
						desc.setPublishTo(pubTo);
						return desc;
					}
				}
			}
		}
		
		return null;
	}
	
	/**
	 * Parse as follow style line, and return that created PublishDescriptor
	 * form line.
	 * <pre>
	 *   file="home3.rb" class="Home" method="home" publish_to="foo/home.html
	 * </pre>
	 */
	PublishDescription createFromMatcher(String code) {
		return createFromMap( lineToMap(code) );
	}
	
	static Map lineToMap(String code){
		Map map = new HashMap();
		Pattern  pattern = Pattern.compile("(.+?)\\s*=\\s*\"(.+?)\"");
		Matcher matcher = pattern.matcher(code);
		while (matcher.find()) {
			map.put( matcher.group(1).trim(), matcher.group(2));
		}
		return map;
	}

	/**
	 * Create PublishDescription from specify map entries.
	 */
	PublishDescription createFromMap(Map map) {
		String publishBy = (String)map.remove("by");
		PublishDescription desc = new PublishDescription(publishBy);
		Object[] keys = map.keySet().toArray();
		for (int i = 0; i < keys.length; i++) {
			String k = (String)keys[i];
			desc.setArgument( k, (String)map.get(k) );
		}
		return desc;
	}

}

