/*
 * $Id: TocFactory.java,v 1.6 2004/06/07 06:46:37 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.toc;

import java.io.*;
import java.util.*;

import javax.xml.parsers.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import com.narucy.webpub.core.WebpubPlugin;

class TocHandler extends DefaultHandler {

	Toc toc = null;
	Stack elementStack = new Stack();

	public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {
		Topic node = null;
		String href = atts.getValue(Topic.HREF);
		String label = atts.getValue(Topic.LABEL);
		
		if (qName.equals("toc")) {
			node = toc = new Toc(href, label);
		} else if (qName.equals("topic")) {
			Topic parent = (Topic) elementStack.lastElement();
			node = parent.createTopic(href, label);
		}
		for(int i=0; i<atts.getLength(); i++){
			node.setAttribute(atts.getQName(i), atts.getValue(i));
		}
		
		elementStack.push(node);
	}

	public final void endElement(String namespaceURI, String localName, String qName) throws SAXException {
		elementStack.pop();
	}

	public Toc getToc() {
		return toc;
	}
}

class TocCheckHandler extends DefaultHandler {

	boolean checked = false;
	boolean isToc;
	
	public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {
		if ( !checked ) {
			isToc = qName.equals("toc");
			checked = true;
		}
	}

	public boolean isToc(){
		return isToc;
	}
}

public class TocFactory {

	final public static String
		ID_MARKER_TOC_PROBLEM = "com.narucy.webpub.core.toc.toc_problem";

	final static String[] TOC_FILE_EXTENSIONS = {"toc", "xml"};
	
	static SAXParserFactory factory = SAXParserFactory.newInstance();
	static {
		Arrays.sort(TOC_FILE_EXTENSIONS);
	}
	
	private TocFactory(){
	}
	
	public static Toc createToc(IFile file) throws CoreException{
		if(file.isAccessible()){
			IMarker[] markers = file.findMarkers(IMarker.PROBLEM, false,IResource.DEPTH_ZERO);
			for (int i = 0; i < markers.length; i++) {
				IMarker m = markers[i];
				if(Boolean.TRUE.equals(m.getAttribute(ID_MARKER_TOC_PROBLEM))){
					m.delete();
				}
			}
			InputStream stream = file.getContents();
			try {
				Toc toc = createToc(stream);
				toc.setTocFile(file);
				return toc;
			} catch (SAXException e) {
				/*
				if(e.getException() != null){
					e.getException().printStackTrace();
				}
				e.printStackTrace();
				*/
				IMarker m = file.createMarker(IMarker.PROBLEM);
				m.setAttribute(ID_MARKER_TOC_PROBLEM, true);
				m.setAttribute(IMarker.MESSAGE, e.getLocalizedMessage());
			}
		}
		return null;
	}

	public static Toc createToc(String tocCode) throws SAXException, CoreException {
		InputSource stream = new InputSource(new StringReader(tocCode));
		try{
			try{
				TocHandler handler = new TocHandler();
				factory.newSAXParser().parse(stream, handler);
				return handler.getToc();
			} catch (ParserConfigurationException e) {
				throw new CoreException(toStatus(e));
			}
		}catch(IOException e){
			throw new CoreException(toStatus(e));
		}
	}

	public static Toc createToc(InputStream stream) throws SAXException, CoreException {
		try{
			try{
				TocHandler handler = new TocHandler();
				factory.newSAXParser().parse(stream, handler);
				return handler.getToc();
			} catch (ParserConfigurationException e) {
				throw new CoreException(toStatus(e));
			}finally{
				stream.close();
			}
		}catch(IOException e){
			throw new CoreException(toStatus(e));
		}
	}

	static IStatus toStatus(Exception e){
		return new Status(
			IStatus.ERROR,
			WebpubPlugin.ID_PLUGIN,
			IStatus.OK,
			e.getMessage(),
			e);
	}
	
	public static boolean isTocResources(IResource res){
		if( res instanceof IStorage &&
			res.getFileExtension() != null &&
			Arrays.binarySearch(TOC_FILE_EXTENSIONS, res.getFileExtension()) >= 0 ){
			
			TocCheckHandler handler = new TocCheckHandler();
			InputStream stream = null;
			try{
				stream = ((IStorage)res).getContents();
				factory.newSAXParser().parse( stream, handler);
				return handler.isToc();
			} catch (SAXException e){
			} catch (IOException e) {
			} catch (ParserConfigurationException e) {
			} catch (CoreException e) {
			} finally{
				if(stream != null){
					try {
						stream.close();
					} catch (IOException ex) {
						WebpubPlugin.handleException(ex);
					}
				}
			}
		}
		return false;
	}

}
