/*
 * $Id: WebResourcesDecorator.java,v 1.11 2004/06/07 06:46:18 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui;

import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.core.runtime.Preferences.PropertyChangeEvent;
import org.eclipse.jface.viewers.*;

import com.narucy.webpub.core.WebProject;

public class WebResourcesDecorator extends LabelProvider implements ILightweightLabelDecorator, IResourceChangeListener {

	final public static String
		ID_DECORATOR = "com.narucy.webpub.ui.WebResourcesDecorator";
	
	static String[] folderKeys = {
		WebProject.HTSOURCES_FOLDER,
		WebProject.SCRIPTS_FOLDER,
		WebProject.PUBLISH_FOLDER,
	};
	
	static int getKeyIndex(String k){
		for(int i=0; i<folderKeys.length; i++){
			if(folderKeys[i].equals(k)){
				return i;
			}
		}
		return -1;
	}
	
	HashMap listenersCache = new HashMap();
	
	public WebResourcesDecorator(){
		try{
			IProject[] projs = ResourcesPlugin.getWorkspace().getRoot().getProjects();
			for (int i = 0; i < projs.length; i++) {
				IProject proj = projs[i];
				if(proj.hasNature(WebProject.ID_NATURE)){
					regist(proj);
				}
			}
		}catch(CoreException ex){
			WebpubUIPlugin.handleException(ex);
		}
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
	}
	
	void regist(final IProject proj) throws CoreException {
		final WebProject wp = (WebProject)proj.getNature(WebProject.ID_NATURE);
		if(wp == null || listenersCache.containsKey(proj)){
			return;
		}
		Preferences.IPropertyChangeListener listener = new Preferences.IPropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				String prop = event.getProperty();
				if(getKeyIndex(prop) != -1){
					refresh(wp, prop);
				}
			}
		};
		wp.addPropertyChangeListener(listener);
		
		Object[] entry = {
			wp.getFolder(WebProject.HTSOURCES_FOLDER),
			wp.getFolder(WebProject.SCRIPTS_FOLDER),
			wp.getFolder(WebProject.PUBLISH_FOLDER),
			listener,
		};
		listenersCache.put(proj, entry);
		
		IResource[] refreshResources = new IResource[3];
		System.arraycopy(entry, 0, refreshResources, 0, 3);
		fireLabelProviderChanged(new LabelProviderChangedEvent(this, refreshResources));
	}
	
	Preferences.IPropertyChangeListener getCachedListener(IProject proj){
		Object[] entry = (Object[])listenersCache.get(proj);
		return entry != null ? (Preferences.IPropertyChangeListener)entry[3] : null;
	}
	
	void refresh(WebProject wp, String key){
		Object[] entry = (Object[])listenersCache.get(wp.getProject());
		int entryKeyIndex = getKeyIndex(key);
		
		// erase decorate old container and new decorate new container
		IContainer[] refreshContainers = {
			(IContainer)entry[entryKeyIndex],
			wp.getFolder(key),
		};
		fireLabelProviderChanged(new LabelProviderChangedEvent(this, refreshContainers));
		
		// update cached entry
		entry[entryKeyIndex] = refreshContainers[1];
	}

	public void decorate(Object element, IDecoration decoration) {
		if(!(element instanceof IAdaptable)){
			return;
		}
		IContainer container = element instanceof IContainer ?
			(IContainer)element :
			(IContainer)((IAdaptable) element).getAdapter(IContainer.class);
		
		if(container != null){
			Object[] projects = listenersCache.keySet().toArray();
			for (int i = 0; i < projects.length; i++) {
				IProject proj = (IProject)projects[i];
				Object[] entry = (Object[])listenersCache.get(proj);
				
				if(proj.equals(container)){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/webproject_ovr.gif"), IDecoration.TOP_RIGHT);
				} else if (entry[0].equals(container)){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/htsource_folder_ovr.gif"), IDecoration.TOP_RIGHT);
				} else if (entry[1].equals(container) ){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/scripts_folder_ovr.gif"), IDecoration.TOP_RIGHT);
				} else if( entry[2].equals(container) ){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/publish_folder_ovr.gif"), IDecoration.TOP_RIGHT);
				}
			}
		}
	}

	public void resourceChanged(IResourceChangeEvent event) {
		if(event.getDelta() == null){
			return;
		}
		try {
			final ArrayList refreshResources = new ArrayList();
			event.getDelta().accept(new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta delta) throws CoreException {
					IResource r = delta.getResource();
					int kind = delta.getKind();
					if(r instanceof IProject){
						if(kind == IResourceDelta.ADDED || kind == IResourceDelta.REMOVED){
							regist((IProject)r);
						}else if(kind == IResourceDelta.REMOVED){
							unregist((IProject)r);
						}
					}
					return true;
				}
			});
		} catch (CoreException ex) {
			WebpubUIPlugin.handleException(ex);
		}
	}
	
	void unregist(IProject proj){
		Object[] entry = (Object[])listenersCache.remove(proj);
		if(entry != null && proj.isAccessible()){
			try{
				WebProject wp = (WebProject)proj.getNature(WebProject.ID_NATURE);
				if(wp != null){
					wp.removePropertyChangeListener( (Preferences.IPropertyChangeListener)entry[3] );
				}
			}catch(CoreException ex){
				WebpubUIPlugin.handleException(ex);
			}
		}
	}

	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		Object[] projects = listenersCache.keySet().toArray();
		for (int i=0; i<projects.length; i++) {
			unregist((IProject)projects[i]);
		}
	}
	

}


