/* 
 * $Id: HTContentAssistProcessor.java,v 1.3 2004/06/07 06:46:16 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.editors.html;

import java.util.*;

import org.eclipse.jface.text.*;
import org.eclipse.jface.text.contentassist.*;

import com.narucy.webpub.ui.HTParser;
import com.narucy.webpub.ui.editors.ColorManager;

/**
 * 
 */
public class HTContentAssistProcessor implements IContentAssistProcessor {

	final static int
		PROPOSAL_MODE_NONE = 0,
		PROPOSAL_MODE_INSERT_ELEMENT = 1,
		PROPOSAL_MODE_MODIFY_ELEMENT = 2,
		PROPOSAL_MODE_CLOSEING_ELEMENT = 3,
		PROPOSAL_MODE_ATTRIBUTE = 4,
		PROPOSAL_MODE_ATTRIBUTE_VALUE = 5;

	static char[] spaceChars = new char[]{' ', '\n', '\r', '\t'};
	static {
		Arrays.sort(spaceChars);
	}

	char[] autoActivationChars = new char[]{'<'};
	ColorManager colorManager;
	HTContentAssistEngine engine;
	
	public HTContentAssistProcessor(ColorManager colorManager) {
		this.colorManager = colorManager;
		engine = new HTContentAssistEngine(HTContentAssistEngine.XHTML_STRICT);
	}

	static int getComplationPrefixIndex(IDocument doc, int offset, char[] foundChar, char[] invalidChar) throws BadLocationException{
		Arrays.sort(invalidChar);
		Arrays.sort(foundChar);
		for(int i=offset-1; i>=0; i--){
			char c = doc.getChar(i);
			if( Arrays.binarySearch(invalidChar, c) >= 0){
				return -1;
			}else if(Arrays.binarySearch(foundChar, c) >= 0){
				return i;
			}
		}
		return -1;
	}
	
	public String getElementComplationPrefix(IDocument doc, int offset) throws BadLocationException{
		int index = getComplationPrefixIndex(doc, offset, new char[]{'<'}, new char[]{'>', ' ', '\t', '\r', '\n'});
		return getComplationPrefix(doc, offset, index);
	}

	public String getComplationPrefix(IDocument doc, int offset, int index) throws BadLocationException{
		return index != -1 ? doc.get(index, offset-index) : null;
	}

	public ICompletionProposal[] computeCompletionProposals(ITextViewer viewer, int offset) {
		IDocument doc = viewer.getDocument();
		try {
			switch(getMode(doc, offset)){
				case PROPOSAL_MODE_INSERT_ELEMENT:
					return createElementCompletionProposals(viewer, offset, null);
				case PROPOSAL_MODE_MODIFY_ELEMENT:
					String prefix = getElementComplationPrefix(doc, offset);
					return  createElementCompletionProposals(viewer, offset, prefix);
				case PROPOSAL_MODE_ATTRIBUTE:
					return createAttributeCompletionProposals(viewer, offset);
			}
		} catch (BadLocationException e) {
		}
		return null;
	}

	int getMode(IDocument doc, int offset) throws BadLocationException{
		if( HTParser.isInToken(doc, offset) ){
			char c = doc.getChar(offset);
			if(doc.getChar(offset-1) == '<'){
				return PROPOSAL_MODE_MODIFY_ELEMENT;
			} else if ( getElementComplationPrefix(doc, offset) != null){
				return PROPOSAL_MODE_MODIFY_ELEMENT;
			} else if( c == '>' || c == ' '){
				return PROPOSAL_MODE_ATTRIBUTE;
			} else{
				return PROPOSAL_MODE_NONE;
			}
		}else{
			return PROPOSAL_MODE_INSERT_ELEMENT;
		}
	}
	
	ICompletionProposal[] createElementCompletionProposals(ITextViewer viewer, int offset, String prefix) throws BadLocationException{
		IDocument doc = viewer.getDocument();
		
		String elemName;
		int replaceLength = 0;
		if(prefix != null){
			elemName = HTParser.chooseElementName(doc, offset, 1);
			int tmpOffset = HTParser.getElementRange(doc, offset).getOffset();
			replaceLength = offset - tmpOffset;
			offset = tmpOffset;
		}else{
			elemName = HTParser.getElementName(doc, offset);
		}
		
		if(elemName != null){
			String[] candities = engine.getCandidateElements(elemName);
			if( candities != null){
				ArrayList proposals = new ArrayList();
				String tagPrefix = prefix != null && prefix.length() >= 2 && prefix.charAt(0) == '<' ? prefix.substring(1) : null;
				for (int i = 0; i < candities.length; i++) {
					if(tagPrefix == null || candities[i].startsWith(tagPrefix) ){
						proposals.add( createElementProposal(candities[i], offset, replaceLength) );
					}
				}
				if(proposals.size() > 0){
					return (ICompletionProposal[])proposals.toArray(new ICompletionProposal[proposals.size()]);
				}
			}
		}
		return null;
	}
	
	CompletionProposal createElementProposal(String elemName, int offset, int replaceLength){
		String insertString = engine.isEmptyElement(elemName) ?
			"<" + elemName + " />" :
			"<" + elemName + "></" + elemName + ">";
		
		return new CompletionProposal(
			insertString,
			offset,
			replaceLength,
			elemName.length() + 2,
			colorManager.getImage("obj16/tag.gif"),
			elemName,
			null,
			null);
	}
	
	ICompletionProposal[] createAttributeCompletionProposals(ITextViewer viewer, int offset) throws BadLocationException{
		IDocument doc = viewer.getDocument();
		String elemName = HTParser.getElementName(doc, offset);

		if(elemName != null){
			String[] candities = engine.getCandidateAttribute(elemName);
			if(candities != null){
				int index = getComplationPrefixIndex(doc, offset, spaceChars, new char[]{'<', '"'});
				String attPrefix = getComplationPrefix(doc, offset, index);
				if(attPrefix != null){
					attPrefix = attPrefix.trim();
					if( attPrefix.length() == 0){
						attPrefix = null;
					}
				}
				int replaceLength = 0;
				if(attPrefix != null){
					int tmpOffset = offset - attPrefix.length();
					replaceLength = offset - tmpOffset;
					offset = tmpOffset;
				}
				
				ArrayList proposals = new ArrayList();
				for (int i = 0; i < candities.length; i++) {
					if( attPrefix == null || candities[i].startsWith(attPrefix)){
						boolean insertBeforeSpace = (Arrays.binarySearch( spaceChars, doc.getChar(offset-1)) < 0);
						proposals.add( createAttributeProposal(candities[i], offset, replaceLength, insertBeforeSpace) );
					}
				}
				return (ICompletionProposal[])proposals.toArray(new ICompletionProposal[proposals.size()]);
			}
		}
		return null;
	}

	CompletionProposal createAttributeProposal(String attrName, int offset, int replaceLength, boolean insertBeforeSpace){
		String insertString = attrName + "=\"\"";
		if(insertBeforeSpace) {
			insertString = ' ' + insertString;
		}
		return new CompletionProposal(
			insertString,
			offset,
			replaceLength,
			attrName.length() + (insertBeforeSpace ? 3 : 2),
			colorManager.getImage("obj16/attribute.gif"),
			attrName,
			null,
			null);
	}

	public IContextInformation[] computeContextInformation(ITextViewer viewer, int documentOffset) {
		return null;
	}

	public char[] getCompletionProposalAutoActivationCharacters() {
		return (char[])autoActivationChars.clone();
	}

	public char[] getContextInformationAutoActivationCharacters() {
		return null;
	}

	public String getErrorMessage() {
		return null;
	}

	public IContextInformationValidator getContextInformationValidator() {
		return null;
	}
	
}
