/*
 * $Id: PublishMappingBlock.java,v 1.3 2004/06/07 06:46:16 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.io.*;
import java.util.*;

import javax.xml.parsers.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.swt.widgets.Text;
import org.w3c.dom.*;
import org.xml.sax.*;

import com.narucy.webpub.core.IStatusChangeListener;
import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.ui.*;

/**
 * 
 */
public class PublishMappingBlock {

	final static String
		COLKEY_PATTERN = "pattern",
		COLKEY_BY = "by",
		COLKEY_PUBLISHTO = "publishTo",
		COLKEY_ARGUMENTS = "arguments";

	static String[] specialAttributes = {
		"by", "publish_to"
	};

	static String[] columnProps = new String[]{
		COLKEY_PATTERN,
		COLKEY_BY,
		COLKEY_PUBLISHTO,
		COLKEY_ARGUMENTS
	};
	
	HashMap propertyNames = new HashMap();
	
	PublisherRegistory registory = PublisherRegistory.getInstance();

	String[] publishKeys;
	IStatusChangeListener statusListener;
	
	DocumentBuilderFactory docBuilderFac = DocumentBuilderFactory.newInstance();

	Composite base;	
	TableViewer publishDescriptionViewer;

	Button
		addDescItemButton,
		removeDescItemButton,
		upDescItemButton,
		downDescItemButton;

	Text publishDescriptionText;

	IFile editTargetResource;

	public PublishMappingBlock(Composite parent, IFile editTargetResource, IStatusChangeListener statusListener) {
		// set column names
		propertyNames.put( COLKEY_PATTERN, "File Match Pattern");
		propertyNames.put( COLKEY_BY, "Publish By");
		propertyNames.put( COLKEY_PUBLISHTO, "Publish To");
		propertyNames.put( COLKEY_ARGUMENTS, "Arguments");
		
		// keys settings.
		publishKeys = registory.getPublishByKeys();
		Arrays.sort(publishKeys);
		
		this.editTargetResource = editTargetResource;
		this.statusListener = statusListener;
		
		// create control.
		base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(2, false) );
		
		// top label
		Label labe = new Label(base, SWT.NONE);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		labe.setLayoutData(gd);
		labe.setText("Define publisher &mappings:");
		labe = new Label(base, SWT.NONE);

		// create table viewer
		publishDescriptionViewer = new TableViewer(
			base, SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		
		// table settings
		Table table = publishDescriptionViewer.getTable();
		gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 4;
		table.setLayoutData(gd);
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// configuration providers
		publishDescriptionViewer.setColumnProperties(columnProps);
		publishDescriptionViewer.setContentProvider( new PublishDescriptionContentProvider() );
		publishDescriptionViewer.setLabelProvider( new PublishDescriptionLabelProvider() );
		publishDescriptionViewer.setCellModifier( new PublishDescriptionCellModifier() );
		publishDescriptionViewer.setCellEditors( createCellEditors() );
		
		// column settings
		for (int i = 0; i < columnProps.length; i++) {
			String key = columnProps[i];
			TableColumn col = new TableColumn(table, SWT.LEFT);
			col.setText((String)propertyNames.get(key));
		}

		// create buttons
		addDescItemButton = createButton( base, "&Add");
		removeDescItemButton = createButton( base, "&Remove");
		upDescItemButton = createButton( base, "&Up");
		downDescItemButton = createButton( base, "&Down");
		
		// create bottom label.
		new Label(base, SWT.NONE).setText("Publisher description:");
		new Label(base, SWT.NONE);
		publishDescriptionText = new Text(
			base,
			SWT.MULTI | SWT.BORDER | SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL | SWT.H_SCROLL);
		
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 2;
		publishDescriptionText.setLayoutData(gd);
		
		publishDescriptionViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshPublishDescriptionText();
				refreshButtonState();
			}
		});
		
		initContent();
	}
	
	void handleButtonPressed(Button b){
		PublishMappingDocumentProxy proxy = (PublishMappingDocumentProxy)publishDescriptionViewer.getInput();
		if(b == addDescItemButton){
			proxy.addInitialPublishDescription(getSelectedElement());
		} else if(b == removeDescItemButton){
			proxy.removeElement( getSelectedElement() );
		} else if(b == upDescItemButton){
			proxy.moveElement( getSelectedElement(), -1);
		} else if(b == downDescItemButton){
			proxy.moveElement( getSelectedElement(), 1);
		}
		refreshAll();
	}

	public IFile getEditTarget() {
		return editTargetResource;
	}

	/**
	 * Return true if all of publish to use publish element attribute name.
	 * (therefore return false if publish specifed argument.)
	 */
	static boolean isSpecialAttribute(String name){
		return Arrays.binarySearch(specialAttributes, name) >= 0;
	}
	
	static int getColumnIndex(String name){
		for(int i=0; i<columnProps.length; i++){
			if( columnProps[i].equals(name) ){
				return i;
			}
		}
		return -1;
	}

	static class PublishDescriptionContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object inputElement) {
			PublishMappingDocumentProxy docFacade = (PublishMappingDocumentProxy)inputElement;
			return docFacade.getPublishElements();
		}

		public void dispose() {
			// do nothing
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			// do nothing
		}
	}
	
	static class PublishDescriptionLabelProvider extends TableLabelProvider {	
		public String getColumnText(Object element, int columnIndex) {
			Element mappingElem = (Element)element;
			Element publishElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			switch (columnIndex) {
				case 0:
					return mappingElem.getAttribute("pattern");
				case 1:
					String by = publishElem.getAttribute("by");
					return by;
				case 2:
					String to = publishElem.getAttribute("publish_to");
					return to != null && to.length() > 0 ? to : "(default)";
				case 3:
					StringBuffer buff = new StringBuffer("{");
					NamedNodeMap attrs = publishElem.getAttributes();
					for(int i=0, j=0; i<attrs.getLength(); i++){
						Node n = attrs.item(i);
						String name = n.getNodeName();
						if( !isSpecialAttribute(name) ){
							if( j++ > 0){
								buff.append(',');
							}
							buff.append(name);
							buff.append('=');
							buff.append(n.getNodeValue());
						}
					}
					return buff.append('}').toString();
			}
			return null;
		}
	}
	

	class PublishDescriptionCellModifier implements ICellModifier {

		public boolean canModify(Object element, String property) {
			return getColumnIndex(property) != -1;
		}

		public Object getValue(Object element, String property) {
			Element mappingElem = (Element)element;
			if (property.equals(COLKEY_PATTERN) ){
				return mappingElem.getAttribute("pattern");
			}
			Element publishElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			if( property.equals(COLKEY_PUBLISHTO)){
				return publishElem.getAttribute("publish_to");
			}
			if( property.equals(COLKEY_BY)){
				String by = publishElem.getAttribute("by");
				int index = Arrays.binarySearch(publishKeys, by);
				return new Integer(index);
			}
			if( property.equals(COLKEY_ARGUMENTS)){
				NamedNodeMap attrs = publishElem.getAttributes();
				ArrayList values = new ArrayList();
				for(int i=0; i<attrs.getLength(); i++){
					Node n = attrs.item(i);
					String name = n.getNodeName();
					if( !isSpecialAttribute(name) ){
						values.add( new String[]{name, n.getNodeValue()} );
					}
				}
				return values.toArray(new String[values.size()][]);
			}
			return null;
		}

		public void modify(Object element, String property, Object value) {
			if( element instanceof TableItem){
				element = ((TableItem)element).getData();
			}
			Element mappingElem = (Element)element;
			Element pubElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			if( property.equals(COLKEY_PATTERN) ){
				mappingElem.setAttribute("pattern", (String)value);
			}else if( property.equals(COLKEY_BY)){
				int index = ((Integer)value).intValue();
				pubElem.setAttribute("by", publishKeys[index]);
			}else if( property.equals(COLKEY_PUBLISHTO)){
				pubElem.setAttribute("publish_to", (String)value);
			}else if( property.equals(COLKEY_ARGUMENTS)){
				// clear attributes
				NamedNodeMap nm = pubElem.getAttributes();
				for(int i=0; i<nm.getLength(); i++){
					Node n = nm.item(i);
					String name = n.getNodeName();
					if( !isSpecialAttribute(name)){
						pubElem.removeAttribute(name);
					}
				}
				
				// add attributes
				String[][] newAttributes = (String[][])value;
				for (int i = 0; i < newAttributes.length; i++) {
					String[] entry = newAttributes[i];
					pubElem.setAttribute(entry[0], entry[1]);
				}
			}
			publishDescriptionViewer.refresh();
		}
	}
	
	CellEditor[] createCellEditors(){
		Table table = publishDescriptionViewer.getTable();
		
		CellEditor[] editors = new CellEditor[ columnProps.length ];
		for(int i=0; i<columnProps.length; i++){
			String prop = columnProps[i];
			CellEditor ed;
			if( prop.equals(COLKEY_PATTERN) ){
				ed = new DialogCellEditor(table){
					protected Object openDialogBox(Control control) {
						String pattern = (String)doGetValue();
						ResourceMatchDialog dialog = new ResourceMatchDialog(
							control.getShell(),
							pattern,
							editTargetResource.getParent() );
						
						return dialog.open() == Window.OK ? dialog.getFilter() : null;
					}
				};
			}else if(prop.equals(COLKEY_BY) ){
				ed = new ComboBoxCellEditor(table, publishKeys);
			}else if(prop.equals(COLKEY_PUBLISHTO)){
				ed = new TextCellEditor(table);
			}else if(prop.equals(COLKEY_ARGUMENTS)){
				ed = new DialogCellEditor(table){
					protected Object openDialogBox(Control control) {
						MapEditDialog dialog = new MapEditDialog(
								control.getShell(),
								(String[][])doGetValue());
						return dialog.open() == Window.OK ? dialog.getValues() : null;
					}
					
					protected void updateContents(Object value) {
						Label labe = getDefaultLabel();
						if (labe != null){
							StringBuffer buff = new StringBuffer("{");
							if (value != null){
								String[][] vs = (String[][])value;
								for(int i=0; i<vs.length; i++){
									if(i > 0){
										buff.append(',');
									}
									buff.append(vs[i][0]);
 									buff.append('=');
									buff.append(vs[i][1]);
								}
							}
							labe.setText(buff.append('}').toString());
						}
					}
				};
			}else{;
				ed = null;
			}
			editors[i] = ed;
		}
		return editors;
	}

	void refreshAll(){
		publishDescriptionViewer.refresh();
		refreshButtonState();
		refreshPublishDescriptionText();
	}
	
	void refreshButtonState(){
		boolean enable = publishDescriptionViewer.getTable().getEnabled();
		Element selectedElem = getSelectedElement();
		PublishMappingDocumentProxy docFacade = getPublishDescriptionDocumentProxy();
		int index = docFacade.nodeIndex(selectedElem);
		
		addDescItemButton.setEnabled(enable);
		removeDescItemButton.setEnabled(enable && selectedElem != null);
		
		upDescItemButton.setEnabled(enable && index >= 1);
		downDescItemButton.setEnabled(enable && index < docFacade.getDescriptionCount()-1);
	}
	
	void refreshPublishDescriptionText(){
		Element mappingElem = getSelectedElement();
		if(mappingElem != null){
			Element publishElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			String by = publishElem.getAttribute("by");
				
			String description = registory.getPublishDescription(by);
			publishDescriptionText.setText(
				description != null ?
					description :
					"Specify publisher (" + by + ") is not exist description.");
		} else {
			publishDescriptionText.setText("");
		}
	}
	
	PublishMappingDocumentProxy getPublishDescriptionDocumentProxy(){
		return (PublishMappingDocumentProxy)publishDescriptionViewer.getInput();
	}
	
	Element getSelectedElement(){
		IStructuredSelection sel =
			(IStructuredSelection)publishDescriptionViewer.getSelection();
		return !sel.isEmpty() ? (Element)sel.getFirstElement() : null;
	}
	
	Button createButton(Composite parent, String title){
		final Button button = new Button(parent, SWT.PUSH);
		button.setText(title);
		button.setLayoutData( new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_BEGINNING));
		button.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				handleButtonPressed(button);
			}
		});
		return button;
	}

	/**
	 * create document object from edit target publish descriptjion file
	 * and set document object to ui widgets.
	 */
	void initContent(){
		// create publisher builder document object
		// and error handling.
		String errorMessage = null;
		try{
			Document doc;
			DocumentBuilder docBuilder = docBuilderFac.newDocumentBuilder();
			if( editTargetResource.exists() ){
				doc = docBuilder.parse(editTargetResource.getContents());
			}else{
				InputSource source = new InputSource(new StringReader("<?xml version=\"1.0\"?>\n\n<publishers>\n</publishers>"));
				doc = docBuilder.parse(source);
			}
			publishDescriptionViewer.setInput(new PublishMappingDocumentProxy(doc));
			
		} catch (SAXException e) {
			errorMessage = "configuration file \".publish\" is invalid.";
		} catch (IOException e) {
			errorMessage = "can not read \".publish\".";
		} catch (ParserConfigurationException e) {
			errorMessage = "publish file read error that is parser configuration exception.";
			WebpubUIPlugin.handleException(e);
		} catch (CoreException e) {
			errorMessage = "can not found publish configuration file \".publish\", (that file is not synchronize in workbench resource, or that file is not exist or not local)";
			WebpubUIPlugin.handleException(e);
		}
		
		Table table = publishDescriptionViewer.getTable();
		for (int i = 0, len = table.getColumnCount(); i < len; i++) {
			TableColumn col = table.getColumn(i);
			col.pack();
			if( col.getWidth() > 100){
				col.setWidth(100);
			}
		}

		boolean enabled = (errorMessage == null);
		setEnabled(enabled);
		if( statusListener != null ){
			if( !enabled){
				statusListener.statusChanged( createStatus( IStatus.ERROR, errorMessage) );
			}else{
				statusListener.statusChanged( createStatus( IStatus.OK, null) );
			}
		}
	}
	
	static IStatus createStatus(int severity, String message){
		return new Status(
			severity,
			WebpubUIPlugin.ID_PLUGIN,
			IStatus.OK,
			message == null ? "" : message,
			null);
	}
	
	public void setEnabled(boolean bool){
		Control[] controls = base.getChildren();
		for (int i = 0; i < controls.length; i++) {
			controls[i].setEnabled(bool);
		}
	}

	/**
	 * Store the edited document instance.
	 */
	public boolean distribute() throws CoreException {
		Document doc = getPublishDescriptionDocumentProxy().getDocument();
		PublishMappingWriter.store( editTargetResource.getLocation().toFile(), doc);
		editTargetResource.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor() );
		return true;
	}

	public Composite getControl() {
		return base;
	}

}
