/*
 * $Id: WebProjectStructureBlock.java,v 1.8 2004/06/07 06:46:16 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */

package com.narucy.webpub.ui.properties;
import java.util.HashMap;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.dialogs.*;

import com.narucy.webpub.core.*;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * <p>
 * WebProjectPathStructureBlock roles to create ui components
 * for web project directory structure.
 * 
 * <p>
 * It is not depend on any container classes, only to create.
 * It used in WebProjectStructureWizardPage, WebProjectStructurePropertyPage.
 */
public class WebProjectStructureBlock {

	IStatusChangeListener statusListener;
	WebProject webProject;

	Composite base;
	HashMap controls = new HashMap();
	
	IStatus status = null;
	boolean enableListener = true;
	
	public WebProjectStructureBlock(IStatusChangeListener statusListener) {
		this.statusListener = statusListener;
	}
	
	/**
	 * Return new Composite instance.
	 */
	public Composite createControl(Composite parent, IProject project) throws CoreException {
		webProject = project != null ? (WebProject)project.getNature(WebProject.ID_NATURE) : null;
		base = new Composite(parent,SWT.NONE);
		
		// create widgets
		GridLayout layout = new GridLayout(2, false);
		layout.verticalSpacing = 9;
		base.setLayout(layout);

		createText(WebProject.HTSOURCES_FOLDER, "Hypertext sources folder:");
		createText(WebProject.SCRIPTS_FOLDER, "&Scripts folder:");
		createText(WebProject.PUBLISH_FOLDER,  "&Publish folder:");
		
		if( webProject != null){
			setFolderData(
				webProject.getString(WebProject.HTSOURCES_FOLDER),
				webProject.getString(WebProject.SCRIPTS_FOLDER),
				webProject.getString(WebProject.PUBLISH_FOLDER) );
		}

		return base;
	}
	
	public void forceFocus(){
		getTextControl(WebProject.HTSOURCES_FOLDER).forceFocus();
	}
	
	void setPath(String key, IPath fullPath){
		Text text = getTextControl(key);
		if( webProject != null && webProject.getProject().getFullPath().isPrefixOf(fullPath) ){
			text.setText(fullPath.removeFirstSegments(1).toString());
		}else{
			text.setText(fullPath.toString());
		}
	}
	
	void callContainerSelectionDialog(final String key){
		// set initial container sellection.
		IContainer container = null;
		if(webProject != null){
			container = webProject.pathToContainer(getPath(key));
		}
		
		// user selection link target resource
		ContainerSelectionDialog dialog =
			new ContainerSelectionDialog(
				base.getShell(),
				container,
				true,
				"Select container");
		
		if(!key.equals(WebProject.PUBLISH_FOLDER)){
			dialog.setValidator(new ISelectionValidator() {
				public String isValid(Object element) {
					IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember((IPath)element);
					if(!(res instanceof IFolder)){
						return "must be specify a folder";
					}
					if(webProject != null){
						IProject proj = webProject.getProject();
						if(!res.getProject().equals(proj)){
							return "must be specify an under folder for " + proj.getName();
						}
					}
					return null;
				}
			});
		}
		
		if( dialog.open() == Window.OK){
			Object[] result = dialog.getResult();
			setPath(key, (IPath)result[0]);
		}
	}
	
	void createText(final String key, String textData){
		new Label(base, SWT.NULL).setText(textData);
		
		Composite inputBase = new Composite(base, SWT.NONE);
		GridLayout gl = new GridLayout(2,false);
		gl.marginHeight = 0;
		inputBase.setLayout(gl);
		inputBase.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Text text = new Text(inputBase, SWT.BORDER | SWT.SINGLE);
		text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		text.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event event) {
				checkValues();
			}
		});
		
		Button button = new Button(inputBase, SWT.PUSH);
		button.setText("Browse...");
		button.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event ev) {
				callContainerSelectionDialog(key);
			}
		});
		controls.put(key, new Object[]{text, button});
	}
		
	public void setFolderData(String htFolderName, String scriptFolderName, String publishFolderName){
		enableListener = false;
		getTextControl(WebProject.HTSOURCES_FOLDER).setText(htFolderName);
		getTextControl(WebProject.SCRIPTS_FOLDER).setText(scriptFolderName);
		getTextControl(WebProject.PUBLISH_FOLDER).setText(publishFolderName);
		enableListener = true;
		
		checkValues();
	}
	
	public Text getTextControl(String key){
		return (Text)((Object[])controls.get(key))[0];
	}
	
	public String getPath(String key){
		return getTextControl(key).getText();
	}
	
	public Button getButton(String key){
		return (Button)((Object[])controls.get(key))[1];
	}

	void checkValues(){
		if(!enableListener){
			return;
		}
		
		String htFolderName = getPath(WebProject.HTSOURCES_FOLDER);
		String scriptFolderName = getPath(WebProject.SCRIPTS_FOLDER);
		String publishFolderName = getPath(WebProject.PUBLISH_FOLDER);
		
		if(htFolderName.length() == 0){
			updateMessage("Soruce folder path is empty", IStatus.ERROR);
		}else if(scriptFolderName.length() == 0){
			updateMessage("Script folder path is empty", IStatus.ERROR);
		}else if(publishFolderName.length() == 0){
			updateMessage("Publish folder path is empty", IStatus.ERROR);
		}else if(publishFolderName.equals(htFolderName)){
			updateMessage("Source folder and publish folder must separeate", IStatus.ERROR);
		}else if(htFolderName.indexOf(0) == '/'){
			updateMessage("Source folder must include oneself project", IStatus.ERROR);
		}else if(scriptFolderName.indexOf(0) == '/'){
			updateMessage("Script folder must include oneself project", IStatus.ERROR);
		}else if(scriptFolderName.equals(htFolderName)){
			updateMessage("Recommendation script folder and source folder are each separate", IStatus.WARNING);
		}else if(scriptFolderName.equals(publishFolderName)){
			updateMessage("Recommendation script folder and publish folder are each separate", IStatus.WARNING);
		}else{
			updateMessage("", IStatus.OK);
		}
	}
	
	void updateMessage(String msg, int severity){
		if(status == null || status.getSeverity() != severity || !status.getMessage().equals(msg)){
			status = new Status(severity, WebpubUIPlugin.ID_PLUGIN, IStatus.OK, msg, null);
			statusListener.statusChanged(status);
		}
	}

	public Composite getControl() {
		return base;
	}

}
