 /*
 * $Id: HTPreView.java,v 1.17 2004/06/07 06:46:22 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.views;

import java.io.File;
import java.net.*;
import java.util.Arrays;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.*;
import org.eclipse.jface.resource.StringConverter;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.publish.SourceFileSeeker;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.actions.RepublishAction;

/**
 * HTPreView role is to define of browser selection 
 * and indication a HTML document and display Title.
 */
public class HTPreView extends ViewPart {

	final static String
		LAST_BROWSE_URL = "last_browse_url",
		KEY_URL_HISTORY = "url_history",
		AUTO_PUBLISH = "auto_publish",
		LINK_WITH_EDITOR = "link_with_editor";

	IMemento memento;
	int maxUrlHistoryCount = 10;
	
	Browser browser;
	String lastSetUrl;
	int loadingProgress = -1;

	Combo addressBar;
	
	Action
		autoPublishAction,
		republishAction,
		openEditorAction,
		backAction,
		forwardAction,
		stopAction;
	
	String pageTitle = null;
	HTPreviewLinkWithEditorManager linkWithEditorMgr;
	
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		this.memento = memento;
	}

	/**
	 * Returns publish original file if url file is publishable.
	 * If other case, return null.
	 */
	public IFile getCurrentPublishFrom() {
		try{
			if (lastSetUrl != null) {
				return SourceFileSeeker.findSource(lastSetUrl);
			}
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
		return null;
	}
	
	/**
	 * Call back from eclipse resources. Reload web browser when changed resource is
	 * current browse file reloational project.
	 */
	IResourceChangeListener resourceListener = new IResourceChangeListener() {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			if(delta !=  null){
				handleResourceChanged(delta);
			}
		}
	};

	void handleResourceChanged(IResourceDelta delta) {
		final boolean[] needReload = { false };
		
		try {
			delta.accept(new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta d) throws CoreException {
					if ( isRequireReload(d.getResource()) ) {
						needReload[0] = true;
						return false;
					}
					return true;
				}
			});
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
		if(needReload[0]){
			IWorkbenchPartSite site = getSite();
			if(site != null){
				site.getShell().getDisplay().asyncExec(new Runnable() {
					public void run() {
						if(!browser.isDisposed()){
							browser.refresh();
						}
					}
				});
			}
		}
	}
	
	boolean isRequireReload(IResource r) throws CoreException{
		final IFile from = getCurrentPublishFrom();
		if(r instanceof IFile && from != null){
			WebProject wp = (WebProject)from.getProject().getNature(WebProject.ID_NATURE);
			if( wp == null){
				return from.getProject().equals(r.getProject());
			}else{
				IContainer pubFolder = wp.getFolder(WebProject.PUBLISH_FOLDER);
				for(IContainer c = r.getParent(); !(c instanceof IWorkspaceRoot); c = c.getParent() ){
					if(c.equals(pubFolder)){
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Create controls.
	 */
	public void createPartControl(Composite parent) {
		// create control
		Composite base = new Composite(parent, SWT.NONE);
		GridLayout gl = new GridLayout();
		gl.marginWidth = 0;
		gl.marginHeight = 0;
		base.setLayout(gl);
		
		// create address bar
		GridData gd = new GridData(GridData.VERTICAL_ALIGN_CENTER);
		gd.horizontalIndent = 2;
		
		addressBar = new Combo(base, SWT.BORDER | SWT.DROP_DOWN);
		addressBar.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		// create browser
		browser = new Browser(base, SWT.NONE);
		browser.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// init menu.
		linkWithEditorMgr = new HTPreviewLinkWithEditorManager(this);
		
		autoPublishAction = new Action( "Auto Publish", IAction.AS_CHECK_BOX){
			public void run(){
			}
		};
		autoPublishAction.setImageDescriptor(WebpubUIPlugin.getImageDescriptor("clcl16/autopublish.gif"));
		autoPublishAction.setToolTipText("Auto Publish");
		
		// create actions
		republishAction = new Action("Publish", WebpubUIPlugin.getImageDescriptor("clcl16/republish.gif")) {
			public void run() {
				IFile republishTarget = getCurrentPublishFrom();
				if (republishTarget != null) {
					RepublishAction act = new RepublishAction();
					act.init(getSite().getWorkbenchWindow());
					act.selectionChanged(this, new StructuredSelection(republishTarget));
					act.run(this);
				}
			}
		};
		republishAction.setToolTipText("Publish");
		
		openEditorAction = new Action("Open Editor", WebpubUIPlugin.getImageDescriptor("clcl16/open_editor.gif")) {
			public void run() {
				IFile f = getCurrentPublishFrom();
				try {
					IDE.openEditor(getSite().getPage(), f, true);
				} catch (PartInitException e) {
					WebpubUIPlugin.handleException(e);
				}
			}
		};
		openEditorAction.setToolTipText("Open Editor");
		
		backAction = new Action("Back", WebpubUIPlugin.getImageDescriptor("clcl16/backward_nav.gif")){
			public void run(){
				browser.back();
			}
		};
		backAction.setDisabledImageDescriptor(WebpubUIPlugin.getImageDescriptor("dlcl16/backward_nav.gif"));
		backAction.setToolTipText("Back");
		
		forwardAction = new Action("Forward", WebpubUIPlugin.getImageDescriptor("clcl16/forward_nav.gif")){
			public void run(){
				browser.forward();
			}
		};
		forwardAction.setDisabledImageDescriptor(WebpubUIPlugin.getImageDescriptor("dlcl16/forward_nav.gif"));
		forwardAction.setToolTipText("Foward");
		
		stopAction = new Action("Stop", WebpubUIPlugin.getImageDescriptor("clcl16/stop.gif")){
			public void run(){
				browser.stop();
			}
		};
		stopAction.setToolTipText("Stop");
		
		IActionBars bars = getViewSite().getActionBars();

		// toolbar and menu bar initialize
		IToolBarManager toolBar = bars.getToolBarManager();
		toolBar.add(autoPublishAction);
		toolBar.add(republishAction);
		toolBar.add(backAction);
		toolBar.add(forwardAction);
		toolBar.add(stopAction);
		toolBar.add(openEditorAction);
		toolBar.add(linkWithEditorMgr.getLinkWithEditorAction());
		
		IMenuManager menuMgr = bars.getMenuManager();
		menuMgr.add(autoPublishAction);
		
		// handle global action.
		bars.setGlobalActionHandler(ActionFactory.CUT.getId(), new Action() {
			public void run() {
				if( addressBar.isFocusControl()){
					addressBar.cut();
				}
			}
		});
		bars.setGlobalActionHandler(ActionFactory.COPY.getId(), new Action() {
			public void run() {
				if( addressBar.isFocusControl()){
					addressBar.copy();
				}
			}
		});
		bars.setGlobalActionHandler(ActionFactory.PASTE.getId(), new Action() {
			public void run() {
				if( addressBar.isFocusControl()){
					addressBar.paste();
				}
			}
		});
		
		// set selection for external web browser from "Run > External Tools".
		getSite().setSelectionProvider(new HTPreviewSelectionProvider(this));
		
		// listener setting
		addressBar.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				if (e.character == '\r' || e.character == '\n') {
					enterUrl();
				}
			}
		});
		addressBar.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				enterUrl();
			}
		});

		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceListener);
		browser.addProgressListener(new ProgressListener() {
			public void changed(ProgressEvent event) {
				if (0 < event.total){
					loadingProgress = event.current * 100 / event.total;
					if(loadingProgress < 100){
						setContentDescription("Loading... " + loadingProgress + "%");
					}else{
						// MS OLE or SWT Browser widget bug? not called completed method.
						completed(event);
					}
				}
			}

			public void completed(ProgressEvent event) {
				loadFinished();
			}
		});
		browser.addTitleListener(new TitleListener() {
			public void changed(TitleEvent event) {
				pageTitle = event.title;
				if(loadingProgress == -1){
					setContentDescription(pageTitle);
				}
			}
		});
		
		browser.addLocationListener(new LocationAdapter() {
			public void changed(LocationEvent event) {
				browse(event.location);
			}
		});
		
		browser.addStatusTextListener(new StatusTextListener() {
			public void changed(StatusTextEvent event) {
				setWorkbenchPageStatusMessage(event.text);
			}
		});

		// set memento values.
		if(memento != null){
			String histories = memento.getString(KEY_URL_HISTORY);
			if(histories != null){
				addressBar.setItems(memento.getString(KEY_URL_HISTORY).split("\\s+"));
			}
			autoPublishAction.setChecked(
				"true".equals( memento.getString(AUTO_PUBLISH) ) );
			
			linkWithEditorMgr.getLinkWithEditorAction().setChecked(
				"true".equals( memento.getString(LINK_WITH_EDITOR)) );
			
			browse(memento.getString(LAST_BROWSE_URL));
		}
		
		refreshActionsState();
	}
	
	void loadFinished(){
		setWorkbenchPageStatusMessage(null);
		refreshActionsState();
		
		// to provide selection for external tools.
		getSite().getSelectionProvider().setSelection(new StructuredSelection(addressBar.getText()));
		setContentDescription(pageTitle);
		loadingProgress = -1;
	}
	
	protected void setContentDescription(String newTitle){
		super.setContentDescription(newTitle != null ? newTitle.replaceAll("&", "&&") : "Untitled");
	}
	
	void setWorkbenchPageStatusMessage(String message){
		getViewSite().getActionBars().getStatusLineManager().setMessage(message);
	}
	
	public boolean isLoading(){
		return loadingProgress != -1;
	}
	
	public int getLoadingProgress(){
		return loadingProgress;
	}
	
	void enterUrl(){
		enterUrl(addressBar.getText());
	}
	
	/**
	 * Browse specify url and add to url history.
	 */
	public void enterUrl(String url){
		if(!isContainURLHistory(url) ){
			addressBar.add(url.toString(), 0);
			if( addressBar.getItemCount() > maxUrlHistoryCount){
				addressBar.remove(addressBar.getItemCount()-1);
			}
		}
		browse(url);
	}

	void refreshActionsState() {
		if(lastSetUrl != null && lastSetUrl.length() > 0){
			try{
				IFile pubFrom = getCurrentPublishFrom();
				republishAction.setEnabled(pubFrom != null && new URL(lastSetUrl).getProtocol().equals("file"));
				openEditorAction.setEnabled(pubFrom != null);
			}catch(MalformedURLException e){
				republishAction.setEnabled(false);
				openEditorAction.setEnabled(false);
			}
		}
		backAction.setEnabled(browser.isBackEnabled());
		forwardAction.setEnabled(browser.isForwardEnabled());
		
		linkWithEditorMgr.handleWebBorwserUrlChanged();
	}
	
	public String[] getHistory(){
		return addressBar.getItems();
	}

	boolean isContainURLHistory(String url){
		String[] history = getHistory();
		Arrays.sort(history);
		return Arrays.binarySearch(history, url) >= 0;
	}

	public boolean isPreviewble(IFile file) {
		String ext = file.getFileExtension();
		if(ext == null){
			return false;
		}
		String[] previewExts = StringConverter.asArray(
			WebpubUIPlugin.
				getDefault().
				getPreferenceStore().
				getString(WebpubUIPlugin.HTPREVIEW_EXTENSIONS));
		Arrays.sort(previewExts);
		return Arrays.binarySearch(previewExts, ext) >= 0;
	}
		
	public void browse(String urlStr) {
		if(browser.isDisposed()){
			return;
		}
		
		boolean available = urlStr != null;
		browser.setEnabled(available);
		
		if(available){
			loadingProgress = 0;
			
			urlStr = adjustUrlString(urlStr);
			if(!urlStr.equals(lastSetUrl)){
				lastSetUrl = urlStr;
				
				boolean samePoint = false;
				try {
					samePoint = new URL(urlStr).equals(new URL(browser.getUrl()));
				} catch (MalformedURLException e) {
				}
				if(!samePoint){
					pageTitle = null;
					browser.setUrl(urlStr);
				}else{
					loadingProgress = -1;
				}
			}
			if(!urlStr.equals(addressBar.getText())){
				// if auto publish is checked, browse url changed when force republish the specify url.
				if(autoPublishAction.isChecked()){
					try {
						final IFile publishFrom = SourceFileSeeker.findSource(urlStr);
						if(publishFrom != null){
							RepublishAction act = new RepublishAction();
							act.init(getSite().getWorkbenchWindow());
							act.selectionChanged(null, new StructuredSelection(publishFrom));
							act.run();
						}
					} catch (CoreException e) {
						WebpubUIPlugin.handleException(e);
					}
				}
				addressBar.setText(urlStr);
			}
		} else {
			lastSetUrl = null;
			browser.setText("");
			setContentDescription("An control is not an available");
			refreshActionsState();
		}
	}
	
	public String adjustUrlString(String urlStr){
		URL u = null;
		try {
			File f = new File(urlStr);
			if(f.exists()){
				u = f.toURL();
			}else{
				u = new URL(urlStr);
			}
		} catch (MalformedURLException e1) {
			try {
				u = new URL("http://" + urlStr);
			} catch (MalformedURLException x) {
			}
		}
		return u != null ? u.toString() : urlStr;
	}

	public void setFocus() {
	}

	public String getUrl(){
		return lastSetUrl;
	}
	
	public HTPreviewLinkWithEditorManager getLinkWithEditorManager(){
		return linkWithEditorMgr;
	}

	public void saveState(IMemento memento) {
		memento.putString(
				AUTO_PUBLISH,
				String.valueOf(autoPublishAction.isChecked()));
		memento.putString(
			LINK_WITH_EDITOR,
			String.valueOf(linkWithEditorMgr.getLinkWithEditorAction().isChecked()));
		
		memento.putString(LAST_BROWSE_URL, lastSetUrl);
		String[] urls = addressBar.getItems();
		memento.putString(KEY_URL_HISTORY, WebpubUIPlugin.join(urls));
	}

	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
		linkWithEditorMgr.dispose();
		super.dispose();
	}


}