/*
 * $Id: HTPreviewLinkWithEditorManager.java,v 1.17 2004/06/07 06:46:22 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.views;

import java.io.IOException;
import java.net.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.action.*;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.*;
import org.eclipse.ui.ide.IDE;

import com.narucy.webpub.core.*;
import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.ui.*;

/**
 * PreviewAutoActivedManager provides editor part bridge.
 * 
 */
public class HTPreviewLinkWithEditorManager {

	final public static String
		M_PREVIEW_TARGETS = "previewTargets",
		M_PREVIEW_BEHAVIOR = "previewBehavior";

	HTPreView preview;
	IMenuManager menuMgr;
	
	MnemonicNamer mnemonicNamer = new MnemonicNamer() {
		public String toLabel(Object obj) {
			IFile file = (IFile)obj;
			String name = file.getName();
			String fullPath = file.getFullPath().toString().substring(1);
			int nameLen = name.length();
			int fullLen = fullPath.length();
			if( nameLen + fullLen > 32){
				fullPath = fullPath.substring(0, fullLen-nameLen) + "...";
			}
			return name + "  [" + fullPath + "]";
		}
	};

	HashMap fileActions = new HashMap();
	Action linkWithEditorAction;

	IResourceChangeListener resourceListener = new IResourceChangeListener() {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			if(delta != null){
				handleResourceChanged(delta);
			}
		}
	};
	
	/**
	 * Call back from part.
	 */
	IPartListener partListener = new IPartListener() {
		public void partBroughtToTop(IWorkbenchPart part) {}
		public void partDeactivated(IWorkbenchPart part) {}


		public void partClosed(IWorkbenchPart part) {
			if( part instanceof IEditorPart){
				unregistObserveEditor( (IEditorPart)part);
			}
		}

		public void partOpened(IWorkbenchPart part) {
			if( part instanceof IEditorPart){
				registObserveEditor( (IEditorPart)part);
			}
		}
		
		public void partActivated(IWorkbenchPart part) {
			if(isLinkWithEditor() && part instanceof IEditorPart){
				IFile editFile = getFile( (IEditorPart)part);
				if(editFile != null){
					changeBrowseFile(editFile);
				}
			}
		}
	};

	/**
	 * Construct PrivewAutoActivedManager.
	 */
	public HTPreviewLinkWithEditorManager(HTPreView view) {
		preview = view;
		menuMgr = preview.getViewSite().getActionBars().getMenuManager();
		menuMgr.add(new Action("&Deselect") {
			public void run() {
				preview.browse(null);
			}
		});

		// addition editor menu item. 
		menuMgr.add( new Separator(M_PREVIEW_TARGETS));
		// create auto mate preview changer action
		menuMgr.add( new Separator(M_PREVIEW_BEHAVIOR));
		linkWithEditorAction = new Action("&Link with Editor", IAction.AS_CHECK_BOX ){
			public void run(){
				handleLinkWithEditorCheck();
			}
		};
		linkWithEditorAction.setToolTipText("Link with Editor");
		linkWithEditorAction.setImageDescriptor(WebpubUIPlugin.getImageDescriptor("clcl16/synced.gif"));
		
		menuMgr.add(linkWithEditorAction);
		
		// init for current open editors 
		IEditorReference[] refs = preview.getSite().getPage().getEditorReferences(); 
		for (int i = 0; i < refs.length; i++) {
			registObserveEditor( refs[i].getEditor(true) );
		}

		view.getSite().getPage().addPartListener(partListener);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceListener);
	}
	
	void handleResourceChanged(IResourceDelta delta){
		final ArrayList removes = new ArrayList();
		try{
			delta.accept(new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta d) throws CoreException {
					IResource r = d.getResource();
					if(r instanceof IFile && d.getKind() == IResourceDelta.REMOVED && fileActions.containsKey(r)){
						removes.add(r);
					}
					return true;
				}
			});
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
		if(removes.size() > 0){
			final Display disp = preview.getSite().getShell().getDisplay();
			disp.asyncExec(new Runnable() {
				public void run() {
					if(!disp.isDisposed()){
						for(int i=0; i<removes.size(); i++){
							undefinePreviewAction((IFile)removes.get(i));
						}
					}
				}
			});
		}
	}

	void handleLinkWithEditorCheck(){
		if(!linkWithEditorAction.isChecked()){
			return;
		}
		
		IFile[] registedFiles = getRegistedFiles();
		if(registedFiles.length > 0){
			IFile editFile = getFile( preview.getSite().getPage().getActiveEditor() );
			
			for (int i=0; i<registedFiles.length; i++) {
				IFile r = registedFiles[i];
				if( editFile.equals(r) ){
					changeBrowseFile(r);
					break;
				}
			}
		}else{
			// link with editor button pressed, but editor is not opened.
			preview.browse(null);
		}
	}
	
	public void handleWebBorwserUrlChanged() {
		try{
			IFile browseFile = preview.getCurrentPublishFrom();
			if(isLinkWithEditor()){
				// auto activiation browser url changed, notify workbench part.
				if(fileActions.containsKey(browseFile) ){
					IDE.openEditor(preview.getSite().getPage(), browseFile, true);
				}
			}
			// update open action radio state
			if(browseFile == null){
				String url = preview.getUrl();
				if(url != null){
					browseFile = SourceFileSeeker.findSource(url);
				}
			}
			IFile[] fs = getRegistedFiles();
			for (int i=0; i<fs.length; i++) {
				IFile f = fs[i];
				getBrowseAction(f).setChecked(f.equals(browseFile));
			}
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
	}
	
	public IFile[] getRegistedFiles(){
		return (IFile[])fileActions.keySet().toArray(new IFile[fileActions.size()]);
	}

	public void changeBrowseFile(IFile openFile){
		if(getBrowseAction(openFile) != null && preview.isPreviewble(openFile)){
			IFile[] pubToResource = getPublishFileBySource(openFile);
			if(pubToResource != null){
				IPath localPath = pubToResource[0].getLocation();
				try {
					preview.browse(localPath.toFile().toURL().toString());
				} catch (MalformedURLException ex) {
					WebpubUIPlugin.handleException(ex);
				}
			}
		}
	}
	
	void checkOpenAction(IFile openFile){
		IFile[] fs = getRegistedFiles();
		for (int i = 0; i < fs.length; i++) {
			IFile f = fs[i];
			getBrowseAction(f).setChecked( f.equals(openFile) );
		}
	}

	public void definePreviewAction(final IFile file){
		if(!fileActions.containsKey(file)){
			Action action = new Action(mnemonicNamer.getLabel(file), IAction.AS_RADIO_BUTTON){
				public void run(){
					changeBrowseFile(file);
				}
			};
			ActionContributionItem actionItem = new ActionContributionItem(action);
			fileActions.put(file, actionItem);
			menuMgr.appendToGroup(M_PREVIEW_TARGETS , actionItem);
		}
	}
	
	public IAction getBrowseAction(IFile file){
		return fileActions.containsKey(file) ?
			((ActionContributionItem)fileActions.get(file)).getAction() :
			null;
	}

	static IFile getFile(IEditorPart editor){
		IEditorInput input = editor.getEditorInput();
		return (input instanceof IFileEditorInput) ? ((IFileEditorInput)input).getFile() : null;
	}

	void registObserveEditor(IEditorPart editor){
		IFile file = getFile(editor);
		if(file != null && preview.isPreviewble(file)){
			definePreviewAction(file);
		}
	}
	
	void unregistObserveEditor(IEditorPart editor){
		IFile file = getFile(editor);
		if(file != null){
			undefinePreviewAction(file);
		}
		if( isLinkWithEditor() && fileActions.isEmpty() ){
			preview.browse(null);
		}
	}
	
	void undefinePreviewAction(IFile file){
		ActionContributionItem actionItem = (ActionContributionItem)fileActions.remove(file);
		if(actionItem != null){
			menuMgr.remove(actionItem);
		}
		mnemonicNamer.release(file);
	}
	
	static URL getPublishUrl(IFile sourceFile){
		if(sourceFile != null){
			IFile[] pubToFile = getPublishFileBySource(sourceFile);
			if(pubToFile != null){
				try {
					return pubToFile[0].getLocation().toFile().toURL();
				} catch (MalformedURLException e) {
					WebpubUIPlugin.handleException(e);
				}
			}
		}
		return null;
	}
	
	static IFile[] getPublishFileBySource(IFile file){
		try{
			WebProject wp = (WebProject)file.getProject().getNature(WebProject.ID_NATURE);
			if(wp != null){
				IContainer publishFolder = wp.getFolder(WebProject.PUBLISH_FOLDER);
				
				PublishDescription desc = wp.getPublishDescriptionFactory().create(file);
				if(desc != null){
					IPath pubTo = desc.getPublishTo();
					
					if(pubTo instanceof Wildcard){
						IFile[] files = WebProject.getFileMembers(publishFolder);
						ArrayList dist = new ArrayList();
						for (int i = 0; i < files.length; i++) {
							IFile f = files[i];
							IPath relPath = f.getFullPath().removeFirstSegments(publishFolder.getFullPath().segmentCount());
							if( ((Wildcard)pubTo).match(relPath) ){
								dist.add(f);
							}
						}
						return dist.size() > 0 ? (IFile[]) dist.toArray(new IFile[dist.size()]) : null;
					}else if(pubTo != null){
						return new IFile[]{ wp.getFolder(WebProject.PUBLISH_FOLDER).getFile(pubTo) };
					}
				}
			}
		} catch (IllegalConfigurationException e) {
			WebpubUIPlugin.handleException(e);
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		} catch (IOException e) {
			WebpubUIPlugin.handleException(e);
		}
		return null;
	}
	
	boolean isLinkWithEditor(){
		return linkWithEditorAction != null && linkWithEditorAction.isChecked();
	}

	public Action getLinkWithEditorAction() {
		return linkWithEditorAction;
	}

	public void dispose(){
		preview.getSite().getPage().removePartListener(partListener);
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
	}
}
