/*-------------------------------------------------------------------------*/
/*  J3W ver 6.43  3D Animation Kit                                         */
/*  j3djob.cpp     4/24/2001                                               */
/*  Copyright (C) 1995 - 2001 Jun Mizutani <mizutani.jun@nifty.ne.jp>      */
/*                      All rights reserved.                               */
/*                                                                         */
/*   This file is part of the J3W 3D Animation Kit, and is covered under   */
/*  the terms of the GNU General Public License, version 2. This file has  */
/*  NO WARRANTY. See file COPYING for copyright details.                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/

#include  <iostream.h>
#include  <fstream.h>
#include  <ctype.h>
#include  "j3djob.h"
#ifndef WS
#include  "xkey.h"
#endif //WS
#include  "misc3d.h"
#include  "pal256.h"

TJob::TJob(int code_size, int max_proc, TSpaceH3D* p_space, TScreen *sc)
         : Code(code_size), numProc(0), Count(0), maxProc(max_proc),
           pSpace(p_space), pScreen(sc)
{
    process = new TProcess *[max_proc];
    CommonMem = new long [MAXCOMMON];
#ifdef WS
	int i;
    for (i=0; i<maxProc; i++) process[i] = NULL;
#else //WS
    for (int i=0; i<maxProc; i++) process[i] = NULL;
#endif //WS
    StrStat = 0;
    Cursor = 0;
    StrColor = 15;
#ifdef WS
    for (i=0; i<MAX_STRING; i++) ScreenString[i] = ' ';
#else //WS
    for (int i=0; i<MAX_STRING; i++) ScreenString[i] = ' ';
#endif //WS
    ScreenString[1000] = char(0);
#ifdef WS
    for (i=0; i<MAXLINES; i++) GrLines[i].stat = 0;
#else //WS
    for (int i=0; i<MAXLINES; i++) GrLines[i].stat = 0;
#endif //WS
    g_mode = 0;
    MidiInUse = 0;
}

TJob::~TJob() {
    for (int i=0; i<maxProc; i++) delete_process(i);
    delete [] process;
    delete CommonMem;
#ifdef MIDI_SUPPORT
    if (MidiInUse) midiOutClose(hMidi);
#endif
}

void TJob::common_func(long delta)
{
    pSpace->display(delta);
    for(int i=0; i<MAXLINES; i++)
        if (GrLines[i].stat) pScreen->line(GrLines[i].p0, GrLines[i].p1,
                                           GrLines[i].color);
    if (StrStat)  pScreen->write_text(0,0, StrColor, ScreenString);
    pScreen->update();
}

long TJob::job_scheduler()
{
    long cur_time = aTimer.GetTime();

    if (numProc > 0) {
        for(int i = 0; i<maxProc; i++) {

            if (process[i] != NULL )
                if ( process[i]->get_new()) process[i]->set_new(0);
                else {
                    current_process = i;
                    int r = process[i]->Run(cur_time - last_time);
                    if (r == 0)  delete_process(i);
                }
        }

        if (g_mode) common_func(cur_time - last_time);

        Count++;
    }
    last_time = cur_time;
    return numProc;
}

int TJob::new_process(int maxData, int StartPos, int on, long* reg)
{
    if (numProc < maxProc) {
        int i = 0;
        while ((i<(maxProc-1)) && (process[i] != NULL)) i++ ;
        process[i] = new TProcess(this, maxData, StartPos, &Code,
                                       pSpace, reg, on);
        numProc++;
        return 1;
    } else return 0;
}

void TJob::delete_process(int i)
{
    if (process[i] != NULL) {
        numProc--;
        if (process[i]->GetObjNo() != -1)
            pSpace->delete_object(process[i]->GetObjNo());
        delete process[i];
        process[i] = NULL;
    }
}

THObj3D* TJob::GetObject(int id)
{
    THObj3D* po = NULL;

    for (int i = 0; i < maxProc; i++)
        if (process[i] != NULL )
            if (process[i]->GetID() == id)
                po = process[i]->GetObject();
    return po;
}

void TJob::SendMessage(int id, long msg)
{
     for (int i = 0; i < maxProc; i++) {

         if (process[i] != NULL )
             if ( ( (process[i]->GetID() == id) || (id == PROC_ALL) ) &&
                  (process[i]->GetMessage() < msg) )
                 process[i]->SetMessage(msg);
     }
}

void TJob::SendMessageTo(int pno, long msg)
{

     if (process[pno] != NULL )
         if (process[pno]->GetMessage() < msg) process[pno]->SetMessage(msg);
}

int TJob::LoadObjFile(char *FileName)
{

  return Code.LoadObjFile(FileName);
}

int TJob::SaveObjFile(char *FileName)
{
  return Code.SaveObjFile(FileName);
}

void TJob::StartTimer()
{
  aTimer.StartTimer();     /* ޡƵư */
  last_time =  aTimer.GetTime();
}

long TJob::GetTime()
{
    return  aTimer.GetTime();
}

void TJob::GrLine(int n, Vector2D pa, Vector2D pb, int color)
{
    if ((n < MAXLINES) && (n>=0)) {
        if (color > 31) {
            GrLines[n].color = color;
        } else if (color > 15) {
            if (color < 43) {
                GrLines[n].color = ColNum * ColWidth + 10 + color - 16;
            }
        } else GrLines[n].color = color * ColWidth + 20;
        GrLines[n].p0 = pa;
        GrLines[n].p1 = pb;
        GrLines[n].stat = 1;
    }
}

void TJob::GrLineClear(int n)
{
   if ((n < MAXLINES) && (n >= 0)) GrLines[n].stat = 0;
}

void TJob::SetEye(THObj3D* eye)
{
    pSpace->set_eye(eye);
}

void TJob::SetGraph()
{
    if (g_mode == 0) {
        aTimer.PauseTimer();
        g_mode = 1;
        pScreen->set_mode(1);
        aTimer.RestartTimer();
#ifdef WS
//DEVELOP...
#else //WS
        InitKeyTable((ScrnX *)pScreen);
#endif //WS
    }
}

void TJob::SetMode(int m)
{
  if (m == 1) pScreen->set_mode(1);
  if (m == 2) pScreen->set_mode(2);
}

void TJob::ResetGraph()
{
    if (g_mode == 1) {
        g_mode = 0;
        pScreen->restore_mode();
#ifdef WS
//DEVELOP...
#else //WS
        InitKeyTable(NULL);
#endif //WS
    }
}

void TJob::SetBackGroundColor(int bg)
{

    SetPaletteColor(-1, (bg & 0xF00) >> 4, bg & 0xF0,
                         (bg << 4) & 0xF0, 0, 0, 0);
    if (g_mode) {
      pScreen->SetPaletteEntry(255, (bg << 4) & 0xF000, (bg << 8 )& 0xF000,
                                      (bg << 12) & 0xF000);
        pScreen->set_BG_Color(255);
    }
}

void TJob::SetZoom(int angle)
{
    pSpace->set_angle(angle);
}

long TJob::Distance(THObj3D* obj1, int id, long &pno, long &cnt) {
     int count  = 0;
     int proc_num = 0;
     long dist1 = -1;
     long dist2;
     Vector vect1 = obj1->GetWorldOrigin();
     for (int i = 0; i < maxProc; i++) {
         if (process[i] != NULL )
             if (process[i]->GetID() == id) {
                 Vector vect2 = process[i]->GetObject()->GetWorldOrigin();
#ifdef WS
                 dist2 = j3w_round(sqrt(
                             double(vect1.x-vect2.x)*double(vect1.x-vect2.x) +
                             double(vect1.y-vect2.y)*double(vect1.y-vect2.y) +
                             double(vect1.z-vect2.z)*double(vect1.z-vect2.z)));
#else //WS
                 dist2 = round(sqrt(
                             double(vect1.x-vect2.x)*double(vect1.x-vect2.x) +
                             double(vect1.y-vect2.y)*double(vect1.y-vect2.y) +
                             double(vect1.z-vect2.z)*double(vect1.z-vect2.z)));
#endif //WS
                 if ((dist1 == -1) || (dist2 < dist1)) {
                     proc_num = i;
                     dist1 = dist2;
                 }
                 count++;
             }
     }
     cnt = count;
     pno = proc_num;
     return dist1;
}

long TJob::NearestVertex(long x,long y, long z, long id, long &nvertex)
{
    double dx, dy, dz;
    long   dist1, dist2;

    dist1 = -1;
    THObj3D* obj = GetObject(id);
    if (obj) {
        for (int j = 0; j < obj->Polygon.Vertex.GetCount(); j++) {
            Vector vect2 = obj->Polygon.Vertex.vert[j]->world;
            dx = x - vect2.x;
            dy = y - vect2.y;
            dz = z - vect2.z;
#ifdef WS
            dist2 = j3w_round(sqrt( dx*dx + dy*dy +dz*dz));
#else //WS
            dist2 = round(sqrt( dx*dx + dy*dy +dz*dz));
#endif //WS
            if ((dist1 == -1) || (dist2 < dist1)) {
                dist1 = dist2;
                nvertex = j;
            }
        }
    }
    return  dist1;
}

void TJob::RelativePosition(THObj3D* obj1, int id, long *reg)
{
     THObj3D* obj2 = GetObject(id);
     Vector vect1, vect2, v;

     if (obj2 != NULL) {
         vect1 = obj1->GetWorldOrigin();
         vect2 = obj2->GetWorldOrigin();
         v = vect2 - vect1;
         v = obj1->GetDirection_local(v);
         reg[RX] = v.x;
         reg[RY] = v.y;
         reg[RZ] = v.z;
         double x = v.x;
         double y = v.y;
         double z = v.z;
         if ((v.x == 0) && (v.y == 0)) reg[RH] = 0;
         else reg[RH] = DEG(atan2(y, x));
         double w = sqrt(x * x + y * y);
         if ((v.z == 0) && (w == 0.0)) reg[RP] = 0;
         else reg[RP] = DEG(atan2(-z, w));
         reg[RB] = 0;
     } else reg[RB] = 1;
}

long TJob::Look(THObj3D* obj1, int id, long &head, long &pitch)
{
     THObj3D* obj2 = GetObject(id);
     Vector vect1, vect2;

     if (obj2 != NULL) {
         if (obj1->parent == NULL) vect1 = obj1->Axis.get_origin();
         else vect1 = obj1->GetWorldOrigin();
         if (obj2->parent == NULL) vect2 = obj2->Axis.get_origin();
         else vect2 = obj2->GetWorldOrigin();
         Vector v = vect2 - vect1;
         double x = v.x;
         double y = v.y;
         double z = v.z;
         if ((v.x == 0) && (v.y == 0)) head = 0;
         else head = DEG(atan2(y, x));
         double w = sqrt(x * x + y * y);
         if ((v.z == 0) && (w == 0.0)) pitch = 0;
         else pitch = DEG(atan2(-z, w));
         return 0;
     } else return 1;
}

void TJob::SetCursorPosition(int pos)
{
  if ((pos < MAX_STRING) && (pos >=0))  Cursor = pos;
}

void TJob::StringColor(int col)
{
    if (col > 31) StrColor = col;
    else if ((col > 15) && (col < 43)) {
        if (col < 43) {
          StrColor = ColNum * ColWidth + 20 + col - 16;
        }
    } else StrColor = col * ColWidth + 20;
}

void TJob::WriteString(char *str)
{
    int len = strlen(str);
    if ((len + Cursor) < MAX_STRING) {
        for(int i=0; i<len; i++) ScreenString[i + Cursor] = str[i];
        Cursor += len;
        StrStat = 1;
    }
}

void TJob::ClearString()
{
  for(int i = 0; i < MAX_STRING; i++) ScreenString[i] = ' ';
  ScreenString[MAX_STRING] = char(0);
  StrStat = 0;
  Cursor = 0;
}

long TJob::WriteCommon(char *filename,long Pos, long Size)
{
    ofstream dest;

    dest.open(filename);

    if (!dest) return 0;

    for(int i=0; i<Size; i++)
      dest << CommonMem[Pos+i] << '\n';

    dest.close();
    return Size;

}

long TJob::ReadCommon(char *filename,long Pos)
{

    ifstream source(filename,ios::in);
    if (!source) return 0;

    long n;
    long p = Pos;
    while(source >> n) CommonMem[p++] = n;
    source.close();
    return p - Pos;
}

void TJob::SetKey(int Key)
{
    InKey = toupper(Key);
}

int TJob::GetKey()
{
    int key = InKey;
    InKey = 0;
    return key;
}

void TJob::SetMouse(int X, int Y, int Shift)
{
    MouseX = X;
    MouseY = Y;
    MouseShift = Shift;
}

void TJob::GetMouse(long &X, long &Y, long &Z)
{
#ifdef WS
//DEVELOP...
  X = 0;
  Y = 0;
  Z = 0;
#else //WS
    MousePointer((int *)&X, (int *)&Y, (unsigned int *)&Z);
#endif //WS
}

#ifdef MIDI_SUPPORT

void TJob::MidiOpen()
{

  MidiInUse = 1;
}

void TJob::Midi(int MidiEvent, long MidiMes)
{

}
#endif

void TJob::Color(long no, long dr, long dg, long db,
               long ar, long ag, long ab)
{
    SetPaletteColor(no, dr, dg, db, ar, ag, ab);
    if (g_mode) pScreen->UpdatePalette();
}

void TJob::ColorReset()
{
    SetDefaultPalette();
    if (g_mode) pScreen->UpdatePalette();
}

void TJob::Light(int intensity, int x, int y, int z, int dx, int dy, int dz)
{
   pSpace->SetLightDirection(Vector(x, y, z));
}

void TJob::SetLightObject(THObj3D * pObj, int emit)
{
    if (emit) pSpace->SetLightObject(pObj);
    else pSpace->SetLightObject(0);
}

void TJob::SetLightType(int parallel)
{
    pSpace->SetLightType(parallel);
}

