package jp.ac.dendai.cdl.mori.wikie.mapper;

import java.io.*;
import java.text.*;
import java.util.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;
import jp.ac.dendai.cdl.mori.wikie.util.*;

import org.apache.hadoop.io.*;
import org.apache.hadoop.mapred.*;

/**
 * history処理用Mapper
 *
 * 出力したいファイルの形式はタブ区切りで下のとおり
 * title    日付    編集回数
 *
 * 日付は1週間の初めの月曜日。
 * 編集回数ゼロの週は出力しない。
 * 編集回数は、1週間で同一のユーザが複数回編集している場合は1度だけカウントする。
 * またロボットによる編集はカウント対象外。
 * @author Mori
 *
 */
public class HistoryMapper  extends WikipediaMapper
implements Mapper<LongWritable, Text, Text, Text>{
    private static WikipediaBot bot;

    @Override
    public void configure(JobConf job) {
        super.configure(job);
        if (bot == null) {
            bot = WikipediaBot.getInstance();
        }
    }

    @Override
    public void map(LongWritable key, Text value,
            OutputCollector<Text, Text> output, Reporter reporter)
    throws IOException {
        try {
            String page = WikipediaNormalizer.removeNonPrintingCharacter(value.toString());
            String title = normalizer.normalize(WikipediaMapper.getValueFromElementName(WikIE.TITLE_ELEMENT, page, 0, 1));

            if (normalizer.getNamespaceNumber(title) == WikIE.ARTICLE_NS_NUM) {
                Iterator<String> itr = getEditData(page).iterator();
                while (itr.hasNext()) {
                    String outputValue = (String) itr.next();
                    output.collect(new Text(title), new Text(outputValue));
                }
            }
        }
        catch (NullPointerException e) {
            // TODO: handle exception
        }
    }

    /**
     * 編集情報を取得する。
     * @param page page要素
     * @return 編集情報を格納したArrayList<br>
     *         週の初めの月曜日    編集回数<br>
     *         というタブ区切り形式のStringが1要素
     */
    public static ArrayList<String> getEditData(String page) {
        ArrayList<String> result = new ArrayList<String>();
        ArrayList<WikipediaRevision> revisionArrayList = getRevision(page);
        if (revisionArrayList.size() == 0) return result;
        ArrayList<String> contributor = new ArrayList<String>();
        Date monday = WikipediaHistoryUtil.getNearestBackwardMonday(revisionArrayList.get(0).getUtc());
        Date nextMonday = WikipediaHistoryUtil.getForwardMonday(monday);
        Date date = new Date();
        Iterator<WikipediaRevision> revItr = revisionArrayList.iterator();
        int count = 0;
        while (revItr.hasNext()) {
            WikipediaRevision revision = revItr.next();
            date = revision.getUtc();
            if (0 <= date.compareTo(monday) && date.compareTo(nextMonday) < 0) { //1週間内なら
                if (!contributor.contains(revision.getUserName())) {             //1週間内で重複しないなら
                    contributor.add(revision.getUserName());
                    count++;
                }
            }
            else {
                if (0 < count) {
                    result.add(createRecord(monday, count));
                }
                monday = WikipediaHistoryUtil.getNearestBackwardMonday(date); //次の月曜日を探す
                nextMonday = WikipediaHistoryUtil.getForwardMonday(monday);   //その次の月曜日
                contributor.clear();
                count = 1;
            }
        }
        result.add(createRecord(monday, count)); //最終週分
        return result;
    }

    /**
     * getEditDataが返すArrayListの要素をつくる
     * @param calendar 編集日が設定されているCalendar
     * @param count 編集回数
     * @return 週の初めの月曜日    編集回数<br>
     *         というタブ区切り形式のStringが1要素
     */
    public static String createRecord(Date monday, int count) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(monday);
        return calendar.get(Calendar.YEAR) +
               "-" + (calendar.get(Calendar.MONTH) + 1) +
               "-" + calendar.get(Calendar.DAY_OF_MONTH) +
               "\t" + String.valueOf(count);
    }

    /**
     * page要素から編集者情報を取得する
     * @param page page要素
     * @return 編集者情報を格納したArrayList<br>
     *         ロボットが排除された結果が返される
     */
    public static ArrayList<WikipediaRevision> getRevision(String page) {
        ArrayList<WikipediaRevision> result = new ArrayList<WikipediaRevision>();
        ArrayList<String> revisionArrayList = getValuesFromElementName(WikIE.REVISION_ELEMENT, page, 0);
        Iterator<String> revItr = revisionArrayList.iterator();
        while (revItr.hasNext()) {
            String revisonString = revItr.next();
            String userName = getValueFromElementName(WikIE.USERNAME_ELEMENT, revisonString, 0, 1);
            String timestamp = getValueFromElementName(WikIE.TIMESTAMP_ELEMENT, revisonString, 0, 1);
            String ip = getValueFromElementName(WikIE.IP_ELEMENT, revisonString, 0, 1);
            try {
                if (userName != null) { //アカウント持ちユーザ
                    if (!bot.contains(userName)) { //ロボットは排除
                        result.add(new WikipediaRevision(userName, timestamp));
                    }
                }
                else  //IPユーザ
                    result.add(new WikipediaRevision(ip, timestamp));
            }
            catch (ParseException e) {
                continue;
            }
        }
        return result;
    }
}
