package jp.ac.dendai.cdl.mori.wikie.util;

import java.io.*;
import java.net.*;
import java.util.*;
import java.util.regex.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;

/**
 * Wikipediaのリンク抽出するクラス
 * @author Mori
 *
 */
public class WikipediaLinkChecker {
    /**
     * 唯一のWikipediaLinkCheckerインスタンス
     */
    private static WikipediaLinkChecker instance = null;
    /**
     * Normalizerインスタンス
     */
    private static WikipediaNormalizer normalizer;
    /**
     * 言語一覧が掲載されている記事のURL
     */
    private static final String languageURL = "http://en.wikipedia.org/wiki/Special:Export/List_of_Wikipedias";
    /**
     * 言語略記(en, jaなど)を格納しておくArrayList
     */
    private static ArrayList<String> languageList;

    /**
     * WikipediaLinkChecker生成用
     * @param normalizer 使用するNormalizerインスタンス
     * @return WikipediaLinkCheckerインスタンス
     */
    public static WikipediaLinkChecker getInstance(WikipediaNormalizer normalizer) {
        if (instance == null) {
            instance = new WikipediaLinkChecker(normalizer);
        }
        return instance;
    }

    /**
     * コンストラクタ
     * @param normalizer 使用するNormalizerインスタンス
     */
    private WikipediaLinkChecker(WikipediaNormalizer normalizer) {
        WikipediaLinkChecker.normalizer = normalizer;
        setLanguageList();
    }

    /**
     * テキストがREDIRECTとして機能しているかどうか判定する
     * @param text　テキスト
     * @return REDIRECTの場合は対象となっているエントリのタイトル
     */
    public String isRedirect(String text) {
        Pattern pattern = Pattern.compile("# *REDIRECT *\\[\\[(.+?)\\]\\]", Pattern.CASE_INSENSITIVE);
        Matcher matcher = pattern.matcher(text);
        if (matcher.find()) {
            String target = matcher.group(1).trim();
            target = normalizeLink(target);
            target = target.split("#")[0];
            target = target.split("\\|")[0];
            return normalizer.normalize(target);
        }
        return null;
    }

    /**
     * ページの種類を判定する
     * @param title　Wikipediaエントリのタイトル
     * @param text テキスト
     * @return 普通の記事ならWikIE.NODE<br>
     *         カテゴリならWikIE.NODE<br>
     *         リダイレクトならWikIE.REDIRECT<br>
     */
    public String getKind(String title, String text) {
        int ns = normalizer.getNamespaceNumber(title);
        String kind = WikIE.OTHER;
        if (ns == WikIE.ARTICLE_NS_NUM) {
            String redirect = isRedirect(text);
            if (redirect != null) {
                kind = WikIE.REDIRECT;
            }
            else {
                kind = WikIE.LEAF;
            }
        }
        else if (ns == WikIE.CATEGOR_NS_NUM){
            kind = WikIE.NODE;
        }
        return kind;
    }

    /**
     * カテゴリを抽出する
     * @param text テキスト
     * @return 記述されているカテゴリを格納したArrayList
     */
    public ArrayList<String> getCategory(String text) {
        ArrayList<String> result = new ArrayList<String>();
        Iterator<String> linkItr = getDoubleBracket(text).iterator();
        while (linkItr.hasNext()) {
            String link = linkItr.next();
            try {
                link = link.split("\\|")[0];
                if (normalizer.getNamespaceNumber(link) == WikIE.CATEGOR_NS_NUM) {
                    result.add(link);
                }
            }
            catch (ArrayIndexOutOfBoundsException e) {
                // TODO: handle exception
            }
        }
        return result;
    }

    /**
     * 言語リストを設定する。<br>
     * languageURLのページから抽出する。
     */
    private void setLanguageList() {
        try {
            languageList = new ArrayList<String>();
            languageList.add("en");
            BufferedReader reader = new BufferedReader(
                    new InputStreamReader(
                    new URL(languageURL).openStream()));
            String line = new String();
            while ((line = reader.readLine()) != null) {
                Pattern p = Pattern.compile("\\| *\\[\\[:(.*?):\\|(.*?)\\]\\].*?");
                Matcher m = p.matcher(line);
                if (m.find()) {
                    languageList.add(m.group(1));
                }
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 二重ブラケット[[]]の中身を収集する
     * リンクは正規化される
     * @param text テキスト
     * @return 二重ブラケットの中身を格納したArrayList
     */
    public ArrayList<String> getDoubleBracket(String text) {
        ArrayList<String> result = new ArrayList<String>();
        Pattern pattern = Pattern.compile("\\[\\[(.+?)\\]\\]");
        Matcher matcher = pattern.matcher(text);
        while (matcher.find()) {
            String link = normalizeLink(matcher.group(1));
            if (link.length() > 0) {
                result.add(link);
            }
        }
        return result;
    }

    /**
     * 言語間リンクを収集する
     * @param text テキスト
     * @return 記述されている言語間リンクを格納したArrayList
     */
    public ArrayList<String> getInterWiki(String text) {
        ArrayList<String> result = new ArrayList<String>();
        Iterator<String> linkItr = getDoubleBracket(text).iterator();
        while (linkItr.hasNext()) {
            StringBuffer link = new StringBuffer();
            String[] split = linkItr.next().split(":");
            split[0] = split[0].toLowerCase();
            if (languageList.contains(split[0])) {
                for (String s :split) {
                    link.append(s + ":");
                }
                link.deleteCharAt(link.length()-1);
                result.add(link.toString());
            }
        }
        return result;
    }

    /**
     * リンクが言語間リンクか判定する
     * @param link リンク文字列
     * @return 言語間リンクならtrue
     */
    public boolean isInterWikiLink(String link) {
        try {
            String[] s = link.split(":");
            return languageList.contains(s[0].trim().toLowerCase());
        }
        catch (ArrayIndexOutOfBoundsException e) {
            return false;
        }
    }

    /**
     * ISBNコードを収集する
     * 10桁の旧コードは13桁のコードに変換される
     * @param text　テキスト
     * @param header 3桁の接頭コード(普通は978)
     * @return 記述されているISBNコードを格納したArrayList
     */
    public static ArrayList<String> getISBNCode(String text, String header) {
        ArrayList<String> result = new ArrayList<String>();
        Pattern pattern = Pattern.compile("ISBN *(([0-9]|-)+)");
        Matcher matcher = pattern.matcher(text);
        while (matcher.find()) {
            String code = matcher.group(1).replaceAll("-", "");
            if (code.length() == 10) {
                code = convertISBN10To13(code, header);
            }
            if (code.length() == 13) {
                result.add(code);
            }
        }
        return result;
    }

    /**
     * ページ間リンクを収集する
     * 対象となるのはアンカーテキストとリンク先エントリのタイトルが異なるもののみ
     * @param text テキスト
     * @return 記述されているページ間リンクを格納したArrayList
     */
    public ArrayList<WikipediaLink> getPageLink(String text) {
        ArrayList<WikipediaLink> result = new ArrayList<WikipediaLink>();
        Iterator<String> linkItr = getDoubleBracket(text).iterator();
        while (linkItr.hasNext()) {
            String link = linkItr.next().replaceAll("<.+?>", "");
            String[] splited = link.split("\\|");
            if (splited.length > 1) {
                String entry = splited[0];
                String anchor = splited[1];
                if (0 == entry.indexOf("#")) {
                    entry = entry.substring(entry.indexOf("#")+1, entry.length());
                }
                else if (1 < entry.indexOf("#")) {
                    entry = entry.substring(0, entry.indexOf("#"));
                }
                entry = normalizer.normalize(entry);
                anchor = normalizer.normalize(anchor);
                if (normalizer.getNamespaceNumber(entry) == 0 &&
                        0 < entry.length() &&
                        0 < anchor.length() &&
                        normalizer.getNamespaceNumber(entry) == WikIE.ARTICLE_NS_NUM ||
                        normalizer.getNamespaceNumber(entry) == WikIE.CATEGOR_NS_NUM) {
                    result.add(new WikipediaLink(WikipediaNormalizer.removeNonPrintingCharacter(entry.trim()),
                                                 WikipediaNormalizer.removeNonPrintingCharacter(anchor.trim())));
                }
            }
        }
        return result;
    }


    /**
     * 旧規格の10桁ISBNコードを13桁に変換する。
     * 変換には指定した3桁コードを使用する。(普通は978）
     * @param code10 10桁のコード
     * @param header 3桁の接頭コード(普通は978)
     * @return 変換後の13桁コード
     */
    public static String convertISBN10To13(String code10, String header) {
        StringBuffer code13 = new StringBuffer(header + code10.substring(0, code10.length()-1));
        int sum = 0;
        for (int i = 0; i < code13.length(); i++) {
            Character c = new Character(code13.charAt(i));
            int n = Integer.parseInt(c.toString());
            if (i % 2 == 0) {
                sum += n * 1;
            }
            else {
                sum += n * 3;
            }
        }
        int checkDigit = 10 - sum % 10;
        if (checkDigit % 10 != 0) {
            code13.append(checkDigit);
        }
        else {
            code13.append(0);
        }
        return code13.toString();
    }

    /**
     * リンク文字列を正規化する。
     * 二重ブラケットの中身はこのメソッドを利用して正規化する。
     * @param link　リンク文字列
     * @return 正規化したリンク文字列
     */
    public String normalizeLink(String link) {
        if (link.equals(":") || link.equals("|")) return "";
        link = spilitLink(link);
        String[] splited = link.split("\t");
        if (splited.length == 0) {
            return "";
        }
        else if (splited.length == 1)
            return normalizer.normalize(WikipediaNormalizer.decodeAnchor(splited[0]));
        else {
            String languege = splited[0].toLowerCase();
            if (languege.equals(":"))
                languege = "";
            String title =  normalizer.normalize(splited[1]);
            return languege + title;
        }
    }

    /**
     * リンク文字列を言語部分とタイトル部分に分ける。
     * 区切りは\t記号。
     * @param link　リンク文字列
     * @return \t記号で分けられたリンク文字列
     */
    public String spilitLink(String link) {
        String[] splited = link.split(":");
        boolean lang = true;
        if (splited.length == 1) {
            return link;
        }
        int index = splited.length;
        if (splited.length == 0) {
            return "";
        }
        else if (isInterWikiLink(link)) {
            index = 0;
        }
        else if (languageList.contains(splited[1].toLowerCase().trim())) {
            index = 1;
        }
        else if (splited[0].length() == 0) {
            index = 0;
        }
        else {
            lang = false;
        }
        StringBuffer result = new StringBuffer();
        for (int i = 0; i < splited.length; i++) {
            if (i <= index && lang) {
                result.append(splited[i].trim() + ":");
            }
            else {
                result.append(splited[i] + ":");
            }
            if (i == index) {
                result.append("\t");
            }
        }
        result.deleteCharAt(result.length()-1);
        return result.toString();
    }
}
