package jp.ac.dendai.cdl.mori.wikie.mapred;

import java.io.*;
import java.util.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;
import jp.ac.dendai.cdl.mori.wikie.parser.*;
import jp.ac.dendai.cdl.mori.wikie.util.*;

import org.apache.hadoop.io.*;
import org.apache.hadoop.mapred.*;

/**
 * edge処理用Mapper
 * @author Mori
 *
 */
public class EdgeMapper extends WMapper {

    /**
     * このMapperが出力するキーは常に「エントリのタイトル」である。<br>
     * 記事・カテゴリ・リダイレクトをエントリとする。<br>
     * 処理中のpage要素のタイトルのほかに、<br>
     * 上位カテゴリのタイトルと、<br>
     * リダイレクト先のエントリのタイトルも「エントリのタイトル」として出力をつくる。<br>
     * Reducerでは例えば次のようなキーと値のペアを受け取ることになる。
     * <pre>
     * category:言語    edge　123
     * category:言語    entry　node　456   <--これが他のedgeが向かっているエントリ
     * category:言語    edge　789
     * category:言語    edge 127
     * </pre>
     * リダイレクト関係なら次のようになる<br>
     * <pre>
     * アメリカ合衆国      target 444
     * アメリカ合衆国      entry leaf  444   <--これが他のredirectが向かっているエントリ
     * アメリカ合衆国      target 555
     * アメリカ合衆国      target t666
     * </pre>
     */
    @Override
    public void map(LongWritable key, Text value, OutputCollector<Text, Text> output, Reporter reporter) throws IOException {
        try {
            WPageElementHandler page = createPageHandler(value);
            WEntry entry = normalizer.normalize(page.getTitle());
            String title = entry.toString();
            String id  = page.getId();
            String text = WNormalizer.deleteNonPrintingChar(page.getText());
            String kind = linkUtils.getKind(entry, text);
            if (kind != null) {
                if (kind.equals(WikIE.LEAF) || kind.equals(WikIE.NODE)) {
                    List<WEntry> categoryArrayList = linkUtils.getCategory(text);
                    output.collect(new Text(title), new Text(WikIE.ENTRY + "\t" + kind + "\t" + id));
                    Iterator<WEntry> itr = categoryArrayList.iterator();
                    while (itr.hasNext()) {
                        String category = itr.next().toString();
                        try {
                            output.collect(new Text(category), new Text(WikIE.EDGE + "\t" + id));
                        }
                        catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
                else if (kind.equals(WikIE.REDIRECT)) {
                    String redirect = linkUtils.isRedirect(text).toString();
                    output.collect(new Text(title), new Text(WikIE.ENTRY + "\t" + WikIE.LEAF + "\t" + id));
                    output.collect(new Text(redirect), new Text(WikIE.TARGET + "\t" + id));
                }
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
}
