/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.ByteArrayOutputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.MediumTests;
import org.apache.hadoop.hbase.ResourceCheckerJUnitRule;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.DoubleOutputStream;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.Compression;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoderImpl;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.io.compress.Compressor;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@Category(value={MediumTests.class})
@RunWith(value=Parameterized.class)
public class TestHFileBlock {
    private static final boolean detailedLogging = false;
    private static final boolean[] BOOLEAN_VALUES = new boolean[]{false, true};
    private static final Log LOG = LogFactory.getLog(TestHFileBlock.class);
    static final Compression.Algorithm[] COMPRESSION_ALGORITHMS = new Compression.Algorithm[]{Compression.Algorithm.NONE, Compression.Algorithm.GZ};
    private static final int NUM_TEST_BLOCKS = 1000;
    private static final int NUM_READER_THREADS = 26;
    private static int NUM_KEYVALUES = 50;
    private static int FIELD_LENGTH = 10;
    private static float CHANCE_TO_REPEAT = 0.6f;
    private static final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
    private FileSystem fs;
    private int uncompressedSizeV1;
    private final boolean includesMemstoreTS;
    @Rule
    public ResourceCheckerJUnitRule cu = new ResourceCheckerJUnitRule();

    public TestHFileBlock(boolean includesMemstoreTS) {
        this.includesMemstoreTS = includesMemstoreTS;
    }

    @Parameterized.Parameters
    public static Collection<Object[]> parameters() {
        return HBaseTestingUtility.BOOLEAN_PARAMETERIZED;
    }

    @Before
    public void setUp() throws IOException {
        this.fs = HFileSystem.get((Configuration)TEST_UTIL.getConfiguration());
    }

    static void writeTestBlockContents(DataOutputStream dos) throws IOException {
        for (int i = 0; i < 1000; ++i) {
            dos.writeInt(i / 100);
        }
    }

    static int writeTestKeyValues(OutputStream dos, int seed, boolean includesMemstoreTS) throws IOException {
        ArrayList<KeyValue> keyValues = new ArrayList<KeyValue>();
        Random randomizer = new Random(42L + (long)seed);
        for (int i = 0; i < NUM_KEYVALUES; ++i) {
            byte[] value;
            byte[] qualifier;
            byte[] family;
            byte[] row;
            if (0 < i && randomizer.nextFloat() < CHANCE_TO_REPEAT) {
                row = ((KeyValue)keyValues.get(randomizer.nextInt(keyValues.size()))).getRow();
            } else {
                row = new byte[FIELD_LENGTH];
                randomizer.nextBytes(row);
            }
            if (0 == i) {
                family = new byte[FIELD_LENGTH];
                randomizer.nextBytes(family);
            } else {
                family = ((KeyValue)keyValues.get(0)).getFamily();
            }
            if (0 < i && randomizer.nextFloat() < CHANCE_TO_REPEAT) {
                qualifier = ((KeyValue)keyValues.get(randomizer.nextInt(keyValues.size()))).getQualifier();
            } else {
                qualifier = new byte[FIELD_LENGTH];
                randomizer.nextBytes(qualifier);
            }
            if (0 < i && randomizer.nextFloat() < CHANCE_TO_REPEAT) {
                value = ((KeyValue)keyValues.get(randomizer.nextInt(keyValues.size()))).getValue();
            } else {
                value = new byte[FIELD_LENGTH];
                randomizer.nextBytes(value);
            }
            long timestamp = 0 < i && randomizer.nextFloat() < CHANCE_TO_REPEAT ? ((KeyValue)keyValues.get(randomizer.nextInt(keyValues.size()))).getTimestamp() : randomizer.nextLong();
            keyValues.add(new KeyValue(row, family, qualifier, timestamp, value));
        }
        int totalSize = 0;
        Collections.sort(keyValues, KeyValue.COMPARATOR);
        DataOutputStream dataOutputStream = new DataOutputStream(dos);
        for (KeyValue kv : keyValues) {
            totalSize += kv.getLength();
            dataOutputStream.write(kv.getBuffer(), kv.getOffset(), kv.getLength());
            if (!includesMemstoreTS) continue;
            long memstoreTS = randomizer.nextLong();
            WritableUtils.writeVLong((DataOutput)dataOutputStream, (long)memstoreTS);
            totalSize += WritableUtils.getVIntSize((long)memstoreTS);
        }
        return totalSize;
    }

    public byte[] createTestV1Block(Compression.Algorithm algo) throws IOException {
        Compressor compressor = algo.getCompressor();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OutputStream os = algo.createCompressionStream((OutputStream)baos, compressor, 0);
        DataOutputStream dos = new DataOutputStream(os);
        BlockType.META.write((DataOutput)dos);
        TestHFileBlock.writeTestBlockContents(dos);
        this.uncompressedSizeV1 = dos.size();
        dos.flush();
        algo.returnCompressor(compressor);
        return baos.toByteArray();
    }

    static HFileBlock.Writer createTestV2Block(Compression.Algorithm algo, boolean includesMemstoreTS) throws IOException {
        BlockType blockType = BlockType.DATA;
        HFileBlock.Writer hbw = new HFileBlock.Writer(algo, null, includesMemstoreTS, 1, HFile.DEFAULT_CHECKSUM_TYPE, 16384);
        DataOutputStream dos = hbw.startWriting(blockType);
        TestHFileBlock.writeTestBlockContents(dos);
        byte[] headerAndData = hbw.getHeaderAndDataForTest();
        Assert.assertEquals((long)4000L, (long)hbw.getUncompressedSizeWithoutHeader());
        hbw.releaseCompressor();
        return hbw;
    }

    public String createTestBlockStr(Compression.Algorithm algo, int correctLength) throws IOException {
        HFileBlock.Writer hbw = TestHFileBlock.createTestV2Block(algo, this.includesMemstoreTS);
        byte[] testV2Block = hbw.getHeaderAndDataForTest();
        int osOffset = 42;
        if (testV2Block.length == correctLength) {
            testV2Block[osOffset] = 3;
        }
        return Bytes.toStringBinary((byte[])testV2Block);
    }

    @Test
    public void testNoCompression() throws IOException {
        Assert.assertEquals((long)4000L, (long)TestHFileBlock.createTestV2Block(Compression.Algorithm.NONE, this.includesMemstoreTS).getBlockForCaching().getUncompressedSizeWithoutHeader());
    }

    @Test
    public void testGzipCompression() throws IOException {
        String correctTestBlockStr = "DATABLK*\\x00\\x00\\x00>\\x00\\x00\\x0F\\xA0\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\x01\\x00\\x00@\\x00\\x00\\x00\\x00[\\x1F\\x8B\\x08\\x00\\x00\\x00\\x00\\x00\\x00\\x03\\xED\\xC3\\xC1\\x11\\x00 \\x08\\xC00DD\\xDD\\x7Fa\\xD6\\xE8\\xA3\\xB9K\\x84`\\x96Q\\xD3\\xA8\\xDB\\xA8e\\xD4c\\xD46\\xEA5\\xEA3\\xEA7\\xE7\\x00LI\\x5Cs\\xA0\\x0F\\x00\\x00\\xAB\\x85g\\x91";
        int correctGzipBlockLength = 95;
        Assert.assertEquals((Object)"DATABLK*\\x00\\x00\\x00>\\x00\\x00\\x0F\\xA0\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\xFF\\x01\\x00\\x00@\\x00\\x00\\x00\\x00[\\x1F\\x8B\\x08\\x00\\x00\\x00\\x00\\x00\\x00\\x03\\xED\\xC3\\xC1\\x11\\x00 \\x08\\xC00DD\\xDD\\x7Fa\\xD6\\xE8\\xA3\\xB9K\\x84`\\x96Q\\xD3\\xA8\\xDB\\xA8e\\xD4c\\xD46\\xEA5\\xEA3\\xEA7\\xE7\\x00LI\\x5Cs\\xA0\\x0F\\x00\\x00\\xAB\\x85g\\x91", (Object)this.createTestBlockStr(Compression.Algorithm.GZ, 95));
    }

    @Test
    public void testReaderV1() throws IOException {
        for (Compression.Algorithm algo : COMPRESSION_ALGORITHMS) {
            for (boolean pread : new boolean[]{false, true}) {
                byte[] block = this.createTestV1Block(algo);
                Path path = new Path(TEST_UTIL.getDataTestDir(), "blocks_v1_" + algo);
                LOG.info((Object)("Creating temporary file at " + path));
                FSDataOutputStream os = this.fs.create(path);
                int totalSize = 0;
                int numBlocks = 50;
                for (int i = 0; i < numBlocks; ++i) {
                    os.write(block);
                    totalSize += block.length;
                }
                os.close();
                FSDataInputStream is = this.fs.open(path);
                HFileBlock.FSReaderV1 hbr = new HFileBlock.FSReaderV1(is, algo, (long)totalSize);
                int numBlocksRead = 0;
                long pos = 0L;
                while (pos < (long)totalSize) {
                    HFileBlock b = hbr.readBlockData(pos, (long)block.length, this.uncompressedSizeV1, pread);
                    b.sanityCheck();
                    pos += (long)block.length;
                    ++numBlocksRead;
                }
                Assert.assertEquals((long)numBlocks, (long)numBlocksRead);
                is.close();
            }
        }
    }

    @Test
    public void testReaderV2() throws IOException {
        for (Compression.Algorithm algo : COMPRESSION_ALGORITHMS) {
            for (boolean pread : new boolean[]{false, true}) {
                LOG.info((Object)("testReaderV2: Compression algorithm: " + algo + ", pread=" + pread));
                Path path = new Path(TEST_UTIL.getDataTestDir(), "blocks_v2_" + algo);
                FSDataOutputStream os = this.fs.create(path);
                HFileBlock.Writer hbw = new HFileBlock.Writer(algo, null, this.includesMemstoreTS, 1, HFile.DEFAULT_CHECKSUM_TYPE, 16384);
                long totalSize = 0L;
                for (int blockId = 0; blockId < 2; ++blockId) {
                    DataOutputStream dos = hbw.startWriting(BlockType.DATA);
                    for (int i = 0; i < 1234; ++i) {
                        dos.writeInt(i);
                    }
                    hbw.writeHeaderAndData(os);
                    totalSize += (long)hbw.getOnDiskSizeWithHeader();
                }
                os.close();
                FSDataInputStream is = this.fs.open(path);
                HFileBlock.FSReaderV2 hbr = new HFileBlock.FSReaderV2(is, algo, totalSize);
                HFileBlock b = hbr.readBlockData(0L, -1L, -1, pread);
                is.close();
                Assert.assertEquals((long)0L, (long)HFile.getChecksumFailuresCount());
                b.sanityCheck();
                Assert.assertEquals((long)4936L, (long)b.getUncompressedSizeWithoutHeader());
                Assert.assertEquals((long)(algo == Compression.Algorithm.GZ ? 2173L : 4936L), (long)(b.getOnDiskSizeWithoutHeader() - b.totalChecksumBytes()));
                String blockStr = b.toString();
                if (algo != Compression.Algorithm.GZ) continue;
                is = this.fs.open(path);
                hbr = new HFileBlock.FSReaderV2(is, algo, totalSize);
                b = hbr.readBlockData(0L, (long)(2206 + b.totalChecksumBytes()), -1, pread);
                Assert.assertEquals((Object)blockStr, (Object)b.toString());
                int wrongCompressedSize = 2172;
                try {
                    b = hbr.readBlockData(0L, (long)(wrongCompressedSize + 33), -1, pread);
                    Assert.fail((String)"Exception expected");
                }
                catch (IOException ex) {
                    String expectedPrefix = "On-disk size without header provided is " + wrongCompressedSize + ", but block header contains " + b.getOnDiskSizeWithoutHeader() + ".";
                    Assert.assertTrue((String)("Invalid exception message: '" + ex.getMessage() + "'.\nMessage is expected to start with: '" + expectedPrefix + "'"), (boolean)ex.getMessage().startsWith(expectedPrefix));
                }
                is.close();
            }
        }
    }

    @Test
    public void testDataBlockEncoding() throws IOException {
        int numBlocks = 5;
        for (Compression.Algorithm algo : COMPRESSION_ALGORITHMS) {
            for (boolean pread : new boolean[]{false, true}) {
                for (DataBlockEncoding encoding : DataBlockEncoding.values()) {
                    Path path = new Path(TEST_UTIL.getDataTestDir(), "blocks_v2_" + algo + "_" + encoding.toString());
                    FSDataOutputStream os = this.fs.create(path);
                    HFileDataBlockEncoderImpl dataBlockEncoder = new HFileDataBlockEncoderImpl(encoding);
                    HFileBlock.Writer hbw = new HFileBlock.Writer(algo, (HFileDataBlockEncoder)dataBlockEncoder, this.includesMemstoreTS, 1, HFile.DEFAULT_CHECKSUM_TYPE, 16384);
                    long totalSize = 0L;
                    ArrayList<Integer> encodedSizes = new ArrayList<Integer>();
                    ArrayList<ByteBuffer> encodedBlocks = new ArrayList<ByteBuffer>();
                    for (int blockId = 0; blockId < 5; ++blockId) {
                        DataOutputStream dos = hbw.startWriting(BlockType.DATA);
                        TestHFileBlock.writeEncodedBlock(encoding, dos, encodedSizes, encodedBlocks, blockId, this.includesMemstoreTS);
                        hbw.writeHeaderAndData(os);
                        totalSize += (long)hbw.getOnDiskSizeWithHeader();
                    }
                    os.close();
                    FSDataInputStream is = this.fs.open(path);
                    HFileBlock.FSReaderV2 hbr = new HFileBlock.FSReaderV2(is, algo, totalSize);
                    hbr.setDataBlockEncoder((HFileDataBlockEncoder)dataBlockEncoder);
                    hbr.setIncludesMemstoreTS(this.includesMemstoreTS);
                    int pos = 0;
                    for (int blockId = 0; blockId < 5; ++blockId) {
                        HFileBlock b = hbr.readBlockData((long)pos, -1L, -1, pread);
                        Assert.assertEquals((long)0L, (long)HFile.getChecksumFailuresCount());
                        b.sanityCheck();
                        pos += b.getOnDiskSizeWithHeader();
                        Assert.assertEquals((long)((Integer)encodedSizes.get(blockId)).intValue(), (long)b.getUncompressedSizeWithoutHeader());
                        ByteBuffer actualBuffer = b.getBufferWithoutHeader();
                        if (encoding != DataBlockEncoding.NONE) {
                            Assert.assertEquals((long)0L, (long)actualBuffer.get(0));
                            Assert.assertEquals((long)encoding.getId(), (long)actualBuffer.get(1));
                            actualBuffer.position(2);
                            actualBuffer = actualBuffer.slice();
                        }
                        ByteBuffer expectedBuffer = (ByteBuffer)encodedBlocks.get(blockId);
                        expectedBuffer.rewind();
                        TestHFileBlock.assertBuffersEqual(expectedBuffer, actualBuffer, algo, encoding, pread);
                    }
                    is.close();
                }
            }
        }
    }

    static void writeEncodedBlock(DataBlockEncoding encoding, DataOutputStream dos, List<Integer> encodedSizes, List<ByteBuffer> encodedBlocks, int blockId, boolean includesMemstoreTS) throws IOException {
        ByteBuffer encodedBuf;
        int encodedSize;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        DoubleOutputStream doubleOutputStream = new DoubleOutputStream((OutputStream)dos, (OutputStream)baos);
        int rawBlockSize = TestHFileBlock.writeTestKeyValues((OutputStream)doubleOutputStream, blockId, includesMemstoreTS);
        ByteBuffer rawBuf = ByteBuffer.wrap(baos.toByteArray());
        rawBuf.rewind();
        if (encoding == DataBlockEncoding.NONE) {
            encodedSize = rawBlockSize;
            encodedBuf = rawBuf;
        } else {
            ByteArrayOutputStream encodedOut = new ByteArrayOutputStream();
            encoding.getEncoder().compressKeyValues(new DataOutputStream(encodedOut), rawBuf.duplicate(), includesMemstoreTS);
            encodedSize = encodedOut.size() + 2;
            encodedBuf = ByteBuffer.wrap(encodedOut.toByteArray());
        }
        encodedSizes.add(encodedSize);
        encodedBlocks.add(encodedBuf);
    }

    static void assertBuffersEqual(ByteBuffer expectedBuffer, ByteBuffer actualBuffer, Compression.Algorithm compression, DataBlockEncoding encoding, boolean pread) {
        if (!actualBuffer.equals(expectedBuffer)) {
            int prefix;
            int minLimit = Math.min(expectedBuffer.limit(), actualBuffer.limit());
            for (prefix = 0; prefix < minLimit && expectedBuffer.get(prefix) == actualBuffer.get(prefix); ++prefix) {
            }
            Assert.fail((String)String.format("Content mismath for compression %s, encoding %s, pread %s, commonPrefix %d, expected %s, got %s", compression, encoding, pread, prefix, TestHFileBlock.nextBytesToStr(expectedBuffer, prefix), TestHFileBlock.nextBytesToStr(actualBuffer, prefix)));
        }
    }

    private static String nextBytesToStr(ByteBuffer buf, int pos) {
        int maxBytes = buf.limit() - pos;
        int numBytes = Math.min(16, maxBytes);
        return Bytes.toStringBinary((byte[])buf.array(), (int)(buf.arrayOffset() + pos), (int)numBytes) + (numBytes < maxBytes ? "..." : "");
    }

    @Test
    public void testPreviousOffset() throws IOException {
        for (Compression.Algorithm algo : COMPRESSION_ALGORITHMS) {
            for (boolean pread : BOOLEAN_VALUES) {
                for (boolean cacheOnWrite : BOOLEAN_VALUES) {
                    Random rand = this.defaultRandom();
                    LOG.info((Object)("testPreviousOffset:Compression algorithm: " + algo + ", pread=" + pread + ", cacheOnWrite=" + cacheOnWrite));
                    Path path = new Path(TEST_UTIL.getDataTestDir(), "prev_offset");
                    ArrayList<Long> expectedOffsets = new ArrayList<Long>();
                    ArrayList<Long> expectedPrevOffsets = new ArrayList<Long>();
                    ArrayList<BlockType> expectedTypes = new ArrayList<BlockType>();
                    ArrayList<ByteBuffer> expectedContents = cacheOnWrite ? new ArrayList<ByteBuffer>() : null;
                    long totalSize = this.writeBlocks(rand, algo, path, expectedOffsets, expectedPrevOffsets, expectedTypes, expectedContents);
                    FSDataInputStream is = this.fs.open(path);
                    HFileBlock.FSReaderV2 hbr = new HFileBlock.FSReaderV2(is, algo, totalSize);
                    long curOffset = 0L;
                    for (int i = 0; i < 1000; ++i) {
                        if (!pread) {
                            Assert.assertEquals((long)is.getPos(), (long)(curOffset + (long)(i == 0 ? 0 : 33)));
                        }
                        Assert.assertEquals((long)((Long)expectedOffsets.get(i)), (long)curOffset);
                        HFileBlock b = hbr.readBlockData(curOffset, -1L, -1, pread);
                        Assert.assertEquals((String)("Invalid block #" + i + "'s type:"), expectedTypes.get(i), (Object)b.getBlockType());
                        Assert.assertEquals((String)("Invalid previous block offset for block " + i + " of " + "type " + b.getBlockType() + ":"), (long)((Long)expectedPrevOffsets.get(i)), (long)b.getPrevBlockOffset());
                        b.sanityCheck();
                        Assert.assertEquals((long)curOffset, (long)b.getOffset());
                        HFileBlock b2 = hbr.readBlockData(curOffset, (long)b.getOnDiskSizeWithHeader(), -1, pread);
                        b2.sanityCheck();
                        Assert.assertEquals((Object)b.getBlockType(), (Object)b2.getBlockType());
                        Assert.assertEquals((long)b.getOnDiskSizeWithoutHeader(), (long)b2.getOnDiskSizeWithoutHeader());
                        Assert.assertEquals((long)b.getOnDiskSizeWithHeader(), (long)b2.getOnDiskSizeWithHeader());
                        Assert.assertEquals((long)b.getUncompressedSizeWithoutHeader(), (long)b2.getUncompressedSizeWithoutHeader());
                        Assert.assertEquals((long)b.getPrevBlockOffset(), (long)b2.getPrevBlockOffset());
                        Assert.assertEquals((long)curOffset, (long)b2.getOffset());
                        Assert.assertEquals((long)b.getBytesPerChecksum(), (long)b2.getBytesPerChecksum());
                        Assert.assertEquals((long)b.getOnDiskDataSizeWithHeader(), (long)b2.getOnDiskDataSizeWithHeader());
                        Assert.assertEquals((long)0L, (long)HFile.getChecksumFailuresCount());
                        curOffset += (long)b.getOnDiskSizeWithHeader();
                        if (!cacheOnWrite) continue;
                        ByteBuffer bufRead = b.getBufferWithHeader();
                        ByteBuffer bufExpected = (ByteBuffer)expectedContents.get(i);
                        boolean bytesAreCorrect = Bytes.compareTo((byte[])bufRead.array(), (int)bufRead.arrayOffset(), (int)(bufRead.limit() - b.totalChecksumBytes()), (byte[])bufExpected.array(), (int)bufExpected.arrayOffset(), (int)bufExpected.limit()) == 0;
                        String wrongBytesMsg = "";
                        if (!bytesAreCorrect) {
                            wrongBytesMsg = "Expected bytes in block #" + i + " (algo=" + algo + ", pread=" + pread + ", cacheOnWrite=" + cacheOnWrite + "):\n";
                            wrongBytesMsg = wrongBytesMsg + Bytes.toStringBinary((byte[])bufExpected.array(), (int)bufExpected.arrayOffset(), (int)Math.min(32, bufExpected.limit())) + ", actual:\n" + Bytes.toStringBinary((byte[])bufRead.array(), (int)bufRead.arrayOffset(), (int)Math.min(32, bufRead.limit()));
                        }
                        Assert.assertTrue((String)wrongBytesMsg, (boolean)bytesAreCorrect);
                    }
                    Assert.assertEquals((long)curOffset, (long)this.fs.getFileStatus(path).getLen());
                    is.close();
                }
            }
        }
    }

    private Random defaultRandom() {
        return new Random(189237L);
    }

    @Test
    public void testConcurrentReading() throws Exception {
        for (Compression.Algorithm compressAlgo : COMPRESSION_ALGORITHMS) {
            int i;
            Path path = new Path(TEST_UTIL.getDataTestDir(), "concurrent_reading");
            Random rand = this.defaultRandom();
            ArrayList<Long> offsets = new ArrayList<Long>();
            ArrayList<BlockType> types = new ArrayList<BlockType>();
            this.writeBlocks(rand, compressAlgo, path, offsets, null, types, null);
            FSDataInputStream is = this.fs.open(path);
            long fileSize = this.fs.getFileStatus(path).getLen();
            HFileBlock.FSReaderV2 hbr = new HFileBlock.FSReaderV2(is, compressAlgo, fileSize);
            ExecutorService exec = Executors.newFixedThreadPool(26);
            ExecutorCompletionService<Boolean> ecs = new ExecutorCompletionService<Boolean>(exec);
            for (i = 0; i < 26; ++i) {
                ecs.submit(new BlockReaderThread("reader_" + (char)(65 + i), (HFileBlock.FSReader)hbr, offsets, types, fileSize));
            }
            for (i = 0; i < 26; ++i) {
                Future result = ecs.take();
                Assert.assertTrue((boolean)((Boolean)result.get()));
            }
            is.close();
        }
    }

    private long writeBlocks(Random rand, Compression.Algorithm compressAlgo, Path path, List<Long> expectedOffsets, List<Long> expectedPrevOffsets, List<BlockType> expectedTypes, List<ByteBuffer> expectedContents) throws IOException {
        boolean cacheOnWrite = expectedContents != null;
        FSDataOutputStream os = this.fs.create(path);
        HFileBlock.Writer hbw = new HFileBlock.Writer(compressAlgo, null, this.includesMemstoreTS, 1, HFile.DEFAULT_CHECKSUM_TYPE, 16384);
        HashMap<BlockType, Long> prevOffsetByType = new HashMap<BlockType, Long>();
        long totalSize = 0L;
        for (int i = 0; i < 1000; ++i) {
            long pos = os.getPos();
            int blockTypeOrdinal = rand.nextInt(BlockType.values().length);
            if (blockTypeOrdinal == BlockType.ENCODED_DATA.ordinal()) {
                blockTypeOrdinal = BlockType.DATA.ordinal();
            }
            BlockType bt = BlockType.values()[blockTypeOrdinal];
            DataOutputStream dos = hbw.startWriting(bt);
            int size = rand.nextInt(500);
            for (int j = 0; j < size; ++j) {
                dos.writeShort(i + 1);
                dos.writeInt(j + 1);
            }
            if (expectedOffsets != null) {
                expectedOffsets.add(os.getPos());
            }
            if (expectedPrevOffsets != null) {
                Long prevOffset = (Long)prevOffsetByType.get(bt);
                expectedPrevOffsets.add(prevOffset != null ? prevOffset : -1L);
                prevOffsetByType.put(bt, os.getPos());
            }
            expectedTypes.add(bt);
            hbw.writeHeaderAndData(os);
            totalSize += (long)hbw.getOnDiskSizeWithHeader();
            if (!cacheOnWrite) continue;
            expectedContents.add(hbw.getUncompressedBufferWithHeader());
        }
        os.close();
        LOG.info((Object)("Created a temporary file at " + path + ", " + this.fs.getFileStatus(path).getLen() + " byte, compression=" + compressAlgo));
        return totalSize;
    }

    @Test
    public void testBlockHeapSize() {
        if (ClassSize.is32BitJVM()) {
            Assert.assertTrue((HFileBlock.BYTE_BUFFER_HEAP_SIZE == 64 ? 1 : 0) != 0);
        } else {
            Assert.assertTrue((HFileBlock.BYTE_BUFFER_HEAP_SIZE == 80 ? 1 : 0) != 0);
        }
        for (int size : new int[]{100, 256, 12345}) {
            byte[] byteArr = new byte[33 + size];
            ByteBuffer buf = ByteBuffer.wrap(byteArr, 0, size);
            HFileBlock block = new HFileBlock(BlockType.DATA, size, size, -1L, buf, true, -1L, this.includesMemstoreTS, 0, 0, ChecksumType.NULL.getCode(), 0);
            long byteBufferExpectedSize = ClassSize.align((long)(ClassSize.estimateBase(buf.getClass(), (boolean)true) + 33L + (long)size));
            long hfileBlockExpectedSize = ClassSize.align((long)ClassSize.estimateBase(HFileBlock.class, (boolean)true));
            long expected = hfileBlockExpectedSize + byteBufferExpectedSize;
            Assert.assertEquals((String)("Block data size: " + size + ", byte buffer expected " + "size: " + byteBufferExpectedSize + ", HFileBlock class expected " + "size: " + hfileBlockExpectedSize + ";"), (long)expected, (long)block.heapSize());
        }
    }

    private class BlockReaderThread
    implements Callable<Boolean> {
        private final String clientId;
        private final HFileBlock.FSReader hbr;
        private final List<Long> offsets;
        private final List<BlockType> types;
        private final long fileSize;

        public BlockReaderThread(String clientId, HFileBlock.FSReader hbr, List<Long> offsets, List<BlockType> types, long fileSize) {
            this.clientId = clientId;
            this.offsets = offsets;
            this.hbr = hbr;
            this.types = types;
            this.fileSize = fileSize;
        }

        @Override
        public Boolean call() throws Exception {
            Random rand = new Random(this.clientId.hashCode());
            long endTime = System.currentTimeMillis() + 10000L;
            int numBlocksRead = 0;
            int numPositionalRead = 0;
            int numWithOnDiskSize = 0;
            while (System.currentTimeMillis() < endTime) {
                HFileBlock b;
                int blockId = rand.nextInt(1000);
                long offset = this.offsets.get(blockId);
                boolean pread = rand.nextBoolean();
                boolean withOnDiskSize = rand.nextBoolean();
                long expectedSize = (blockId == 999 ? this.fileSize : this.offsets.get(blockId + 1)) - offset;
                try {
                    long onDiskSizeArg = withOnDiskSize ? expectedSize : -1L;
                    b = this.hbr.readBlockData(offset, onDiskSizeArg, -1, pread);
                }
                catch (IOException ex) {
                    LOG.error((Object)("Error in client " + this.clientId + " trying to read block at " + offset + ", pread=" + pread + ", withOnDiskSize=" + withOnDiskSize), (Throwable)ex);
                    return false;
                }
                Assert.assertEquals((Object)this.types.get(blockId), (Object)b.getBlockType());
                Assert.assertEquals((long)expectedSize, (long)b.getOnDiskSizeWithHeader());
                Assert.assertEquals((long)offset, (long)b.getOffset());
                ++numBlocksRead;
                if (pread) {
                    ++numPositionalRead;
                }
                if (!withOnDiskSize) continue;
                ++numWithOnDiskSize;
            }
            LOG.info((Object)("Client " + this.clientId + " successfully read " + numBlocksRead + " blocks (with pread: " + numPositionalRead + ", with onDiskSize " + "specified: " + numWithOnDiskSize + ")"));
            return true;
        }
    }
}

