/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.snapshot;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.LargeTests;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.regionserver.ConstantSizeRegionSplitPolicy;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.snapshot.HSnapshotDescription;
import org.apache.hadoop.hbase.snapshot.SnapshotCreationException;
import org.apache.hadoop.hbase.snapshot.SnapshotDescriptionUtils;
import org.apache.hadoop.hbase.snapshot.SnapshotTestingUtils;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSTableDescriptors;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.JVMClusterUtil;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={LargeTests.class})
public class TestFlushSnapshotFromClient {
    private static final Log LOG = LogFactory.getLog(TestFlushSnapshotFromClient.class);
    private static final HBaseTestingUtility UTIL = new HBaseTestingUtility();
    private static final int NUM_RS = 2;
    private static final String STRING_TABLE_NAME = "test";
    private static final byte[] TEST_FAM = Bytes.toBytes((String)"fam");
    private static final byte[] TABLE_NAME = Bytes.toBytes((String)"test");

    @BeforeClass
    public static void setupCluster() throws Exception {
        TestFlushSnapshotFromClient.setupConf(UTIL.getConfiguration());
        UTIL.startMiniCluster(2);
    }

    private static void setupConf(Configuration conf) {
        conf.setInt("hbase.regionsever.info.port", -1);
        conf.setInt("hbase.hregion.memstore.flush.size", 25000);
        conf.setInt("hbase.hstore.compaction.min", 10);
        conf.setInt("hbase.hstore.compactionThreshold", 10);
        conf.setInt("hbase.hstore.blockingStoreFiles", 12);
        conf.setInt("hbase.client.retries.number", 1);
        conf.setBoolean("hbase.snapshot.enabled", true);
        conf.set("hbase.regionserver.region.split.policy", ConstantSizeRegionSplitPolicy.class.getName());
    }

    @Before
    public void setup() throws Exception {
        UTIL.createTable(TABLE_NAME, TEST_FAM);
    }

    @After
    public void tearDown() throws Exception {
        UTIL.deleteTable(TABLE_NAME);
        try {
            UTIL.getTestFileSystem().delete(new Path(UTIL.getDefaultRootDirPath(), ".archive"), true);
        }
        catch (IOException e) {
            LOG.warn((Object)"Failure to delete archive directory", (Throwable)e);
        }
    }

    @AfterClass
    public static void cleanupTest() throws Exception {
        try {
            UTIL.shutdownMiniCluster();
        }
        catch (Exception e) {
            LOG.warn((Object)"failure shutting down cluster", (Throwable)e);
        }
    }

    @Test
    public void testFlushTableSnapshot() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        HTable table = new HTable(UTIL.getConfiguration(), TABLE_NAME);
        UTIL.loadTable(table, TEST_FAM);
        HashSet<String> snapshotServers = new HashSet<String>();
        List<JVMClusterUtil.RegionServerThread> servers = UTIL.getMiniHBaseCluster().getLiveRegionServerThreads();
        for (JVMClusterUtil.RegionServerThread server : servers) {
            if (server.getRegionServer().getOnlineRegions(TABLE_NAME).size() <= 0) continue;
            snapshotServers.add(server.getRegionServer().getServerName().toString());
        }
        LOG.debug((Object)"FS state before snapshot:");
        FSUtils.logFileSystemState((FileSystem)UTIL.getTestFileSystem(), (Path)FSUtils.getRootDir((Configuration)UTIL.getConfiguration()), (Log)LOG);
        String snapshotString = "offlineTableSnapshot";
        byte[] snapshot = Bytes.toBytes((String)snapshotString);
        admin.snapshot(snapshotString, STRING_TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH);
        LOG.debug((Object)"Snapshot completed.");
        List snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshot, TABLE_NAME);
        FileSystem fs = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getFileSystem();
        Path rootDir = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getRootDir();
        LOG.debug((Object)"FS state after snapshot:");
        FSUtils.logFileSystemState((FileSystem)UTIL.getTestFileSystem(), (Path)FSUtils.getRootDir((Configuration)UTIL.getConfiguration()), (Log)LOG);
        SnapshotTestingUtils.confirmSnapshotValid(snapshots.get(0), TABLE_NAME, TEST_FAM, rootDir, admin, fs, false, new Path(rootDir, ".logs"), snapshotServers);
        admin.deleteSnapshot(snapshot);
        snapshots = admin.listSnapshots();
        SnapshotTestingUtils.assertNoSnapshots(admin);
    }

    @Test
    public void testSnapshotFailsOnNonExistantTable() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        String tableName = "_not_a_table";
        boolean fail = false;
        do {
            try {
                admin.getTableDescriptor(Bytes.toBytes((String)tableName));
                fail = true;
                LOG.error((Object)("Table:" + tableName + " already exists, checking a new name"));
                tableName = tableName + "!";
            }
            catch (TableNotFoundException e) {
                fail = false;
            }
        } while (fail);
        try {
            admin.snapshot("fail", tableName, HBaseProtos.SnapshotDescription.Type.FLUSH);
            Assert.fail((String)"Snapshot succeeded even though there is not table.");
        }
        catch (SnapshotCreationException e) {
            LOG.info((Object)("Correctly failed to snapshot a non-existant table:" + e.getMessage()));
        }
    }

    @Test(timeout=15000L)
    public void testAsyncFlushSnapshot() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        HBaseProtos.SnapshotDescription snapshot = HBaseProtos.SnapshotDescription.newBuilder().setName("asyncSnapshot").setTable(STRING_TABLE_NAME).setType(HBaseProtos.SnapshotDescription.Type.FLUSH).build();
        admin.takeSnapshotAsync(snapshot);
        HMaster master = UTIL.getMiniHBaseCluster().getMaster();
        SnapshotTestingUtils.waitForSnapshotToComplete(master, new HSnapshotDescription(snapshot), 200L);
        LOG.info((Object)" === Async Snapshot Completed ===");
        FSUtils.logFileSystemState((FileSystem)UTIL.getTestFileSystem(), (Path)FSUtils.getRootDir((Configuration)UTIL.getConfiguration()), (Log)LOG);
        SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshot);
        admin.deleteSnapshot(snapshot.getName());
        LOG.info((Object)" === Async Snapshot Deleted ===");
        FSUtils.logFileSystemState((FileSystem)UTIL.getTestFileSystem(), (Path)FSUtils.getRootDir((Configuration)UTIL.getConfiguration()), (Log)LOG);
        SnapshotTestingUtils.assertNoSnapshots(admin);
        LOG.info((Object)" === Async Snapshot Test Completed ===");
    }

    @Test
    public void testFlushCreateListDestroy() throws Exception {
        LOG.debug((Object)"------- Starting Snapshot test -------------");
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        UTIL.loadTable(new HTable(UTIL.getConfiguration(), TABLE_NAME), TEST_FAM);
        HRegionServer rs = UTIL.getRSForFirstRegionInTable(TABLE_NAME);
        List onlineRegions = rs.getOnlineRegions(TABLE_NAME);
        for (HRegion region : onlineRegions) {
            region.waitForFlushesAndCompactions();
        }
        String snapshotName = "flushSnapshotCreateListDestroy";
        admin.snapshot(snapshotName, STRING_TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH);
        this.logFSTree(new Path(UTIL.getConfiguration().get("hbase.rootdir")));
        List<HBaseProtos.SnapshotDescription> snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshotName, STRING_TABLE_NAME);
        FileSystem fs = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getFileSystem();
        Path rootDir = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getRootDir();
        Path snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir((HBaseProtos.SnapshotDescription)snapshots.get(0), (Path)rootDir);
        Assert.assertTrue((boolean)fs.exists(snapshotDir));
        FSUtils.logFileSystemState((FileSystem)UTIL.getTestFileSystem(), (Path)snapshotDir, (Log)LOG);
        Path snapshotinfo = new Path(snapshotDir, ".snapshotinfo");
        Assert.assertTrue((boolean)fs.exists(snapshotinfo));
        HTableDescriptor desc = FSTableDescriptors.getTableDescriptor((FileSystem)fs, (Path)rootDir, (byte[])TABLE_NAME);
        HTableDescriptor snapshotDesc = FSTableDescriptors.getTableDescriptor((FileSystem)fs, (Path)SnapshotDescriptionUtils.getSnapshotsDir((Path)rootDir), (byte[])Bytes.toBytes((String)snapshotName));
        Assert.assertEquals((Object)desc, (Object)snapshotDesc);
        List regions = admin.getTableRegions(TABLE_NAME);
        for (HRegionInfo info : regions) {
            String regionName = info.getEncodedName();
            Path regionDir = new Path(snapshotDir, regionName);
            HRegionInfo snapshotRegionInfo = HRegion.loadDotRegionInfoFileContent((FileSystem)fs, (Path)regionDir);
            Assert.assertEquals((Object)info, (Object)snapshotRegionInfo);
            Path familyDir = new Path(regionDir, Bytes.toString((byte[])TEST_FAM));
            Assert.assertTrue((boolean)fs.exists(familyDir));
            Assert.assertTrue((fs.listStatus(familyDir).length > 0 ? 1 : 0) != 0);
        }
        admin.deleteSnapshot(snapshotName);
        FSUtils.logFileSystemState((FileSystem)UTIL.getTestFileSystem(), (Path)FSUtils.getRootDir((Configuration)UTIL.getConfiguration()), (Log)LOG);
        SnapshotTestingUtils.assertNoSnapshots(admin);
        LOG.debug((Object)"------- Flush-Snapshot Create List Destroy-------------");
    }

    @Test(timeout=60000L)
    public void testConcurrentSnapshottingAttempts() throws IOException, InterruptedException {
        int ssNum = 10;
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        UTIL.loadTable(new HTable(UTIL.getConfiguration(), TABLE_NAME), TEST_FAM);
        HRegionServer rs = UTIL.getRSForFirstRegionInTable(TABLE_NAME);
        List onlineRegions = rs.getOnlineRegions(TABLE_NAME);
        for (HRegion region : onlineRegions) {
            region.waitForFlushesAndCompactions();
        }
        HBaseProtos.SnapshotDescription[] descs = new HBaseProtos.SnapshotDescription[ssNum];
        for (int i = 0; i < ssNum; ++i) {
            HBaseProtos.SnapshotDescription.Builder builder = HBaseProtos.SnapshotDescription.newBuilder();
            builder.setTable(STRING_TABLE_NAME);
            builder.setName("ss" + i);
            builder.setType(HBaseProtos.SnapshotDescription.Type.FLUSH);
            descs[i] = builder.build();
        }
        final CountDownLatch toBeSubmitted = new CountDownLatch(ssNum);
        for (int i = 0; i < ssNum; ++i) {
            class SSRunnable
            implements Runnable {
                HBaseProtos.SnapshotDescription ss;

                SSRunnable(HBaseProtos.SnapshotDescription ss) {
                    this.ss = ss;
                }

                @Override
                public void run() {
                    try {
                        HBaseAdmin admin = UTIL.getHBaseAdmin();
                        LOG.info((Object)("Submitting snapshot request: " + SnapshotDescriptionUtils.toString((HBaseProtos.SnapshotDescription)this.ss)));
                        admin.takeSnapshotAsync(this.ss);
                    }
                    catch (Exception e) {
                        LOG.info((Object)("Exception during snapshot request: " + SnapshotDescriptionUtils.toString((HBaseProtos.SnapshotDescription)this.ss) + ".  This is ok, we expect some"), (Throwable)e);
                    }
                    LOG.info((Object)("Submitted snapshot request: " + SnapshotDescriptionUtils.toString((HBaseProtos.SnapshotDescription)this.ss)));
                    toBeSubmitted.countDown();
                }
            }
            new Thread(new SSRunnable(descs[i])).start();
        }
        toBeSubmitted.await();
        while (true) {
            int doneCount = 0;
            for (HBaseProtos.SnapshotDescription ss : descs) {
                try {
                    if (!admin.isSnapshotFinished(ss)) continue;
                    ++doneCount;
                }
                catch (Exception e) {
                    LOG.warn((Object)("Got an exception when checking for snapshot " + ss.getName()), (Throwable)e);
                    ++doneCount;
                }
            }
            if (doneCount == descs.length) break;
            Thread.sleep(100L);
        }
        this.logFSTree(new Path(UTIL.getConfiguration().get("hbase.rootdir")));
        List taken = admin.listSnapshots();
        int takenSize = taken.size();
        LOG.info((Object)("Taken " + takenSize + " snapshots:  " + taken));
        Assert.assertTrue((String)"We expect at least 1 request to be rejected because of we concurrently issued many requests", (takenSize < ssNum && takenSize > 0 ? 1 : 0) != 0);
        for (HBaseProtos.SnapshotDescription ss : taken) {
            admin.deleteSnapshot(ss.getName());
        }
    }

    private void logFSTree(Path root) throws IOException {
        FSUtils.logFileSystemState((FileSystem)UTIL.getDFSCluster().getFileSystem(), (Path)root, (Log)LOG);
    }
}

