/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheColumnFamilySummary;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheStats;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CachedBlock;
import org.apache.hadoop.hbase.io.hfile.CachedBlockQueue;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.regionserver.metrics.SchemaMetrics;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.HasThread;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.util.StringUtils;

public class LruBlockCache
implements BlockCache,
HeapSize {
    static final Log LOG = LogFactory.getLog(LruBlockCache.class);
    static final String LRU_MIN_FACTOR_CONFIG_NAME = "hbase.lru.blockcache.min.factor";
    static final String LRU_ACCEPTABLE_FACTOR_CONFIG_NAME = "hbase.lru.blockcache.acceptable.factor";
    static final float DEFAULT_LOAD_FACTOR = 0.75f;
    static final int DEFAULT_CONCURRENCY_LEVEL = 16;
    static final float DEFAULT_MIN_FACTOR = 0.75f;
    static final float DEFAULT_ACCEPTABLE_FACTOR = 0.85f;
    static final float DEFAULT_SINGLE_FACTOR = 0.25f;
    static final float DEFAULT_MULTI_FACTOR = 0.5f;
    static final float DEFAULT_MEMORY_FACTOR = 0.25f;
    static final int statThreadPeriod = 300;
    private final ConcurrentHashMap<BlockCacheKey, CachedBlock> map;
    private final ReentrantLock evictionLock = new ReentrantLock(true);
    private volatile boolean evictionInProgress = false;
    private final EvictionThread evictionThread;
    private final ScheduledExecutorService scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("LRU Statistics #%d").setDaemon(true).build());
    private final AtomicLong size;
    private final AtomicLong elements;
    private final AtomicLong count;
    private final CacheStats stats;
    private long maxSize;
    private long blockSize;
    private float acceptableFactor;
    private float minFactor;
    private float singleFactor;
    private float multiFactor;
    private float memoryFactor;
    private long overhead;
    public static final long CACHE_FIXED_OVERHEAD = ClassSize.align(24 + 8 * ClassSize.REFERENCE + 20 + 1 + ClassSize.OBJECT);

    public LruBlockCache(long maxSize, long blockSize, Configuration conf) {
        this(maxSize, blockSize, true, conf);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread, Configuration conf) {
        this(maxSize, blockSize, evictionThread, (int)Math.ceil(1.2 * (double)maxSize / (double)blockSize), 0.75f, 16, conf.getFloat(LRU_MIN_FACTOR_CONFIG_NAME, 0.75f), conf.getFloat(LRU_ACCEPTABLE_FACTOR_CONFIG_NAME, 0.85f), 0.25f, 0.5f, 0.25f);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread, int mapInitialSize, float mapLoadFactor, int mapConcurrencyLevel, float minFactor, float acceptableFactor, float singleFactor, float multiFactor, float memoryFactor) {
        if (singleFactor + multiFactor + memoryFactor != 1.0f) {
            throw new IllegalArgumentException("Single, multi, and memory factors  should total 1.0");
        }
        if (minFactor >= acceptableFactor) {
            throw new IllegalArgumentException("minFactor must be smaller than acceptableFactor");
        }
        if (minFactor >= 1.0f || acceptableFactor >= 1.0f) {
            throw new IllegalArgumentException("all factors must be < 1");
        }
        this.maxSize = maxSize;
        this.blockSize = blockSize;
        this.map = new ConcurrentHashMap(mapInitialSize, mapLoadFactor, mapConcurrencyLevel);
        this.minFactor = minFactor;
        this.acceptableFactor = acceptableFactor;
        this.singleFactor = singleFactor;
        this.multiFactor = multiFactor;
        this.memoryFactor = memoryFactor;
        this.stats = new CacheStats();
        this.count = new AtomicLong(0L);
        this.elements = new AtomicLong(0L);
        this.overhead = LruBlockCache.calculateOverhead(maxSize, blockSize, mapConcurrencyLevel);
        this.size = new AtomicLong(this.overhead);
        if (evictionThread) {
            this.evictionThread = new EvictionThread(this);
            this.evictionThread.start();
        } else {
            this.evictionThread = null;
        }
        this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(this), 300L, 300L, TimeUnit.SECONDS);
    }

    public void setMaxSize(long maxSize) {
        this.maxSize = maxSize;
        if (this.size.get() > this.acceptableSize() && !this.evictionInProgress) {
            this.runEviction();
        }
    }

    @Override
    public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf, boolean inMemory) {
        CachedBlock cb = this.map.get(cacheKey);
        if (cb != null) {
            throw new RuntimeException("Cached an already cached block");
        }
        cb = new CachedBlock(cacheKey, buf, this.count.incrementAndGet(), inMemory);
        long newSize = this.updateSizeMetrics(cb, false);
        this.map.put(cacheKey, cb);
        this.elements.incrementAndGet();
        if (newSize > this.acceptableSize() && !this.evictionInProgress) {
            this.runEviction();
        }
    }

    @Override
    public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf) {
        this.cacheBlock(cacheKey, buf, false);
    }

    protected long updateSizeMetrics(CachedBlock cb, boolean evict) {
        Cacheable cachedBlock;
        SchemaMetrics schemaMetrics;
        long heapsize = cb.heapSize();
        if (evict) {
            heapsize *= -1L;
        }
        if ((schemaMetrics = (cachedBlock = cb.getBuffer()).getSchemaMetrics()) != null) {
            schemaMetrics.updateOnCachePutOrEvict(cachedBlock.getBlockType().getCategory(), heapsize, evict);
        }
        return this.size.addAndGet(heapsize);
    }

    @Override
    public Cacheable getBlock(BlockCacheKey cacheKey, boolean caching, boolean repeat) {
        CachedBlock cb = this.map.get(cacheKey);
        if (cb == null) {
            if (!repeat) {
                this.stats.miss(caching);
            }
            return null;
        }
        this.stats.hit(caching);
        cb.access(this.count.incrementAndGet());
        return cb.getBuffer();
    }

    @Override
    public boolean evictBlock(BlockCacheKey cacheKey) {
        CachedBlock cb = this.map.get(cacheKey);
        if (cb == null) {
            return false;
        }
        this.evictBlock(cb);
        return true;
    }

    @Override
    public int evictBlocksByHfileName(String hfileName) {
        int numEvicted = 0;
        for (BlockCacheKey key : this.map.keySet()) {
            if (!key.getHfileName().equals(hfileName) || !this.evictBlock(key)) continue;
            ++numEvicted;
        }
        return numEvicted;
    }

    protected long evictBlock(CachedBlock block) {
        this.map.remove(block.getCacheKey());
        this.updateSizeMetrics(block, true);
        this.elements.decrementAndGet();
        this.stats.evicted();
        return block.heapSize();
    }

    private void runEviction() {
        if (this.evictionThread == null) {
            this.evict();
        } else {
            this.evictionThread.evict();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void evict() {
        if (!this.evictionLock.tryLock()) {
            return;
        }
        try {
            BlockBucket bucket;
            this.evictionInProgress = true;
            long currentSize = this.size.get();
            long bytesToFree = currentSize - this.minSize();
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Block cache LRU eviction started; Attempting to free " + StringUtils.byteDesc((long)bytesToFree) + " of total=" + StringUtils.byteDesc((long)currentSize)));
            }
            if (bytesToFree <= 0L) {
                return;
            }
            BlockBucket bucketSingle = new BlockBucket(bytesToFree, this.blockSize, this.singleSize());
            BlockBucket bucketMulti = new BlockBucket(bytesToFree, this.blockSize, this.multiSize());
            BlockBucket bucketMemory = new BlockBucket(bytesToFree, this.blockSize, this.memorySize());
            for (CachedBlock cachedBlock : this.map.values()) {
                switch (cachedBlock.getPriority()) {
                    case SINGLE: {
                        bucketSingle.add(cachedBlock);
                        break;
                    }
                    case MULTI: {
                        bucketMulti.add(cachedBlock);
                        break;
                    }
                    case MEMORY: {
                        bucketMemory.add(cachedBlock);
                    }
                }
            }
            PriorityQueue<BlockBucket> bucketQueue = new PriorityQueue<BlockBucket>(3);
            bucketQueue.add(bucketSingle);
            bucketQueue.add(bucketMulti);
            bucketQueue.add(bucketMemory);
            int remainingBuckets = 3;
            long bytesFreed = 0L;
            while ((bucket = (BlockBucket)bucketQueue.poll()) != null) {
                long overflow = bucket.overflow();
                if (overflow > 0L) {
                    long bucketBytesToFree = Math.min(overflow, (bytesToFree - bytesFreed) / (long)remainingBuckets);
                    bytesFreed += bucket.free(bucketBytesToFree);
                }
                --remainingBuckets;
            }
            if (LOG.isDebugEnabled()) {
                long single = bucketSingle.totalSize();
                long multi = bucketMulti.totalSize();
                long memory = bucketMemory.totalSize();
                LOG.debug((Object)("Block cache LRU eviction completed; freed=" + StringUtils.byteDesc((long)bytesFreed) + ", " + "total=" + StringUtils.byteDesc((long)this.size.get()) + ", " + "single=" + StringUtils.byteDesc((long)single) + ", " + "multi=" + StringUtils.byteDesc((long)multi) + ", " + "memory=" + StringUtils.byteDesc((long)memory)));
            }
        }
        finally {
            this.stats.evict();
            this.evictionInProgress = false;
            this.evictionLock.unlock();
        }
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    @Override
    public long getCurrentSize() {
        return this.size.get();
    }

    @Override
    public long getFreeSize() {
        return this.getMaxSize() - this.getCurrentSize();
    }

    @Override
    public long size() {
        return this.elements.get();
    }

    @Override
    public long getBlockCount() {
        return this.elements.get();
    }

    public long getEvictionCount() {
        return this.stats.getEvictionCount();
    }

    @Override
    public long getEvictedCount() {
        return this.stats.getEvictedCount();
    }

    EvictionThread getEvictionThread() {
        return this.evictionThread;
    }

    public void logStats() {
        if (!LOG.isDebugEnabled()) {
            return;
        }
        long totalSize = this.heapSize();
        long freeSize = this.maxSize - totalSize;
        LOG.debug((Object)("Stats: total=" + StringUtils.byteDesc((long)totalSize) + ", " + "free=" + StringUtils.byteDesc((long)freeSize) + ", " + "max=" + StringUtils.byteDesc((long)this.maxSize) + ", " + "blocks=" + this.size() + ", " + "accesses=" + this.stats.getRequestCount() + ", " + "hits=" + this.stats.getHitCount() + ", " + "hitRatio=" + (this.stats.getHitCount() == 0L ? "0" : StringUtils.formatPercent((double)this.stats.getHitRatio(), (int)2) + ", ") + ", " + "cachingAccesses=" + this.stats.getRequestCachingCount() + ", " + "cachingHits=" + this.stats.getHitCachingCount() + ", " + "cachingHitsRatio=" + (this.stats.getHitCachingCount() == 0L ? "0" : StringUtils.formatPercent((double)this.stats.getHitCachingRatio(), (int)2) + ", ") + ", " + "evictions=" + this.stats.getEvictionCount() + ", " + "evicted=" + this.stats.getEvictedCount() + ", " + "evictedPerRun=" + this.stats.evictedPerEviction()));
    }

    @Override
    public CacheStats getStats() {
        return this.stats;
    }

    @Override
    public long heapSize() {
        return this.getCurrentSize();
    }

    public static long calculateOverhead(long maxSize, long blockSize, int concurrency) {
        return CACHE_FIXED_OVERHEAD + (long)ClassSize.CONCURRENT_HASHMAP + (long)Math.ceil((double)maxSize * 1.2 / (double)blockSize) * (long)ClassSize.CONCURRENT_HASHMAP_ENTRY + (long)(concurrency * ClassSize.CONCURRENT_HASHMAP_SEGMENT);
    }

    @Override
    public List<BlockCacheColumnFamilySummary> getBlockCacheColumnFamilySummaries(Configuration conf) throws IOException {
        Map<String, Path> sfMap = FSUtils.getTableStoreFilePathMap(FileSystem.get((Configuration)conf), FSUtils.getRootDir(conf));
        HashMap<BlockCacheColumnFamilySummary, BlockCacheColumnFamilySummary> bcs = new HashMap<BlockCacheColumnFamilySummary, BlockCacheColumnFamilySummary>();
        for (CachedBlock cb : this.map.values()) {
            String sf = cb.getCacheKey().getHfileName();
            Path path = sfMap.get(sf);
            if (path == null) continue;
            BlockCacheColumnFamilySummary lookup = BlockCacheColumnFamilySummary.createFromStoreFilePath(path);
            BlockCacheColumnFamilySummary bcse = (BlockCacheColumnFamilySummary)bcs.get(lookup);
            if (bcse == null) {
                bcse = BlockCacheColumnFamilySummary.create(lookup);
                bcs.put(lookup, bcse);
            }
            bcse.incrementBlocks();
            bcse.incrementHeapSize(cb.heapSize());
        }
        ArrayList<BlockCacheColumnFamilySummary> list = new ArrayList<BlockCacheColumnFamilySummary>(bcs.values());
        Collections.sort(list);
        return list;
    }

    private long acceptableSize() {
        return (long)Math.floor((float)this.maxSize * this.acceptableFactor);
    }

    private long minSize() {
        return (long)Math.floor((float)this.maxSize * this.minFactor);
    }

    private long singleSize() {
        return (long)Math.floor((float)this.maxSize * this.singleFactor * this.minFactor);
    }

    private long multiSize() {
        return (long)Math.floor((float)this.maxSize * this.multiFactor * this.minFactor);
    }

    private long memorySize() {
        return (long)Math.floor((float)this.maxSize * this.memoryFactor * this.minFactor);
    }

    @Override
    public void shutdown() {
        this.scheduleThreadPool.shutdown();
        for (int i = 0; i < 10; ++i) {
            if (this.scheduleThreadPool.isShutdown()) continue;
            Threads.sleep(10L);
        }
        if (!this.scheduleThreadPool.isShutdown()) {
            List<Runnable> runnables = this.scheduleThreadPool.shutdownNow();
            LOG.debug((Object)("Still running " + runnables));
        }
        this.evictionThread.shutdown();
    }

    public void clearCache() {
        this.map.clear();
    }

    SortedSet<String> getCachedFileNamesForTest() {
        TreeSet<String> fileNames = new TreeSet<String>();
        for (BlockCacheKey cacheKey : this.map.keySet()) {
            fileNames.add(cacheKey.getHfileName());
        }
        return fileNames;
    }

    Map<BlockType, Integer> getBlockTypeCountsForTest() {
        EnumMap<BlockType, Integer> counts = new EnumMap<BlockType, Integer>(BlockType.class);
        for (CachedBlock cb : this.map.values()) {
            BlockType blockType;
            Integer count = (Integer)counts.get((Object)(blockType = ((HFileBlock)cb.getBuffer()).getBlockType()));
            counts.put(blockType, (count == null ? 0 : count) + 1);
        }
        return counts;
    }

    public Map<DataBlockEncoding, Integer> getEncodingCountsForTest() {
        EnumMap<DataBlockEncoding, Integer> counts = new EnumMap<DataBlockEncoding, Integer>(DataBlockEncoding.class);
        for (BlockCacheKey cacheKey : this.map.keySet()) {
            DataBlockEncoding encoding;
            Integer count = (Integer)counts.get((Object)(encoding = cacheKey.getDataBlockEncoding()));
            counts.put(encoding, (count == null ? 0 : count) + 1);
        }
        return counts;
    }

    static class StatisticsThread
    extends Thread {
        LruBlockCache lru;

        public StatisticsThread(LruBlockCache lru) {
            super("LruBlockCache.StatisticsThread");
            this.setDaemon(true);
            this.lru = lru;
        }

        @Override
        public void run() {
            this.lru.logStats();
        }
    }

    static class EvictionThread
    extends HasThread {
        private WeakReference<LruBlockCache> cache;
        private boolean go = true;
        private boolean enteringRun = false;

        public EvictionThread(LruBlockCache cache) {
            super(Thread.currentThread().getName() + ".LruBlockCache.EvictionThread");
            this.setDaemon(true);
            this.cache = new WeakReference<LruBlockCache>(cache);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            this.enteringRun = true;
            while (this.go) {
                EvictionThread evictionThread = this;
                synchronized (evictionThread) {
                    try {
                        this.wait();
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                }
                LruBlockCache cache = (LruBlockCache)this.cache.get();
                if (cache == null) break;
                cache.evict();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void evict() {
            EvictionThread evictionThread = this;
            synchronized (evictionThread) {
                this.notify();
            }
        }

        void shutdown() {
            this.go = false;
            this.interrupt();
        }

        boolean isEnteringRun() {
            return this.enteringRun;
        }
    }

    private class BlockBucket
    implements Comparable<BlockBucket> {
        private CachedBlockQueue queue;
        private long totalSize = 0L;
        private long bucketSize;

        public BlockBucket(long bytesToFree, long blockSize, long bucketSize) {
            this.bucketSize = bucketSize;
            this.queue = new CachedBlockQueue(bytesToFree, blockSize);
            this.totalSize = 0L;
        }

        public void add(CachedBlock block) {
            this.totalSize += block.heapSize();
            this.queue.add(block);
        }

        public long free(long toFree) {
            CachedBlock cb;
            long freedBytes = 0L;
            while ((cb = this.queue.pollLast()) != null) {
                if ((freedBytes += LruBlockCache.this.evictBlock(cb)) < toFree) continue;
                return freedBytes;
            }
            return freedBytes;
        }

        public long overflow() {
            return this.totalSize - this.bucketSize;
        }

        public long totalSize() {
            return this.totalSize;
        }

        @Override
        public int compareTo(BlockBucket that) {
            if (this.overflow() == that.overflow()) {
                return 0;
            }
            return this.overflow() > that.overflow() ? 1 : -1;
        }
    }
}

