/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Random;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.TableNotDisabledException;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HConnection;
import org.apache.hadoop.hbase.client.HConnectionManager;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.RegionScanner;
import org.apache.hadoop.hbase.regionserver.wal.HLog;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSTableDescriptors;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.Writable;

class HMerge {
    static final Log LOG = LogFactory.getLog(HMerge.class);
    static final Random rand = new Random();

    private HMerge() {
    }

    public static void merge(Configuration conf, FileSystem fs, byte[] tableName) throws IOException {
        HMerge.merge(conf, fs, tableName, true);
    }

    public static void merge(Configuration conf, FileSystem fs, byte[] tableName, boolean testMasterRunning) throws IOException {
        boolean masterIsRunning = false;
        if (testMasterRunning) {
            masterIsRunning = HConnectionManager.execute(new HConnectionManager.HConnectable<Boolean>(conf){

                @Override
                public Boolean connect(HConnection connection) throws IOException {
                    return connection.isMasterRunning();
                }
            });
        }
        if (Bytes.equals(tableName, HConstants.META_TABLE_NAME)) {
            if (masterIsRunning) {
                throw new IllegalStateException("Can not compact META table if instance is on-line");
            }
            new OfflineMerger(conf, fs).process();
        } else {
            if (!masterIsRunning) {
                throw new IllegalStateException("HBase instance must be running to merge a normal table");
            }
            HBaseAdmin admin = new HBaseAdmin(conf);
            if (!admin.isTableDisabled(tableName)) {
                throw new TableNotDisabledException(tableName);
            }
            new OnlineMerger(conf, fs, tableName).process();
        }
    }

    private static class OfflineMerger
    extends Merger {
        private final List<HRegionInfo> metaRegions = new ArrayList<HRegionInfo>();
        private final HRegion root;

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        OfflineMerger(Configuration conf, FileSystem fs) throws IOException {
            super(conf, fs, HConstants.META_TABLE_NAME);
            Path rootTableDir = HTableDescriptor.getTableDir(fs.makeQualified(new Path(conf.get("hbase.rootdir"))), HConstants.ROOT_TABLE_NAME);
            this.root = HRegion.newHRegion(rootTableDir, this.hlog, fs, conf, HRegionInfo.ROOT_REGIONINFO, HTableDescriptor.ROOT_TABLEDESC, null);
            this.root.initialize();
            Scan scan = new Scan();
            scan.addColumn(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
            RegionScanner rootScanner = this.root.getScanner(scan);
            try {
                boolean hasMore;
                ArrayList<KeyValue> results = new ArrayList<KeyValue>();
                do {
                    hasMore = rootScanner.next(results);
                    for (KeyValue kv : results) {
                        HRegionInfo info = Writables.getHRegionInfoOrNull(kv.getValue());
                        if (info == null) continue;
                        this.metaRegions.add(info);
                    }
                } while (hasMore);
            }
            finally {
                rootScanner.close();
                try {
                    this.root.close();
                }
                catch (IOException e) {
                    LOG.error((Object)e);
                }
            }
        }

        @Override
        protected HRegionInfo[] next() {
            HRegionInfo[] results = null;
            if (this.metaRegions.size() > 0) {
                results = this.metaRegions.toArray(new HRegionInfo[this.metaRegions.size()]);
                this.metaRegions.clear();
            }
            return results;
        }

        @Override
        protected void updateMeta(byte[] oldRegion1, byte[] oldRegion2, HRegion newRegion) throws IOException {
            byte[][] regionsToDelete = new byte[][]{oldRegion1, oldRegion2};
            for (int r = 0; r < regionsToDelete.length; ++r) {
                Delete delete = new Delete(regionsToDelete[r]);
                delete.deleteColumns(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
                delete.deleteColumns(HConstants.CATALOG_FAMILY, HConstants.SERVER_QUALIFIER);
                delete.deleteColumns(HConstants.CATALOG_FAMILY, HConstants.STARTCODE_QUALIFIER);
                delete.deleteColumns(HConstants.CATALOG_FAMILY, HConstants.SPLITA_QUALIFIER);
                delete.deleteColumns(HConstants.CATALOG_FAMILY, HConstants.SPLITB_QUALIFIER);
                this.root.delete(delete, null, true);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug((Object)("updated columns in row: " + Bytes.toStringBinary(regionsToDelete[r])));
            }
            HRegionInfo newInfo = newRegion.getRegionInfo();
            newInfo.setOffline(true);
            Put put2 = new Put(newRegion.getRegionName());
            put2.add(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER, Writables.getBytes((Writable)newInfo));
            this.root.put(put2);
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("updated columns in row: " + Bytes.toStringBinary(newRegion.getRegionName())));
            }
        }
    }

    private static class OnlineMerger
    extends Merger {
        private final byte[] tableName;
        private final HTable table;
        private final ResultScanner metaScanner;
        private HRegionInfo latestRegion;

        OnlineMerger(Configuration conf, FileSystem fs, byte[] tableName) throws IOException {
            super(conf, fs, tableName);
            this.tableName = tableName;
            this.table = new HTable(conf, HConstants.META_TABLE_NAME);
            this.metaScanner = this.table.getScanner(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
            this.latestRegion = null;
        }

        private HRegionInfo nextRegion() throws IOException {
            try {
                Result results = this.getMetaRow();
                if (results == null) {
                    return null;
                }
                byte[] regionInfoValue = results.getValue(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
                if (regionInfoValue == null || regionInfoValue.length == 0) {
                    throw new NoSuchElementException("meta region entry missing " + Bytes.toString(HConstants.CATALOG_FAMILY) + ":" + Bytes.toString(HConstants.REGIONINFO_QUALIFIER));
                }
                HRegionInfo region = Writables.getHRegionInfo(regionInfoValue);
                if (!Bytes.equals(region.getTableName(), this.tableName)) {
                    return null;
                }
                return region;
            }
            catch (IOException e) {
                e = RemoteExceptionHandler.checkIOException(e);
                LOG.error((Object)"meta scanner error", (Throwable)e);
                this.metaScanner.close();
                throw e;
            }
        }

        private Result getMetaRow() throws IOException {
            Result currentRow = this.metaScanner.next();
            boolean foundResult = false;
            while (currentRow != null) {
                LOG.info((Object)("Row: <" + Bytes.toStringBinary(currentRow.getRow()) + ">"));
                byte[] regionInfoValue = currentRow.getValue(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
                if (regionInfoValue == null || regionInfoValue.length == 0) {
                    currentRow = this.metaScanner.next();
                    continue;
                }
                foundResult = true;
                break;
            }
            return foundResult ? currentRow : null;
        }

        @Override
        protected HRegionInfo[] next() throws IOException {
            ArrayList<HRegionInfo> regions = new ArrayList<HRegionInfo>();
            if (this.latestRegion == null) {
                this.latestRegion = this.nextRegion();
            }
            if (this.latestRegion != null) {
                regions.add(this.latestRegion);
            }
            this.latestRegion = this.nextRegion();
            if (this.latestRegion != null) {
                regions.add(this.latestRegion);
            }
            return regions.toArray(new HRegionInfo[regions.size()]);
        }

        @Override
        protected void updateMeta(byte[] oldRegion1, byte[] oldRegion2, HRegion newRegion) throws IOException {
            byte[][] regionsToDelete = new byte[][]{oldRegion1, oldRegion2};
            for (int r = 0; r < regionsToDelete.length; ++r) {
                if (Bytes.equals(regionsToDelete[r], this.latestRegion.getRegionName())) {
                    this.latestRegion = null;
                }
                Delete delete = new Delete(regionsToDelete[r]);
                this.table.delete(delete);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug((Object)("updated columns in row: " + Bytes.toStringBinary(regionsToDelete[r])));
            }
            newRegion.getRegionInfo().setOffline(true);
            Put put2 = new Put(newRegion.getRegionName());
            put2.add(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER, Writables.getBytes((Writable)newRegion.getRegionInfo()));
            this.table.put(put2);
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("updated columns in row: " + Bytes.toStringBinary(newRegion.getRegionName())));
            }
        }
    }

    private static abstract class Merger {
        protected final Configuration conf;
        protected final FileSystem fs;
        protected final Path tabledir;
        protected final HTableDescriptor htd;
        protected final HLog hlog;
        private final long maxFilesize;

        protected Merger(Configuration conf, FileSystem fs, byte[] tableName) throws IOException {
            this.conf = conf;
            this.fs = fs;
            this.maxFilesize = conf.getLong("hbase.hregion.max.filesize", 0x280000000L);
            this.tabledir = new Path(fs.makeQualified(new Path(conf.get("hbase.rootdir"))), Bytes.toString(tableName));
            this.htd = FSTableDescriptors.getTableDescriptor(this.fs, this.tabledir);
            Path logdir = new Path(this.tabledir, "merge_" + System.currentTimeMillis() + ".logs");
            Path oldLogDir = new Path(this.tabledir, ".oldlogs");
            this.hlog = new HLog(fs, logdir, oldLogDir, conf);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void process() throws IOException {
            try {
                HRegionInfo[] regionsToMerge = this.next();
                while (regionsToMerge != null) {
                    if (!this.merge(regionsToMerge)) {
                        return;
                    }
                    regionsToMerge = this.next();
                }
            }
            finally {
                try {
                    this.hlog.closeAndDelete();
                }
                catch (IOException e) {
                    LOG.error((Object)e);
                }
            }
        }

        protected boolean merge(HRegionInfo[] info) throws IOException {
            if (info.length < 2) {
                LOG.info((Object)"only one region - nothing to merge");
                return false;
            }
            HRegion currentRegion = null;
            long currentSize = 0L;
            HRegion nextRegion = null;
            long nextSize = 0L;
            for (int i = 0; i < info.length - 1; ++i) {
                if (currentRegion == null) {
                    currentRegion = HRegion.newHRegion(this.tabledir, this.hlog, this.fs, this.conf, info[i], this.htd, null);
                    currentRegion.initialize();
                    currentSize = currentRegion.getLargestHStoreSize();
                }
                nextRegion = HRegion.newHRegion(this.tabledir, this.hlog, this.fs, this.conf, info[i + 1], this.htd, null);
                nextRegion.initialize();
                nextSize = nextRegion.getLargestHStoreSize();
                if (currentSize + nextSize <= this.maxFilesize / 2L) {
                    LOG.info((Object)("Merging regions " + currentRegion.getRegionNameAsString() + " and " + nextRegion.getRegionNameAsString()));
                    HRegion mergedRegion = HRegion.mergeAdjacent(currentRegion, nextRegion);
                    this.updateMeta(currentRegion.getRegionName(), nextRegion.getRegionName(), mergedRegion);
                    break;
                }
                LOG.info((Object)("not merging regions " + Bytes.toStringBinary(currentRegion.getRegionName()) + " and " + Bytes.toStringBinary(nextRegion.getRegionName())));
                currentRegion.close();
                currentRegion = nextRegion;
                currentSize = nextSize;
            }
            if (currentRegion != null) {
                currentRegion.close();
            }
            return true;
        }

        protected abstract HRegionInfo[] next() throws IOException;

        protected abstract void updateMeta(byte[] var1, byte[] var2, HRegion var3) throws IOException;
    }
}

