/**
 * @file  PathContext.cpp
 *
 * @brief Implementation file for PathInfo and PathContext
 *
 */
// RCS ID line follows -- this is updated by CVS
// $Id: PathContext.cpp 2838 2005-12-17 04:49:05Z elsapo $

#include "stdafx.h"
#include "PathContext.h"
#include "paths.h"

/**
 * @brief Copy constructor.
 */
PathInfo::PathInfo(const PathInfo &pi)
{
	m_sPath = pi.m_sPath;
}

/**
 * @brief Get path.
 * @param [in] sbNormalized TRUE if path is wanted in normalized format.
 */
CString PathInfo::GetPath(BOOL bNormalized /*= TRUE*/) const
{ 
	if (!bNormalized)
	{
		if (!paths_EndsWithSlash(m_sPath))
			return m_sPath + _T("\\");
		else
			return m_sPath;
	}
	else
		return m_sPath;
}

CString& PathInfo::GetRef()
{ 
	return m_sPath;
}

/**
 * @brief Set path.
 * @param [in] sPath New path for item.
 */
void PathInfo::SetPath(CString sPath)
{
	m_sPath = sPath;
}

/**
 * @brief Normalize path.
 */
void PathInfo::NormalizePath()
{
	paths_normalize(m_sPath);
}

PathContext::PathContext()
{
	m_nFiles = 0;
}

PathContext::PathContext(CString sLeft)
{
	m_nFiles = 1;
	m_path[0].SetPath(sLeft);
}

PathContext::PathContext(CString sLeft, CString sRight)
{
	m_nFiles = 2;
	m_path[0].SetPath(sLeft);
	m_path[1].SetPath(sRight);
}

PathContext::PathContext(CString sLeft, CString sMiddle, CString sRight)
{
	m_nFiles = 3;
	m_path[0].SetPath(sLeft);
	m_path[1].SetPath(sMiddle);
	m_path[2].SetPath(sRight);
}

PathContext::PathContext(CStringArray &sFiles)
{
	m_nFiles = sFiles.GetSize() > 3 ? 3 : sFiles.GetSize();
	for (int i = 0; i < m_nFiles; i++)
		m_path[i].SetPath(sFiles[i]);
}

PathContext::PathContext(const PathContext &paths)
{
	m_nFiles = paths.m_nFiles;
	for (int i = 0; i < m_nFiles; i++)
		m_path[i].SetPath(paths[i]);
}

CString PathContext::GetAt(int nIndex) const
{
	ASSERT(nIndex < m_nFiles);
	return m_path[nIndex].GetPath();
}

CString& PathContext::GetElement(int nIndex)
{
	ASSERT(nIndex < m_nFiles);
	return m_path[nIndex].GetRef();
}

void PathContext::SetAt(int nIndex, const CString& newElement)
{
	ASSERT(nIndex < m_nFiles);
	m_path[nIndex].SetPath(newElement);
}

CString PathContext::operator[](int nIndex) const
{
	return GetAt(nIndex);
}

CString& PathContext::operator[](int nIndex)
{
	return GetElement(nIndex);
}

/**
 * @brief set number of files.
 */
void PathContext::SetSize(int nFiles)
{
	m_nFiles = nFiles;
}

/**
 * @brief Return number of files.
 */
int PathContext::GetSize() const
{
	return m_nFiles;
}

/**
 * @brief Empty m_path array
 */
void PathContext::RemoveAll()
{
	m_nFiles = 0;
	m_path[0].SetPath(_T(""));
	m_path[1].SetPath(_T(""));
	m_path[2].SetPath(_T(""));
}

/**
 * @brief Return left path.
 * @param [in] sNormalized If TRUE normalized path is returned.
 */
CString PathContext::GetLeft(BOOL bNormalized) const
{
	if (m_nFiles == 0)
		return "";
	return m_path[0].GetPath(bNormalized);
}

/**
 * @brief Return right path.
 * @param [in] sNormalized If TRUE normalized path is returned.
 */
CString PathContext::GetRight(BOOL bNormalized) const
{
	if (m_nFiles < 2)
		return "";
	return m_path[m_nFiles - 1].GetPath(bNormalized);
}

/**
 * @brief Return middle path.
 * @param [in] sNormalized If TRUE normalized path is returned.
 */
CString PathContext::GetMiddle(BOOL bNormalized) const
{
	if (m_nFiles < 3)
		return "";
	return m_path[1].GetPath(bNormalized);
}

/**
 * @brief Return path
 * @param [in] index index of path to return
 * @param [in] sNormalized If TRUE normalized path is returned.
 */
CString PathContext::GetPath(int index, BOOL bNormalized) const
{
	return m_path[index].GetPath(bNormalized);
}

/**
 * @brief Set left path.
 * @param [in] path New path for item.
 */
void PathContext::SetLeft(LPCTSTR path)
{
	if (m_nFiles == 0)
		m_nFiles = 1;
	m_path[0].SetPath(path);
	m_path[0].NormalizePath();
}

/**
 * @brief Set right path.
 * @param [in] path New path for item.
 */
void PathContext::SetRight(LPCTSTR path)
{
	if (m_nFiles < 2)
		m_nFiles = 2;
	m_path[m_nFiles - 1].SetPath(path);
	m_path[m_nFiles - 1].NormalizePath();
}

/**
 * @brief Set middle path.
 * @param [in] path New path for item.
 */
void PathContext::SetMiddle(LPCTSTR path)
{
	if (m_nFiles < 3)
	{
		m_nFiles = 3;
		m_path[2] = m_path[1];
	}
	m_path[1].SetPath(path);
	m_path[1].NormalizePath();
}

/**
 * @brief Set path
 * @param [in] index index of path to set
 * @param [in] path New path for item.
 */
void PathContext::SetPath(int index, LPCTSTR path)
{
	if (index >= m_nFiles) m_nFiles = index + 1;
	m_path[index].SetPath(path);
	m_path[index].NormalizePath();
}

/**
 * @brief Destructor, deletes existing temp files.
 */
TempFileContext::~TempFileContext()
{
	if (FilesExist())
		DeleteFiles();
}

/**
 * @brief Create temp files.
 *
 * Creates new temp files. Temp files are named based on
 * PathContext paths given as parameter.
 *
 * @param [in] paths PathContext whose paths are used as basis.
 */
BOOL TempFileContext::CreateFiles(const PathContext &paths)
{
	TCHAR strTempPath[MAX_PATH] = {0};

	if (!::GetTempPath(MAX_PATH, strTempPath))
	{
		LogErrorString(Fmt(_T("GetTempPath() failed: %s"),
			GetSysError(GetLastError())));
		return FALSE;
	}

	m_sTempPath = strTempPath;

	SetSize(paths.GetSize());
	for (int file = 0; file < paths.GetSize(); file++)
	{
		if (GetPath(file).IsEmpty())
		{
			TCHAR *prefix[3] = {_T("_T0"), _T("_T1"), _T("_T2")};
			int nerr=0;
			CString sTempPath = paths_GetTempFileName(strTempPath, prefix[file], &nerr);
			if (sTempPath.IsEmpty())
			{
				LogErrorString(Fmt(_T("GetTempFileName() failed: %s"),
					GetSysError(nerr)));
				return FALSE;
			}
			SetPath(file, sTempPath);

			if (!paths.GetPath(file).IsEmpty())
			{
				if (!::CopyFile(paths.GetPath(file), GetPath(file), FALSE))
				{
					LogErrorString(Fmt(_T("CopyFile() (copy temp file) failed: %s"),
						GetSysError(GetLastError())));
					return FALSE;
				}
			}
			::SetFileAttributes(GetPath(file), FILE_ATTRIBUTE_NORMAL);
		}
	}
		
	return TRUE;
}

/**
 * @brief Check if temp files exist.
 * @return TRUE if one of temp files exist.
 */
BOOL TempFileContext::FilesExist()
{
	BOOL bLeftExists = FALSE;
	BOOL bMiddleExists = FALSE;
	BOOL bRightExists = FALSE;

	if (!GetLeft().IsEmpty())
		bLeftExists = (paths_DoesPathExist(GetLeft()) == IS_EXISTING_FILE);
	if (!GetMiddle().IsEmpty())
		bMiddleExists = (paths_DoesPathExist(GetMiddle()) == IS_EXISTING_FILE);
	if (!GetRight().IsEmpty())
		bLeftExists = (paths_DoesPathExist(GetRight()) == IS_EXISTING_FILE);
	
	return bLeftExists || bMiddleExists || bRightExists;
}

/**
 * @brief Delete temp files.
 */
void TempFileContext::DeleteFiles()
{
	int index;
	for (index = 0; index < GetSize(); index++)
	{
		if (!GetPath(index).IsEmpty())
		{
			if (!::DeleteFile(GetPath(index)))
			{
				LogErrorString(Fmt(_T("DeleteFile(%s) (deleting file%d temp file) failed: %s"),
					index, GetPath(index), GetSysError(GetLastError())));
			}
			SetPath(index, _T(""));
		}
	}
}
