//
// TimedNotificationQueueTest.cpp
//
// $Id: //poco/1.4/Foundation/testsuite/src/TimedNotificationQueueTest.cpp#1 $
//
// Copyright (c) 2009, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "TimedNotificationQueueTest.h"
#include "CppUnit/TestCaller.h"
#include "CppUnit/TestSuite.h"
#include "Poco/TimedNotificationQueue.h"
#include "Poco/Notification.h"
#include "Poco/Timestamp.h"


using Poco::TimedNotificationQueue;
using Poco::Notification;
using Poco::Timestamp;


namespace 
{
	class QTestNotification: public Notification
	{
	public:
		QTestNotification(const std::string& data): _data(data)
		{
		}
		~QTestNotification()
		{
		}
		const std::string& data() const
		{
			return _data;
		}

	private:
		std::string _data;
	};
}


TimedNotificationQueueTest::TimedNotificationQueueTest(const std::string& name): CppUnit::TestCase(name)
{
}


TimedNotificationQueueTest::~TimedNotificationQueueTest()
{
}


void TimedNotificationQueueTest::testDequeue()
{
	TimedNotificationQueue queue;
	assert (queue.empty());
	assert (queue.size() == 0);
	Notification* pNf = queue.dequeueNotification();
	assertNullPtr(pNf);
	queue.enqueueNotification(new Notification, Timestamp());
	assert (!queue.empty());
	assert (queue.size() == 1);
	pNf = queue.dequeueNotification();
	assertNotNullPtr(pNf);
	assert (queue.empty());
	assert (queue.size() == 0);
	pNf->release();
	
	Poco::Timestamp ts1;
	ts1 += 100000;
	Poco::Timestamp ts2;
	ts2 += 200000;
	Poco::Timestamp ts3;
	ts3 += 300000;
	Poco::Timestamp ts4;
	ts4 += 400000;
	
	queue.enqueueNotification(new QTestNotification("first"), ts1);
	queue.enqueueNotification(new QTestNotification("fourth"), ts4);
	queue.enqueueNotification(new QTestNotification("third"), ts3);
	queue.enqueueNotification(new QTestNotification("second"), ts2);
	assert (!queue.empty());
	assert (queue.size() == 4);
	QTestNotification* pTNf = 0;
	while (!pTNf) 
	{
		pTNf = dynamic_cast<QTestNotification*>(queue.dequeueNotification());
	}
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "first");
	pTNf->release();
	assert (ts1.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 3);
	
	pTNf = 0;
	while (!pTNf) 
	{
		pTNf = dynamic_cast<QTestNotification*>(queue.dequeueNotification());
	}
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "second");
	pTNf->release();
	assert (ts2.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 2);
	
	pTNf = 0;
	while (!pTNf) 
	{
		pTNf = dynamic_cast<QTestNotification*>(queue.dequeueNotification());
	}
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "third");
	pTNf->release();
	assert (ts3.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 1);
	
	pTNf = 0;
	while (!pTNf) 
	{
		pTNf = dynamic_cast<QTestNotification*>(queue.dequeueNotification());
	}
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "fourth");
	pTNf->release();
	assert (ts4.elapsed() >= 0);
	assert (queue.empty());
	assert (queue.size() == 0);

	pNf = queue.dequeueNotification();
	assertNullPtr(pNf);
}


void TimedNotificationQueueTest::testWaitDequeue()
{
	TimedNotificationQueue queue;
	
	Poco::Timestamp ts1;
	ts1 += 100000;
	Poco::Timestamp ts2;
	ts2 += 200000;
	Poco::Timestamp ts3;
	ts3 += 300000;
	Poco::Timestamp ts4;
	ts4 += 400000;
	
	queue.enqueueNotification(new QTestNotification("first"), ts1);
	queue.enqueueNotification(new QTestNotification("fourth"), ts4);
	queue.enqueueNotification(new QTestNotification("third"), ts3);
	queue.enqueueNotification(new QTestNotification("second"), ts2);
	assert (!queue.empty());
	assert (queue.size() == 4);
	QTestNotification* pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification());
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "first");
	pTNf->release();
	assert (ts1.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 3);
	
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification());
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "second");
	pTNf->release();
	assert (ts2.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 2);
	
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification());
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "third");
	pTNf->release();
	assert (ts3.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 1);
	
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification());
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "fourth");
	pTNf->release();
	assert (ts4.elapsed() >= 0);
	assert (queue.empty());
	assert (queue.size() == 0);
}


void TimedNotificationQueueTest::testWaitDequeueTimeout()
{
	TimedNotificationQueue queue;
	
	Poco::Timestamp ts1;
	ts1 += 200000;
	Poco::Timestamp ts2;
	ts2 += 400000;
	Poco::Timestamp ts3;
	ts3 += 600000;
	Poco::Timestamp ts4;
	ts4 += 800000;
	
	queue.enqueueNotification(new QTestNotification("first"), ts1);
	queue.enqueueNotification(new QTestNotification("fourth"), ts4);
	queue.enqueueNotification(new QTestNotification("third"), ts3);
	queue.enqueueNotification(new QTestNotification("second"), ts2);
	assert (!queue.empty());
	assert (queue.size() == 4);
	QTestNotification* pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification(10));
	assertNullPtr(pTNf);
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification(20));
	assertNullPtr(pTNf);
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification(200));
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "first");
	pTNf->release();
	assert (ts1.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 3);
	
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification(220));
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "second");
	pTNf->release();
	assert (ts2.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 2);
	
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification(220));
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "third");
	pTNf->release();
	assert (ts3.elapsed() >= 0);
	assert (!queue.empty());
	assert (queue.size() == 1);
	
	pTNf = dynamic_cast<QTestNotification*>(queue.waitDequeueNotification(220));
	assertNotNullPtr(pTNf);
	assert (pTNf->data() == "fourth");
	pTNf->release();
	assert (ts1.elapsed() >= 0);
	assert (queue.empty());
	assert (queue.size() == 0);
}


void TimedNotificationQueueTest::setUp()
{
}


void TimedNotificationQueueTest::tearDown()
{
}


CppUnit::Test* TimedNotificationQueueTest::suite()
{
	CppUnit::TestSuite* pSuite = new CppUnit::TestSuite("TimedNotificationQueueTest");

	CppUnit_addTest(pSuite, TimedNotificationQueueTest, testDequeue);
	CppUnit_addTest(pSuite, TimedNotificationQueueTest, testWaitDequeue);
	CppUnit_addTest(pSuite, TimedNotificationQueueTest, testWaitDequeueTimeout);

	return pSuite;
}
