// HLObject is a class that represents an object of data stored in a
// Hotline packet. An HLPacket contains a list of HLObjects.
// HLObject provides functions for accessing the data in a useful way.

#ifndef HL_OBJECT_H
#define HL_OBJECT_H 1

#if !defined(WIN32)
#include <arpa/inet.h>
#endif // !WIN32
#include <string>
#include <vector>
#include <list>
#include <stdexcept>

typedef struct
{
    u_int16_t type PACKED;
    u_int16_t size PACKED;
    u_int8_t data[ZERO_SIZE_ARRAY_SIZE] PACKED;
} HL_OBJECT_HEADER;

using namespace std;

class HLObjectList;

typedef vector<string> PathVector;

class HLObject
{
    public:
        HLObject(u_int16_t inType, u_int16_t inSize, const char *inData);
        HLObject(const HLObject &inObj);
        ~HLObject();
        
        const char *GetData() const;
        u_int16_t GetType() const;
        u_int16_t GetSize() const;
        
        void GetDataResumeSizes(u_int32_t &outDataForkSize,
                u_int32_t &outResourceForkSize) const;
        void GetDataAsString(string& outString, u_int32_t inMaxLen = 0) const;
        //void GetDataAsPath(string &outPath) const;
        void GetDataAsPathVector(PathVector &outPathVector) const;
		void GetDataAsObjectList(HLObjectList &outObjectList) const;
        u_int32_t GetDataAsUInt32() const;
        u_int16_t GetDataAsUInt16() const;
        u_int8_t GetDataAsUInt8() const;
        
        void Encrypt();
            
    private:	
        u_int16_t mType;
        u_int16_t mSize;
        char *mData;
};

typedef list<HLObject *> hl_object_list;
class HLObjectList
{
	public:
		typedef hl_object_list::iterator iterator;
		iterator begin()
			{ return mList.begin(); }
		iterator end()
			{ return mList.end(); }
		
		HLObjectList() : mSize(2) { }
		
		virtual ~HLObjectList()
		{
			Clear();
		}
		
		void AddStringObject(u_int16_t inType, const string &inData, bool inEncrypt = false)
		{
			HLObject *newObject = AddObject(inType, inData.length(), (char *)inData.c_str());
			if (inEncrypt)
				newObject->Encrypt();
		}
		
		void AddUInt8Object(u_int16_t inType, u_int8_t inData)
		{
			AddObject(inType, sizeof(u_int8_t), (char *)&inData);
		}
		
		void AddUInt16Object(u_int16_t inType, u_int16_t inData)
		{
			u_int16_t newData = htons(inData);
			AddObject(inType, sizeof(u_int16_t), (char *)&newData);
		}
		
		void AddUInt32Object(u_int16_t inType, u_int32_t inData)
		{
			u_int32_t newData = htonl(inData);
			AddObject(inType, sizeof(u_int32_t), (char *)&newData);
		}
		
		void AddListObject(u_int16_t inType, HLObjectList &inData)
		{
			u_int32_t bufSize = inData.GetSize();
			// need to fix this later because Flatten can throw
			char *tmpBuf = new char[bufSize];
			inData.Flatten(tmpBuf, bufSize);
			AddObject(inType, bufSize, tmpBuf);
			delete[] tmpBuf;
		}
		
		HLObject *AddObject(u_int16_t inType, u_int16_t inSize, const char *inData)
		{
			mSize += (inSize + 4);
			HLObject *newObject = new HLObject(inType, inSize, inData);
			mList.push_back(newObject);
			return newObject;
		}
		
		void AddObject(const HLObject &inObject)
		{
			mSize += (inObject.GetSize() + 4);
			mList.push_back(new HLObject(inObject));
		}
		
		void Clear()
		{
			mSize = 2; // object counter size
			HLObjectList::iterator iter = mList.begin();
			while (iter != mList.end())
			{
				delete (*iter);
				iter++;
			}
			mList.clear();
		}
		
		u_int16_t GetObjCount() const
			{ return mList.size(); }
		
		virtual u_int32_t GetSize() const
            { return mSize; }
		
		bool RemoveObject(u_int16_t inIndex)
		{
			bool found = false;
			if (inIndex < mList.size())
			{
				HLObjectList::iterator iter = mList.begin();
				while (iter != mList.end())
				{
					if (inIndex == 0)
					{
						found = true;
						mList.erase(iter);
						break;
					}
					inIndex--;
					iter++;
				}
			}
			return found;
		}
		
		void Flatten(char *outBytes, u_int32_t &ioSize) throw (logic_error)
		{
			if (ioSize >= mSize)
			{
				HL_OBJECT_HEADER *o_hdr;
				u_int16_t pos = 0;
				// save the object count
				*((u_int16_t *)&outBytes[pos]) = htons(mList.size());
				pos += 2;
				
				HLObjectList::iterator iter = mList.begin();
				while (iter != mList.end())
				{
					o_hdr = (HL_OBJECT_HEADER *)&outBytes[pos];
					o_hdr->type = htons((*iter)->GetType());
					o_hdr->size = htons((*iter)->GetSize());
					memcpy(o_hdr->data, (*iter)->GetData(), (*iter)->GetSize());
					pos += ((*iter)->GetSize() + 4);
					iter++;
				}
			}
			else
				 throw logic_error("buffer too small");
		}
		
		bool Instantiate(char *inStream, u_int32_t inSize)
		{
			HL_OBJECT_HEADER *o_hdr;
			u_int32_t pos = 0;
			u_int16_t counter = 0;
			u_int16_t obj_count = 0;
			Clear();
			obj_count = ntohs(*((u_int16_t *)&inStream[pos]));
			pos += sizeof(u_int16_t);
			while ((counter < obj_count) && (pos < inSize))
			{
				o_hdr = (HL_OBJECT_HEADER *)&inStream[pos];
				// make sure the object size is realistic
				if ((ntohs(o_hdr->size) + mSize + 4) > inSize)
					return false;
				AddObject(ntohs(o_hdr->type), ntohs(o_hdr->size), (char *)o_hdr->data);
				pos += (ntohs(o_hdr->size) + 4);
				counter++;
			}
			return true;
		}
		
	private:
		hl_object_list mList;
		u_int32_t mSize;
};

#endif // HL_OBJECT_H


