// This is a class for file transfers to and from the server.
// Each file transfer occures in its own thread.
// The HLTransferManager is resposible for transfer queues,
// limits, clean up of finished/canceled transfers, etc...
#ifndef HL_TRANSFER_H
#define HL_TRANSFER_H
#include <string>
#include "BThread.h"
#include <list>
#include "NetworkSpeed.h"
#include "AsyncTCPSocket.h"
#include "HLProtocol.h"
#include "HLUser.h"
#include "DynamicBuffer.h"
using namespace std;

class HLTransferSocket;
class HLTransfer;

enum TransferType
{
	kInvalidTransfer = -1,
	kDownloadTransfer = 0,
	kPreviewTransfer = 1,
	kUploadTransfer = 2
};

class HLTransferInfo
{
public:
	HLTransferInfo()
		: localDataForkSize(0), remoteDataForkSize(0),
		transferSize(0), type(kInvalidTransfer) {}

		const bool valid() const
		{ return type != kInvalidTransfer; }

		u_int32_t localDataForkSize;
		u_int32_t remoteDataForkSize;
		u_int32_t transferSize;
		string filePath;
		string fileName;
		HLUser user;
		TransferType type;
		u_int32_t ref;
};

typedef list< HLTransferInfo > TransferInfoList;
typedef list< HLTransfer* > TransferList;

class HLTransfer : public BThread
{
public:
	HLTransfer(HLTransferSocket *inSocket);
	~HLTransfer();

	inline const HLUser &User() const
	{ return mInfo.user; }

	inline const u_int32_t Ref() const
	{ return mInfo.ref; }

	inline const HLTransferInfo &Info() const
	{ return mInfo; }

	void Cancel();

	void GetStats(string &outStats, u_int16_t inQueuePosition = 0);

	void SendKeepAlive();

	inline const u_int32_t TransferID() const
	{ return mTransferID; }

	inline void SetTransferID(const u_int32_t inTransferID)
	{ mTransferID = inTransferID; }

	inline const u_int32_t TransferedBytes() const
	{ return mSpeed.TransferedBytes(); }

	inline const u_int32_t TotalBytes() const
	{ return mSpeed.TotalBytes(); }

	inline NetworkSpeed &networkSpeed()
	{ return mSpeed; }

	unsigned int CalculateBufferSize();

//2003/08/07 added by ortana.
#ifdef WIN32
	static long GetRate(const std::string& loginName);
	static DWORD64 LoadRateValue(const char *loginName);
#endif//WIN32

protected:
	void Run();

	void DoDownload();
	void DoUpload();

	NetworkSpeed mSpeed;
	HLTransferInfo mInfo;
	HLTransferSocket *mSocket;
	int mFD;
	DynamicBuffer mBuf;
	u_int32_t mTransferID; // id in the database

	friend class HLTransferManager;
	friend class HLTransferSocket;

	//2003/08/07 added by ortana.
#ifdef WIN32
private :
	void SaveRate(const DWORD64& rateValue,const char *loginName);
	bool IsRateOver(const long& downloaded,const char *loginName);

	void SendRateOverMsg();
	void PostUploadEndToNews();
	void PostUploadEndToChat();
public :
	DWORD64 r_value;
	long rate;
#endif//WIN32
};

class HLTransferManager
{
public:
	HLTransferManager();
	~HLTransferManager();

	bool QueueTransferInfo(HLTransferInfo &inTransferInfo, bool &isLeech);
	bool GetTransferInfo(u_int32_t inRef, HLTransferInfo &outInfo);
	unsigned int SecondsUntilDownloadSpotOpens();
	void CancelStalledTransfers();
	void KeepAliveQueuedTransfers();
	void CancelTransfersForUser(HLUser &inUser);
	inline const unsigned int QueueSize() const
	{ return (unsigned int)mDownloadQueue.size(); }

	void BeginDownload(HLTransfer *inTransfer, bool &outQueued);
	void BeginUpload(HLTransfer *inTransfer);
	void EndDownload(HLTransfer *inTransfer);
	void EndUpload(HLTransfer *inTransfer);

	void CheckWaitingQueue();//2003/11/22 added

	void GetNetStat(char * outBuff);

	static u_int32_t MakeRef(const HLTransferInfo &inInfo);

protected:
	TransferInfoList mTransferInfoList;
	TransferList mDownloadList;
	TransferList mDownloadQueue;
	TransferList mUploadList;

	u_int32_t mTotalUploads;
	u_int32_t mTotalDownloads;
	u_int64_t mTotalUploadBytes;
	u_int64_t mTotalDownloadBytes;

	friend class HLServer;
	friend class HLClient;
	friend class WebServer;
};

class HLTransferSocket : public AsyncTCPSocket
{
public:
	HLTransferSocket(int inDescriptor, struct sockaddr_in* inRemoteAddress);
	virtual ~HLTransferSocket();

protected:
	void StartTransferThread();

	void OnRecv();
	void OnClose();

	HLTransfer *mTransfer;

	char mBuf[SIZEOF_HTXF_HDR];
	unsigned int mBufPos;
	bool mGotMagic;
	bool mThreadStarted;

	friend class HLTransfer;
	friend class HLTransferManager;
};


#endif // HL_TRANSFER_H

